/* -*- mode: c; c-basic-offset: 4; indent-tabs-mode: nil -*- */
/*
 * Copyright (C) 2013 Canonical, Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Author: Pawel Stolowski <pawel.stolowski@canonical.com>
 */

#include "OnlineMusicScope.h"
#include "SmartScopesClient.h"
#include "Utils.h"
#include <QCoreApplication>
#include <QStringList>
#include <QDebug>
#include <map>
#include <list>
#include <memory>
#include <string>
#include "config.h"

const char* OnlineMusicScope::UNIQUE_NAME = "/com/canonical/unity/scope/onlinemusic";
const char* OnlineMusicScope::GROUP_NAME = "com.canonical.Unity.Scope.OnlineMusic";

extern "C" {

int unity_scope_module_get_version()
{
    return UNITY_SCOPE_API_VERSION;
}

GList *unity_scope_module_load_scopes(GError **/* error */)
{
    bindtextdomain(GETTEXT_PACKAGE, LOCALE_DIR);
    textdomain(GETTEXT_PACKAGE);

    UnityAbstractScope *scope = OnlineMusicScope::instance().getUnityScope();
    return g_list_append(NULL, scope);
}

}

OnlineMusicScope::OnlineMusicScope()
    : m_disabledScopesSig(0)
{
    // determine locale for Smart Scopes queries
    auto languages = g_get_language_names();
    m_locale = (languages != nullptr && languages[0] != nullptr) ? QString(languages[0]) : QString("en");

    int argc = 1;
    char *argv[1] = {nullptr};

    // an instance of Application is needed to avoid warnings from Qt
    m_app = new QCoreApplication(argc, argv);
    setupUnityScope();
}

OnlineMusicScope::~OnlineMusicScope()
{
    if (m_disabledScopesSig > 0) {
        g_signal_handler_disconnect(unity_preferences_manager_get_default(), m_disabledScopesSig);
    }
    delete m_app;
}

void OnlineMusicScope::setupUnityScope()
{
    g_set_prgname("unity-scope-onlinemusic"); // set program name for better error messages

    updateDisabledScopes();

    auto prefs = unity_preferences_manager_get_default();
    m_disabledScopesSig = g_signal_connect(prefs, "notify::disabled-scopes", G_CALLBACK(disabled_scopes_changed), this);

    m_unityScope = unity_simple_scope_new();

    unity_simple_scope_set_group_name(m_unityScope, GROUP_NAME);
    unity_simple_scope_set_unique_name(m_unityScope, UNIQUE_NAME);

    auto icon = g_themed_icon_new(CATEGORY_ICON_PATH);
    auto cat = unity_category_new("popular", _("Popular online"), icon, UNITY_CATEGORY_RENDERER_VERTICAL_TILE);
    auto cats = unity_category_set_new();
    unity_category_set_add(cats, cat);
    cat = unity_category_new("online", _("Online"), icon, UNITY_CATEGORY_RENDERER_VERTICAL_TILE);
    unity_category_set_add(cats, cat);

    unity_simple_scope_set_category_set(m_unityScope, cats);

    unity_simple_scope_set_search_async_func(m_unityScope, search_func, reinterpret_cast<void *>(this), nullptr);
    unity_simple_scope_set_preview_async_func(m_unityScope, preview_func, reinterpret_cast<void *>(this), nullptr);

    g_object_unref(icon);
    unity_object_unref(cat);
    unity_object_unref(cats);
}

OnlineMusicScope& OnlineMusicScope::instance()
{
    static OnlineMusicScope m_scope;
    return m_scope;
}

UnityAbstractScope* OnlineMusicScope::getUnityScope()
{
    return UNITY_ABSTRACT_SCOPE(m_unityScope);
}

UnityAbstractPreview *OnlineMusicScope::previewFunc(UnityResultPreviewer *previewer)
{
    gpointer sessionId = g_hash_table_lookup(previewer->result.metadata, "session_id");
    if (sessionId == nullptr) {
        qWarning() << "Missing 'session_id'";
        return nullptr;
    }

    gpointer serverSid = g_hash_table_lookup(previewer->result.metadata, "server_sid");
    if (serverSid == nullptr) {
        qWarning() << "Missing 'server_sid'";
        return nullptr;
    }

    gpointer resultId = g_hash_table_lookup(previewer->result.metadata, "id");
    if (resultId == nullptr) {
        qWarning() << "Missing result id";
        return nullptr;
    }

    SmartScopesClient sss_client(m_locale);
    auto preview = sss_client.preview(QString(g_variant_get_string(static_cast<GVariant*>(serverSid), nullptr)),
                                      QString(g_variant_get_string(static_cast<GVariant*>(sessionId), nullptr)),
                                      QString(g_variant_get_string(static_cast<GVariant*>(resultId), nullptr)),
                                      previewer->result);
    return preview;
}

void OnlineMusicScope::appendIfEnabled(QStringList &scopeToQuery, const QString &scopeId) const
{
    if (!m_disabledScopes.contains(scopeId)) {
        scopeToQuery << scopeId;
    }
}

void OnlineMusicScope::searchFunc(UnityScopeSearchBase* search)
{
    // no search results in Home
    if (search->search_context->search_type == UNITY_SEARCH_TYPE_GLOBAL)
        return;

    QString serverSid;
    QString sessionId = randomizedTimeUuid();

    const QString query = QString(search->search_context->search_query).trimmed();
    unsigned cat = query.isEmpty() ? Category::POPULAR_ONLINE : Category::ONLINE;

    QStringList scopes;
    if (query.isEmpty()) {
        appendIfEnabled(scopes, "more_suggestions-populartracks.scope");
    } else {
        appendIfEnabled(scopes, "more_suggestions-u1ms.scope");
    }
    appendIfEnabled(scopes, "music-grooveshark.scope");
    appendIfEnabled(scopes, "music-soundcloud.scope");

    if (scopes.size() == 0) // no scopes to query?
        return;

    // maps scopeId to list of results
    std::map<QString, std::list<std::shared_ptr<SmartScopesResult>>> results;

    SmartScopesClient sss_client(m_locale);
    sss_client.search(query,
                      sessionId,
                      scopes,
                      QStringList(),
                      "music-onlinemusic.scope",
                      [search, sessionId, cat, &serverSid, &results](const QString& scopeId, std::shared_ptr<SmartScopesResult> result) {
                          if (!serverSid.isEmpty()) {
                              result->category = cat;
                              result->metadata.insert(QString("server_sid"), serverSid);
                              result->metadata.insert(QString("session_id"), sessionId);
                              if (results.find(scopeId) == results.end()) {
                                  results[scopeId] = std::list<std::shared_ptr<SmartScopesResult>>();
                              }
                              results[scopeId].push_back(result);                              
                          }
                      },
                      [&serverSid](const QString &sid) {
                          serverSid = sid;
                      },
                      search->search_context->cancellable);

    // redistribute results from all scopes (scopeA, scopeB, scopeA, scopeB ...)
    bool haveData = false;
    do
    {
        haveData = false;
        for (auto &r: results) // iterate over (scope, resultlist) pairs
        {
            auto &reslist = r.second;
            haveData = haveData || (!reslist.empty());
            if (!reslist.empty()) {
                // pop first result of this scope and add it to the output result set
                auto result = reslist.front();
                reslist.pop_front();
                auto unityResult = unity_scope_result_create_from_variant(result->toGVariant());
                unity_result_set_add_result(search->search_context->result_set, unityResult);
                unity_scope_result_free(unityResult);
            }
        }
    } while (haveData);

    if (serverSid.isEmpty()) {
        qWarning() << "server_sid is missing in the reply";
    }
}

void OnlineMusicScope::updateDisabledScopes()
{
    int count = 0;
    m_disabledScopes.clear();
    auto prefs = unity_preferences_manager_get_default();
    if (prefs) {
        auto disabled = unity_preferences_manager_get_disabled_scopes(prefs, &count);
        for (int i = 0; i<count; i++) {
            const QString scopeId(disabled[i]);
            m_disabledScopes.insert(scopeId);
        }
    }
}

void OnlineMusicScope::search_func(UnityScopeSearchBase* search, UnityScopeSearchBaseCallback cb, void* cb_target, void* user_data)
{
    auto scope = static_cast<OnlineMusicScope *>(user_data);
    scope->searchFunc(search);

    cb(search, cb_target);
}

void OnlineMusicScope::preview_func(UnityResultPreviewer *previewer, UnityAbstractPreviewCallback cb, void *cb_target, void *user_data)
{
    auto scope = static_cast<OnlineMusicScope *>(user_data);
    auto preview = scope->previewFunc(previewer);
    cb(previewer, preview, cb_target);
}

void OnlineMusicScope::disabled_scopes_changed(GObject* /* obj */, GParamSpec* /*spec*/, gpointer user_data)
{
    qDebug() << "disabled-scopes changed";
    auto scope = static_cast<OnlineMusicScope *>(user_data);
    scope->updateDisabledScopes();
}
