/**
 * Copyright (C) 2007-2013 Lawrence Murray
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * @author Lawrence Murray <lawrence@indii.org>
 * $Rev$
 * $Date$
 */
#include "ClusterModel.hpp"

#include "ClusterModelObserver.hpp"

//#include "wx/app.h"
//#include "wx/busyinfo.h"

#include <cassert>
#include <ctime>
#include <limits>

namespace ublas = boost::numeric::ublas;

using namespace indii;
using namespace std;

ClusterModel::ClusterModel(ImageResource* res) :
    Model(res), minClusterer(NULL), locked(false) {
  this->saturationThreshold = 0.0625f;
  this->maxPixels = 50000;
  this->reps = 12;
  this->k = 4;
  this->hard = 0.9f;
  this->saturationDecay = 0.0625f;
  this->centroidDecay = 0.0f;
  this->saturationSoftness = 0.125f;
  this->centroidSoftness = 0.0f;
  this->colours.resize(k, *wxWHITE);
  this->hues.resize(k, 0.0f);
  this->sats.resize(k, -1.0f);
  this->lights.resize(k, 0.0f);
  this->alphas.resize(k, 1.0f);
  this->minClusterer = new KMeansClusterer<>(k, time(NULL));
}

ClusterModel::~ClusterModel() {
  delete minClusterer;
}

void ClusterModel::setDefaults() {
  lock();
  setHard(0.9f);
  setSaturationThreshold(0.0625f);
  setMaxPixels(50000);
  setNumRepetitions(12);
  setNumClusters(4);
  setSaturationDecay(0.0625f);
  setCentroidDecay(0.0f);
  setSaturationSoftness(0.125f);
  setCentroidSoftness(0.0f);
  setChannelMix(0.299f, 0.587f, 0.114f);
  setForDialog();
  unlock();
  prepare();
  cluster();

  notifyAll();
  notifyNumClustersChange();
}

void ClusterModel::setForDialog() {
  fill(hues.begin(), hues.end(), 0.0f);
  fill(sats.begin(), sats.end(), -1.0f);
  fill(lights.begin(), lights.end(), 0.0f);
  fill(alphas.begin(), alphas.end(), 1.0f);
}

void ClusterModel::setNumClusters(const unsigned k) {
  if (this->k != k) {
    this->k = k;
    colours.resize(k, *wxWHITE);
    hues.resize(k);
    sats.resize(k);
    lights.resize(k);

    fill(sats.begin(), sats.end(), -1.0f);
    fill(hues.begin(), hues.end(), 0.0f);
    fill(lights.begin(), lights.end(), 0.0f);
    fill(alphas.begin(), alphas.end(), 1.0f);

    cluster();
    notifyAll();
    notifyNumClustersChange();
  }
}

void ClusterModel::setHard(const float hard) {
  /* pre-condition */
  assert(hard >= 0.0f && hard <= 1.0f);

  if (this->hard != hard) {
    this->hard = hard;
    notifyAll();
    notifyIsHardChange();
  }
}

void ClusterModel::setNumRepetitions(const unsigned reps) {
  if (this->reps != reps) {
    this->reps = reps;
    notifyAll();
    notifyNumRepetitionsChange();
  }
}

void ClusterModel::setSaturationThreshold(const float x) {
  if (this->saturationThreshold != x) {
    this->saturationThreshold = x;
    prepare();
    notifyAll();
    notifySaturationThresholdChange();
  }
}

void ClusterModel::setMaxPixels(const unsigned n) {
  if (this->maxPixels != n) {
    this->maxPixels = n;
    prepare();
    notifyAll();
    notifyMaxPixelsChange();
  }
}

void ClusterModel::setSaturationDecay(const float x) {
  /* pre-condition */
  assert(x >= 0.0f && x <= 1.0f);

  if (this->saturationDecay != x) {
    this->saturationDecay = x;
    notifyAll();
    notifySaturationDecayChange();
  }
}

void ClusterModel::setCentroidDecay(const float x) {
  /* pre-condition */
  assert(x >= 0.0f && x <= 1.0f);

  if (this->centroidDecay != x) {
    this->centroidDecay = x;
    notifyAll();
    notifyCentroidDecayChange();
  }
}

void ClusterModel::setSaturationSoftness(const float x) {
  /* pre-condition */
  assert(x >= 0.0 && x <= 1.0f);

  if (this->saturationSoftness != x) {
    this->saturationSoftness = x;
    notifyAll();
    notifySaturationSoftnessChange();
  }
}

void ClusterModel::setCentroidSoftness(const float x) {
  /* pre-condition */
  assert(x >= 0.0 && x <= 1.0f);

  if (this->centroidSoftness != x) {
    this->centroidSoftness = x;
    notifyAll();
    notifyCentroidSoftnessChange();
  }
}

void ClusterModel::setColour(const unsigned i, const wxColour& col) {
  /* pre-condition */
  assert(i < k);

  ClusterVector<>::type x(ClusterVector<>::N);
  x(0) = col.Red()/255.0f;
  x(1) = col.Green()/255.0f;
  x(2) = col.Blue()/255.0f;
  indii::KMeansClusterer<>::distance_type::prepare(x);

  colours[i] = col;
  minClusterer->setCentroid(i, x);
}

void ClusterModel::setHue(const unsigned i, const float x) {
  /* pre-condition */
  assert(i < k);
  assert(x >= -0.5f && x <= 0.5f);

  if (hues[i] != x) {
    hues[i] = x;
    notifyAll();
    notifyHueChange(i);
  }
}

void ClusterModel::setSat(const unsigned i, const float x) {
  /* pre-condition */
  assert(i < k);
  assert(x >= -1.0f && x <= 1.0f);

  if (sats[i] != x) {
    sats[i] = x;
    notifyAll();
    notifySatChange(i);
  }
}

void ClusterModel::setLight(const unsigned i, const float x) {
  /* pre-condition */
  assert(i < k);
  assert(x >= -1.0f && x <= 1.0f);

  if (lights[i] != x) {
    lights[i] = x;

    /* notify observers */
    notifyAll();
    notifyLightChange(i);
  }
}

void ClusterModel::setAlpha(const unsigned i, const float x) {
  /* pre-condition */
  assert(i < k);
  assert(x >= 0.0f && x <= 1.0f);

  if (alphas[i] != x) {
    alphas[i] = x;

    /* notify observers */
    notifyAll();
    notifyAlphaChange(i);
  }
}

void ClusterModel::show(const unsigned i, const bool on) {
  /* pre-condition */
  assert(i < k);

  if (isShown(i) != on) {
    if (on) {
      sats[i] = 0.0f;
    } else {
      sats[i] = -1.0f;
    }

    notifyAll();
    notifySatChange(i);
  }
}

void ClusterModel::showAll(const bool on) {
  unsigned i;
  for (i = 0; i < k; i++) {
    show(i, on);
  }
}

void ClusterModel::calcFg(const wxRect& rect, const float scale, Image& o) {
  /* pre-conditions */
  assert(o.getWidth() == rect.width);
  assert(o.getHeight() == rect.height);

  Image& img = *res->get(scale);

  #pragma omp parallel
  {
    ColourSpace::rgb_t rgb(3);
    ColourSpace::hsl_t hsl(3);
    std::vector<float> ds(k);

    int x, y, cluster;
    float decay, light, sat;

    #pragma omp for schedule(dynamic)
    for (y = 0; y < rect.height; ++y) {
      for (x = 0; x < rect.width; ++x) {
        rgb(0) = img.r(rect.y + y, rect.x + x);
        rgb(1) = img.g(rect.y + y, rect.x + x);
        rgb(2) = img.b(rect.y + y, rect.x + x);
        cs.rgb2hsl(rgb, hsl);
        decay = threshold(hsl(1), saturationDecay, saturationSoftness);

        /* cluster adjustments */
        indii::KMeansClusterer<>::distance_type::prepare(rgb);
        if (isHard()) {
          cluster = minClusterer->assign(rgb);

          /* hue */
          hsl(0) = fmod(1.0f + hsl(0) + hues[cluster], 1.0f);

          /* saturation */
          sat = decay * sats[cluster];
          if (sat >= 0.0f) {
            hsl(1) /= 1.0f - sat;
          } else {
            hsl(1) *= 1.0f + sat;
          }
          hsl(1) = bound(0.0f, decay * hsl(1), 1.0f);
          //hsl(1) = bound(0.0f, decay*(sats[cluster] + 1.0f)*hsl(1), 1.0f);

          /* lightness */
          light = lights[cluster];
          if (light >= 0.0f) {
            hsl(2) /= 1.0f - light;
          } else {
            hsl(2) *= 1.0f + light;
          }
          hsl(2) = bound(0.0f, hsl(2), 1.0f);

          /* alpha */
          //a = bound(0.0f, decay*alphas[cluster], 255.0f);
        } else {
          minClusterer->distances(rgb, ds);

          unsigned i;
          float h = 0.0f, s = 0.0f, l = 0.0f/*, a = 0.0f*/, D = 0.0f;
          for (i = 0; i < k; ++i) {
            ds[i] = std::pow(1.0f - ds[i], 256.0f * hard);
            D += ds[i];
          }
          if (D >= 0.0f) {
            for (i = 0; i < k; ++i) {
              ds[i] /= D;
              if (ds[i] > 0.0f && ds[i] <= 1.0f) { // D >= 0.0 check may use extended precision, so ds[i] may not be finite
                h += ds[i] * hues[i];
                s += ds[i] * sats[i];
                l += ds[i] * lights[i];
              }
            }
          }
          //a += ds[i]*alphas[i];

          /* hue */
          hsl(0) = fmod(1.0f + hsl(0) + h, 1.0f);

          /* saturation */
          sat = decay * s;
          if (sat >= 0.0f) {
            hsl(1) /= 1.0f - sat;
          } else {
            hsl(1) *= 1.0f + sat;
          }
          hsl(1) = bound(0.0f, decay * hsl(1), 1.0f);
          //hsl(1) = bound(0.0f, decay*(s + 1.0f)*hsl(1), 1.0f);

          /* lightness */
          light = l;
          if (light >= 0.0f) {
            hsl(2) /= 1.0f - light;
          } else {
            hsl(2) *= 1.0f + light;
          }
          hsl(2) = bound(0.0f, hsl(2), 1.0f);

          /* alpha */
          // already set
        }

        /* convert back */
        cs.hsl2rgb(hsl, rgb);
        o.r(y, x) = rgb(0);
        o.g(y, x) = rgb(1);
        o.b(y, x) = rgb(2);
        //o.a(y, x) = a;
      }
    }
  }
}

void ClusterModel::calcAlpha(const unsigned i, const wxRect& rect,
    const float scale, channel& c) {
  /* pre-conditions */
  assert(i < k);
  assert((int)c.size1() == rect.height && (int)c.size2() == rect.width);

  Image* img = res->get(scale);

#pragma omp parallel default(shared)
  {
    ColourSpace::rgb_t rgb(3);
    ColourSpace::hsl_t hsl(3);
    ClusterVector<>::type pixel(ClusterVector<>::N);

    int x, y;

#pragma omp for schedule(dynamic)
    for (y = 0; y < rect.height; ++y) {
      for (x = 0; x < rect.width; ++x) {
        rgb(0) = img->r(rect.y + y, rect.x + x);
        rgb(1) = img->g(rect.y + y, rect.x + x);
        rgb(2) = img->b(rect.y + y, rect.x + x);
        cs.rgb2hsl(rgb, hsl);
        pixel = rgb;

        /* cluster adjustments */
        indii::KMeansClusterer<>::distance_type::prepare(pixel);
        if (minClusterer->assign(pixel) == i) {
          c(y, x) = threshold(hsl(1), saturationDecay, saturationSoftness);
        } else {
          c(y, x) = 0;
        }
      }
    }
  }
}

sparse_mask ClusterModel::calcMask(const unsigned i, const wxRect& rect,
    const float scale) {
  /* pre-condition */
  assert(i < k);

  sparse_mask m(rect.height, rect.width);
  Image* img = res->get(scale);

#pragma omp parallel default(shared)
  {
    ColourSpace::rgb_t rgb(3);
    ColourSpace::hsl_t hsl(3);
    ClusterVector<>::type pixel(ClusterVector<>::N);
    int x, y;

#pragma omp for schedule(dynamic)
    for (y = 0; y < rect.height; ++y) {
      for (x = 0; x < rect.width; ++x) {
        rgb(0) = img->r(rect.y + y, rect.x + x);
        rgb(1) = img->g(rect.y + y, rect.x + x);
        rgb(2) = img->b(rect.y + y, rect.x + x);
        cs.rgb2hsl(rgb, hsl);
        pixel = rgb;

        indii::KMeansClusterer<>::distance_type::prepare(pixel);
        if (minClusterer->assign(pixel) == i) {
#pragma omp critical
          {
            /* sparse matrix, so need to control access, elements may be
             * rearranged */
            m(y, x) = true;
          }
        }
      }
    }
  }
  return m;
}

void ClusterModel::prepare() {
  if (locked) {
    return;
  }

  /* prepare data set for clustering */
  int originalWidth = res->getWidth();
  int originalHeight = res->getHeight();
  int pixels = originalWidth * originalHeight;
  float scale = static_cast<float>(pixels) / maxPixels;
  Image* working = res->getLow(scale);
  int workingWidth = working->getWidth();
  int workingHeight = working->getHeight();

  /* prepare data set for clustering */
  data.clear();
#pragma omp parallel
  {
    ClusterVector<>::type pixel(ClusterVector<>::N);
    int x, y;
    float s;

#pragma omp for
    for (x = 0; x < workingWidth; ++x) {
      for (y = 0; y < workingHeight; ++y) {
        pixel(0) = working->r(y, x);
        pixel(1) = working->g(y, x);
        pixel(2) = working->b(y, x);

        /* saturation threshold */
        s = saturation(pixel(0), pixel(1), pixel(2));
        if (s >= saturationThreshold) {
          indii::KMeansClusterer<>::distance_type::prepare(pixel);
#pragma omp critical
          {
            data.add(pixel);
          }
        }
      }
    }
  }
}

void ClusterModel::cluster() {
  if (locked) {
    return;
  }
  const unsigned MAX_ITERS = 100;
  const unsigned seed = time(NULL);
  float minError = std::numeric_limits<float>::max();

#pragma omp parallel shared(minError)
  {
    /* cluster */
    KMeansClusterer<>* clusterer;
    float error;
    int i;
    ClusterVector<>::type pixel(ClusterVector<>::N);

#pragma omp for schedule(dynamic)
    for (i = 0; i < (int) reps; i++) {
      clusterer = new KMeansClusterer<>(k, seed + i);
      clusterer->cluster(data, MAX_ITERS);
      error = clusterer->getError();
#pragma omp critical
      {
        if (error < minError) {
          minError = error;
          delete minClusterer;
          minClusterer = clusterer;
        } else {
          delete clusterer;
        }
        //wxYield();
      }
    }
    assert(minClusterer != NULL);

    /* cluster colours */
    ColourSpace cs;
    ColourSpace::rgb_t rgb(3);
    ColourSpace::hsl_t hsl(3);
#pragma omp for
    for (i = 0; i < (int) k; i++) {
      pixel = 0.5f * minClusterer->getCentroid(i)
          + ublas::scalar_vector<float>(3, 0.5f);
      rgb = pixel;
      cs.rgb2hsl(rgb, hsl);
      hsl(1) = 0.8f;
      hsl(2) = 0.8f;
      cs.hsl2rgb(hsl, rgb);
      colours[i].Set(uround(rgb(0)), uround(rgb(1)), uround(rgb(2)));
    }
  }
}
