//
// HashMap.h
//
// $Id: //poco/1.3/Foundation/include/Poco/HashMap.h#1 $
//
// Library: Foundation
// Package: Hashing
// Module:  HashMap
//
// Definition of the HashMap class.
//
// Copyright (c) 2006, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#ifndef Foundation_HashMap_INCLUDED
#define Foundation_HashMap_INCLUDED


#include "Poco/Foundation.h"
#include "Poco/LinearHashTable.h"
#include "Poco/Exception.h"
#include <utility>


namespace Poco {


template <class Key, class Value>
struct HashMapEntry
	/// This class template is used internally by HashMap.
{
	Key   first;
	Value second;
	
	HashMapEntry():
		first(),
		second()
	{
	}
	
	HashMapEntry(const Key& key):
		first(key),
		second()
	{
	}

	HashMapEntry(const Key& key, const Value& value):
		first(key),
		second(value)
	{
	}
	
	bool operator == (const HashMapEntry& entry) const
	{
		return first == entry.first;
	}

	bool operator != (const HashMapEntry& entry) const
	{
		return first != entry.first;
	}
};


template <class HME, class KeyHashFunc>
struct HashMapEntryHash
	/// This class template is used internally by HashMap.
{
	std::size_t operator () (const HME& entry) const
	{
		return _func(entry.first);
	}

private:
	KeyHashFunc _func;
};


template <class Key, class Mapped, class HashFunc = Hash<Key> >
class HashMap
	/// This class implements a map using a LinearHashTable.
	///
	/// A HashMap can be used just like a std::map.
{
public:
	typedef Key                 KeyType;
	typedef Mapped              MappedType;
	typedef Mapped&             Reference;
	typedef const Mapped&       ConstReference;
	typedef Mapped*             Pointer;
	typedef const Mapped*       ConstPointer;
	
	typedef HashMapEntry<Key, Mapped>      ValueType;
	typedef std::pair<KeyType, MappedType> PairType;
	
	typedef HashMapEntryHash<ValueType, HashFunc> HashType;
	typedef LinearHashTable<ValueType, HashType>  HashTable;
	
	typedef typename HashTable::Iterator      Iterator;
	typedef typename HashTable::ConstIterator ConstIterator;
	
	HashMap()
		/// Creates an empty HashMap.
	{
	}
	
	HashMap(std::size_t initialReserve):
		_table(initialReserve)
		/// Creates the HashMap with room for initialReserve entries.
	{
	}
	
	HashMap& operator = (const HashMap& map)
		/// Assigns another HashMap.
	{
		HashMap tmp(map);
		swap(tmp);
		return *this;
	}
	
	void swap(HashMap& map)
		/// Swaps the HashMap with another one.
	{
		_table.swap(map._table);
	}
	
	ConstIterator begin() const
	{
		return _table.begin();
	}
	
	ConstIterator end() const
	{
		return _table.end();
	}
	
	Iterator begin()
	{
		return _table.begin();
	}
	
	Iterator end()
	{
		return _table.end();
	}
	
	ConstIterator find(const KeyType& key) const
	{
		ValueType value(key);
		return _table.find(value);
	}

	Iterator find(const KeyType& key)
	{
		ValueType value(key);
		return _table.find(value);
	}

	std::pair<Iterator, bool> insert(const PairType& pair)
	{
		ValueType value(pair.first, pair.second);
		return _table.insert(value);
	}

	std::pair<Iterator, bool> insert(const ValueType& value)
	{
		return _table.insert(value);
	}
	
	void erase(Iterator it)
	{
		_table.erase(it);
	}
	
	void erase(const KeyType& key)
	{
		Iterator it = find(key);
		_table.erase(it);
	}
	
	void clear()
	{
		_table.clear();
	}

	std::size_t size() const
	{
		return _table.size();
	}

	bool empty() const
	{
		return _table.empty();
	}

	ConstReference operator [] (const KeyType& key) const
	{
		ConstIterator it = _table.find(key);
		if (it != _table.end())
			return it->second;
		else
			throw NotFoundException();
	}

	Reference operator [] (const KeyType& key)
	{
		ValueType value(key);
		std::pair<Iterator, bool> res = _table.insert(value);
		return res.first->second;
	}

private:
	HashTable _table;
};


} // namespace Poco


#endif // Foundation_HashMap_INCLUDED
