<?xml version="1.0"?>
<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
                xmlns:func="http://exslt.org/functions"
                xmlns:pml="http://wohlberg.net/xml/photoml"
                exclude-result-prefixes="func pml"
                extension-element-prefixes="func"
                version="1.0">

<!-- 
     This file contains templates for converting a limited subset of
     PhotoML data to XMP format.

     Copyright © 2010,2011 Brendt Wohlberg <photoml@wohlberg.net>

     This is free software; you can redistribute it and/or modify it 
     under the terms of version 2 of the GNU General Public License 
     at http://www.gnu.org/licenses/gpl-2.0.txt.

     This software is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the 
     GNU General Public License for more details.
-->

  <xsl:output method="xml" omit-xml-declaration="yes" indent="yes"
              encoding="utf-8"/>


  <!-- Stylesheet parameter determining output path -->
  <xsl:param name="output-path"></xsl:param>
  <!-- Stylesheet parameter determining ID string format -->
  <xsl:param name="idstring-format">%gf%f</xsl:param>
  <!-- Stylesheet parameter selecting a group id for output -->
  <xsl:param name="gid-param"></xsl:param>
  <!-- Stylesheet parameter selecting a frame id for output -->
  <xsl:param name="fid-param"></xsl:param>


  <!-- Top level template matching document root -->
  <xsl:template match="/">
    <!-- The gid variable takes on the value of the gid-param
    parameter if it is not an empty string. If it is empty, and there
    is only a single id attribute of a roll, sheet, or digital
    element, it is set to the value of that attribute. Otherwise, it
    is set to an empty string. -->
    <xsl:variable name="gid">
      <xsl:choose>
        <xsl:when test="$gid-param != ''">
          <xsl:value-of select="$gid-param"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:choose>
            <xsl:when test="count((//roll|//sheet|//digital)/@id) = 1">
              <xsl:value-of select="(//roll|//sheet|//digital)/@id[1]"/>
            </xsl:when>
            <xsl:otherwise>
              <xsl:text></xsl:text>
            </xsl:otherwise>
          </xsl:choose>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>
    <!-- The fid variable takes on the value of the fid-param parameter. -->
    <xsl:variable name="fid" select="$fid-param"/>
    <!-- If the gid or fid variable is an empty string, apply the
    template matching frame elements with mode "multiple" to all frame
    elements which are not children of a defaults element, otherwise
    apply the template matching frame elements with mode "single" to
    the single frame matching the gid and fid variables. -->
    <xsl:choose>
      <xsl:when test="$gid = '' or $fid = ''">
        <xsl:apply-templates select="//frame[not(parent::defaults)]"
                             mode="multiple"/>
      </xsl:when>
      <xsl:otherwise>
         <xsl:apply-templates select="//frame[@id=$fid or 
                                              number(@id)=number($fid)]
                [(ancestor::roll|ancestor::sheet|ancestor::digital)/@id=$gid]" 
           mode="single"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>


  <!-- Template matching frame elements, processing in "single" mode,
  with standard output. -->
  <xsl:template match="//frame[not(parent::defaults)]" mode="single">
    <!-- Set variable with relevant gid value -->
    <xsl:variable name="gid" 
      select="(ancestor::roll|ancestor::sheet|ancestor::digital)/@id"/>
    <xsl:call-template name="construct-xmp"/>    
  </xsl:template>

  
  <!-- Template matching frame elements, processing in "single" mode,
       with output to a document determined by gid and fid values. -->
  <xsl:template match="//frame[not(parent::defaults)]" mode="multiple">
    <!-- Set variable with relevant gid value -->
    <xsl:variable name="gid" 
      select="(ancestor::roll|ancestor::sheet|ancestor::digital)/@id"/>
    <!-- Set variable with output filename -->
    <xsl:variable name="filename" 
      select="concat($output-path, pml:replace(
                       pml:replace($idstring-format, '%g',$gid),
                       '%f',@id),'.xmp')"/>
    <!-- Send output to specified filename -->
    <xsl:document href="{$filename}" method="xml" omit-xml-declaration="yes"
      indent="yes">
      <xsl:call-template name="construct-xmp"/>
    </xsl:document>
  </xsl:template>


  <!-- Named template doing actual XMP format output -->
  <xsl:template name="construct-xmp">
    
    <xsl:variable name="datetime">
      <xsl:value-of select="scene/occasion/date"/>
      <xsl:if test="scene/occasion/time">
        <xsl:text>T</xsl:text>
        <xsl:value-of select="scene/occasion/time"/>
        <xsl:value-of select="scene/occasion/time/@zone"/>
      </xsl:if>
    </xsl:variable>
    <xsl:variable name="gid" 
      select="(ancestor::roll|ancestor::sheet|ancestor::digital)/@id"/>
    <xsl:variable name="docid" select="pml:replace(
                                   pml:replace($idstring-format, '%g',$gid),
                                   '%f',@id)"/>
    
    <xsl:processing-instruction name="xpacket">
      <xsl:text>begin='&#xFEFF;' id='W5M0MpCehiHzreSzNTczkc9d'</xsl:text>
    </xsl:processing-instruction>
    <xsl:text>&#10;</xsl:text>
    <x:xmpmeta xmlns:x='adobe:ns:meta/'>
      <rdf:RDF xmlns:rdf='http://www.w3.org/1999/02/22-rdf-syntax-ns#'>

        <xsl:if test="scene/location/*[name()!='country' and 
                      name()!='region' and name()!='coordinates']">
          <xsl:comment/> <!-- Hack required for blank line -->
          <rdf:Description rdf:about=''
            xmlns:Iptc4xmpCore='http://iptc.org/std/Iptc4xmpCore/1.0/xmlns/'>
            <Iptc4xmpCore:Location>
              <xsl:for-each select="scene/location/*[name()!='country' and 
                                  name()!='region' and name()!='coordinates']">
                <xsl:sort select="position()" data-type="text" 
                          order="descending"/>
                <xsl:value-of select="."/>
                <xsl:if test="position()!=last()">
                  <xsl:text>, </xsl:text>
                </xsl:if>
              </xsl:for-each>
            </Iptc4xmpCore:Location>
          </rdf:Description>
        </xsl:if>

        <xsl:comment/> <!-- Hack required for blank line -->
        <rdf:Description rdf:about=''
          xmlns:exif='http://ns.adobe.com/exif/1.0/'>
          <exif:DocumentName>
            <xsl:value-of select="$docid"/>
          </exif:DocumentName> 
          <xsl:if test="photographer/name">
            <exif:Artist>
              <xsl:value-of select="photographer/name"/>
              <xsl:if test="photographer/email">
                <xsl:text> &lt;</xsl:text>
                <xsl:value-of select="photographer/email"/>
                <xsl:text>&gt;</xsl:text>
              </xsl:if>
            </exif:Artist>
          </xsl:if>
          <xsl:if test="ipr/copyright or ipr/owner/name">
            <exif:Copyright>
              <xsl:if test="ipr/owner/name">
                <xsl:value-of select="ipr/owner/name"/>
              </xsl:if>
              <xsl:if test="ipr/owner/email">
                <xsl:text> &lt;</xsl:text>
                <xsl:value-of select="ipr/owner/email"/>
                <xsl:text>&gt;</xsl:text>
              </xsl:if>
              <xsl:if test="ipr/copyright">
                <xsl:text>, </xsl:text>
                <xsl:value-of select="ipr/copyright"/>
              </xsl:if>
            </exif:Copyright>
          </xsl:if>
          <xsl:if test="evaluation/@score">
            <exif:Rating>
              <xsl:value-of select="evaluation/@score"/>
            </exif:Rating>
          </xsl:if>

          <xsl:if test="scene/description">
            <exif:UserComment>
              <xsl:value-of select="scene/description"/>
            </exif:UserComment>
          </xsl:if>
          <xsl:if test="scene/caption">
            <exif:ImageDescription>
              <xsl:value-of select="scene/caption"/>
            </exif:ImageDescription>
          </xsl:if>
          <xsl:if test="scene/occasion/date">
            <exif:DateTimeOriginal>
              <xsl:value-of select="$datetime"/>
            </exif:DateTimeOriginal>
            <!-- For a digital camera image, the original date/time
                 and digitized date/time are the same -->
            <xsl:if test="ancestor::digital">
              <exif:DateTimeDigitized>
                <xsl:value-of select="$datetime"/>
              </exif:DateTimeDigitized>
            </xsl:if>
          </xsl:if>
          <xsl:if test="scene/location/coordinates/altitude">
            <exif:GPSAltitude>
              <xsl:value-of select="pml:rational(pml:abs(
                                    scene/location/coordinates/altitude
                                    ))"/>
            </exif:GPSAltitude>
            <exif:GPSAltitudeRef>
              <xsl:choose>
                <xsl:when test="scene/location/coordinates/altitude&gt;=0">
                  <xsl:text>0</xsl:text>
                </xsl:when>
                <xsl:otherwise>
                  <xsl:text>1</xsl:text>
                </xsl:otherwise>
              </xsl:choose>
            </exif:GPSAltitudeRef>
          </xsl:if>
          <xsl:if test="scene/location/coordinates/latitude">
            <exif:GPSLatitude>
              <xsl:value-of select="pml:latdegtodegmin(
                                    scene/location/coordinates/latitude
                                    )"/>
            </exif:GPSLatitude>
          </xsl:if>
          <xsl:if test="scene/location/coordinates/longitude">
            <exif:GPSLongitude>
              <xsl:value-of select="pml:lngdegtodegmin(
                                    scene/location/coordinates/longitude
                                    )"/>
            </exif:GPSLongitude>
          </xsl:if>
          <xsl:if test="scene/location/coordinates/@datum">
            <exif:GPSMapDatum>
              <xsl:choose>
                <xsl:when test="scene/location/coordinates/@datum='WGS 84'">
                  <xsl:text>WGS-84</xsl:text>
                </xsl:when>
                <xsl:otherwise>
                  <xsl:value-of select="scene/location/coordinates/@datum"/>
                </xsl:otherwise>
              </xsl:choose>
            </exif:GPSMapDatum>
          </xsl:if>
        </rdf:Description>

        <xsl:if test="scene/location/country or scene/location/region">
          <xsl:comment/> <!-- Hack required for blank line -->
          <rdf:Description rdf:about=''
            xmlns:photoshop='http://ns.adobe.com/photoshop/1.0/'>
            <xsl:if test="scene/location/country">
              <photoshop:Country>
                <xsl:value-of select="scene/location/country"/>
              </photoshop:Country>
            </xsl:if>
            <xsl:if test="scene/location/region">
              <photoshop:State>
                <xsl:value-of select="scene/location/region"/>
              </photoshop:State>
            </xsl:if>
          </rdf:Description>
        </xsl:if>

        <!-- Make and model refer to scanner for roll and sheet frames  -->
        <xsl:if test="ancestor::digital and (equipment/body/make or 
                      equipment/body/model)">
          <xsl:comment/> <!-- Hack required for blank line -->
          <rdf:Description rdf:about=''
            xmlns:tiff='http://ns.adobe.com/tiff/1.0/'>
            <xsl:if test="equipment/body/make">
              <tiff:Make>
                <xsl:value-of select="equipment/body/make"/>
              </tiff:Make>
            </xsl:if>
            <xsl:if test="equipment/body/model">
              <tiff:Model>
                <xsl:value-of select="equipment/body/model"/>
              </tiff:Model>
            </xsl:if>
          </rdf:Description>
        </xsl:if>

        <!-- The occasion/date only corresponds to xmp:CreateDate for
             children of the digital element -->
        <xsl:if test="ancestor::digital and scene/occasion/date">
          <xsl:comment/> <!-- Hack required for blank line --> 
          <rdf:Description rdf:about=''
            xmlns:xmp='http://ns.adobe.com/xap/1.0/'>
            <xmp:CreateDate>
              <xsl:value-of select="$datetime"/>
            </xmp:CreateDate>
            <!-- Since xmp:ModifyDate is a property of a specific image version,                 it is safer not to write it out here -->
            <!-- <xmp:ModifyDate>
              <xsl:value-of select="$datetime"/>
            </xmp:ModifyDate> -->
          </rdf:Description>
        </xsl:if>

        <xsl:comment/> <!-- Hack required for blank line -->
      </rdf:RDF>
    </x:xmpmeta>
    <xsl:text>&#10;</xsl:text>
    <xsl:processing-instruction name="xpacket">
      <xsl:text>end='w'</xsl:text>
    </xsl:processing-instruction>

  </xsl:template>



  <!-- Extension function ... -->
  <func:function name="pml:replace">
    <xsl:param name="str" select="''"/>
    <xsl:param name="fnd" select="''"/>
    <xsl:param name="rpl" select="''"/>

    <func:result>
      <xsl:choose>
        <xsl:when test="contains($str,$fnd)">
          <xsl:value-of select="concat(substring-before($str,$fnd),
                                $rpl,
                                substring-after($str,$fnd))"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="$str"/>
        </xsl:otherwise>
      </xsl:choose>
    </func:result>
  </func:function>


 <!-- Extension function ... -->
  <func:function name="pml:rational">
    <xsl:param name="x" select="0"/>

    <func:result>
      <xsl:choose>
        <xsl:when test="$x = 0">
          <xsl:value-of select="string(0)"/>
        </xsl:when>
        <xsl:when test="round($x) = $x">
          <xsl:value-of select="string($x)"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="concat(string(round(100.0 * $x)),'/100')"/>
        </xsl:otherwise>
      </xsl:choose>
    </func:result>
  </func:function>


 <!-- Extension function ... -->
  <func:function name="pml:latdegtodegmin">
    <xsl:param name="x" select="0"/>

    <xsl:variable name="deg" select="pml:fractrunc($x)"/>
    <xsl:variable name="min" select="60.0*pml:abs($x - pml:fractrunc($x))"/>
    <xsl:variable name="ind">
      <xsl:choose>
        <xsl:when test="$x &gt;= 0">
          <xsl:text>N</xsl:text>
        </xsl:when>
        <xsl:otherwise>
          <xsl:text>S</xsl:text>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>

    <func:result>
      <xsl:value-of select="concat($deg,',',
                            format-number($min,'00.000000'),$ind)"/>
    </func:result>
  </func:function>


 <!-- Extension function ... -->
  <func:function name="pml:lngdegtodegmin">
    <xsl:param name="x" select="0"/>

    <xsl:variable name="deg" select="pml:fractrunc($x)"/>
    <xsl:variable name="min" select="60.0*pml:abs($x - pml:fractrunc($x))"/>
    <xsl:variable name="ind">
      <xsl:choose>
        <xsl:when test="$x &gt;= 0">
          <xsl:text>E</xsl:text>
        </xsl:when>
        <xsl:otherwise>
          <xsl:text>W</xsl:text>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>

    <func:result>
      <xsl:value-of select="concat($deg,',',
                            format-number($min,'00.000000'),$ind)"/>
    </func:result>
  </func:function>


 <!-- Extension function ... -->
  <func:function name="pml:fractrunc">
    <xsl:param name="x" select="0"/>

    <func:result>
      <xsl:choose>
        <xsl:when test="$x &gt;= 0">
          <xsl:value-of select="floor($x)"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="ceiling($x)"/>
        </xsl:otherwise>
      </xsl:choose>
    </func:result>
  </func:function>


  <!-- Extension function ... -->
  <func:function name="pml:abs">
    <xsl:param name="x" select="0"/>

    <func:result>
      <xsl:choose>
        <xsl:when test="$x &gt;= 0">
          <xsl:value-of select="$x"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="-$x"/>
        </xsl:otherwise>
      </xsl:choose>
    </func:result>
  </func:function>



</xsl:stylesheet>
