/*
    scconfig - command line argument processing
    Copyright (C) 2009..2012  Tibor Palinkas

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

		Project page: http://repo.hu/projects/scconfig
		Contact via email: scconfig [at] igor2.repo.hu
*/

#include <stdlib.h>
#include <string.h>
#include "db.h"
#include "arg.h"
#include "dep.h"
#include "log.h"
#include "libs.h"

typedef struct {
	char *arg;
	char *path;
	int (*callback)(const char *key, const char *value);
} argtbl_t;

argtbl_t argument_table[] = {
	{"import",      NULL,                   import_args},
	{"target",      "/arg/sys/target",      NULL},
	{"target-name", "/arg/sys/target-name", NULL},
	{"target-shell","/arg/sys/target-shell",NULL},
	{"emu",         "/arg/sys/emu",         NULL},
	{"pkg-config",  "/arg/sys/pkg-config",  NULL},
	{"pkg-config-zap","/arg/sys/pkg-config-zap",NULL},

/* the followings are autoconf compatibility translations */
	{"CC",          "/arg/cc/cc",           NULL},
	{"CFLAGS",      "/arg/cc/cflags",       NULL},
	{"LDFLAGS",     "/arg/cc/ldflags",      NULL},
	{"LDL",         "/arg/libs/ldl",        NULL},

	{"gpmi-prefix", "/arg/gpmi/prefix",     NULL},
	{NULL,          NULL,                   NULL}
};

void process_args(int argc, char *argv[])
{
	int n;
	char *key, *value;
	argtbl_t *a;
	int found, tainted = 0;

	db_mkdir("/arg");

	logprintf(0, "CLI arg 0: '%s'\n", argv[0]);
	for(n = 1; n < argc; n++) {
		key = argv[n];
		logprintf(0, "CLI arg %d: '%s'\n", n, key);
		while(*key == '-') key++;
		value = str_chr(key, '=');
		found = 0;
		if (value != NULL) {
			*value = '\0';
			value++;
			/* Look in the argument translate table */
			for(a = argument_table; a->arg != NULL; a++) {
				if (strcmp(a->arg, key) == 0) {
					found = 1;
					if (a->callback != NULL) {
						if (!(a->callback(key, value))) {
							error("Processing argument '%s' failed in the callback\n", argv[n]);
							abort();
						}
					}
					if (a->path != NULL)
						put(a->path, value);
				}
			}
			/* Look in known deps table or /arg */
			if (found == 0) {
				if ((is_dep_known(key)) || (strncmp(key, "/arg/", 5) == 0)) {
					put(key, value);
					found = 1;
				}
			}
		}
		if (found == 0) {
			if (custom_arg(key, value) == 0) {
				error("Unknown argument '%s'\n", key);
				tainted++;
			}
		}
	}
	if (tainted)
		exit(1);
}
