/*
 * This file is part of Flowplayer, http://flowplayer.org
 *
 * By: Anssi Piirainen, <support@flowplayer.org>
 * Copyright (c) 2008-2011 Flowplayer Oy *
 * Released under the MIT License:
 * http://www.opensource.org/licenses/mit-license.php
 */
package org.flowplayer.content {
    import flash.display.DisplayObject;
    import flash.events.MouseEvent;
    import flash.filters.DropShadowFilter;

    import org.flowplayer.controller.ResourceLoader;
    import org.flowplayer.model.DisplayPluginModel;
    import org.flowplayer.model.Plugin;
    import org.flowplayer.model.PluginEventType;
    import org.flowplayer.model.PluginModel;
    import org.flowplayer.util.Assert;
    import org.flowplayer.view.AbstractSprite;
    import org.flowplayer.view.FlowStyleSheet;
    import org.flowplayer.view.Flowplayer;
    import org.flowplayer.view.Styleable;

    /**
     * Content plugin.
     *
     * @author api
     */
    public class Content extends AbstractSprite implements Plugin, Styleable {
        private var _styleSheetFile:String;
        private var _player:Flowplayer;
        private var _model:DisplayPluginModel;
        private var _contentView:ContentView;
        private var _closeButton:CloseButton;
        private var _html:String;
        private var _useCloseButton:Boolean = false;
        private var _closeImageUrl:String;
        private var _dropShadow:Boolean;
        private var _originalAlpha:Number;

        public function Content() {
            addListeners();
        }

        internal function addListeners():void {
            addEventListener(MouseEvent.ROLL_OVER, onMouseOver);
            addEventListener(MouseEvent.ROLL_OUT, onMouseOut);
            addEventListener(MouseEvent.CLICK, onClick);
        }

        internal function removeListeners():void {
            removeEventListener(MouseEvent.ROLL_OVER, onMouseOver);
            removeEventListener(MouseEvent.ROLL_OUT, onMouseOut);
            removeEventListener(MouseEvent.CLICK, onClick);
        }

        override protected function onResize():void {
            if (!_contentView) return;
            arrangeCloseButton();
            _contentView.setSize(width, height);
            _contentView.x = 0;
            _contentView.y = 0;
        }

        /**
         * Sets the plugin model. This gets called before the plugin
         * has been added to the display list and before the player is set.
         * @param plugin
         */
        public function onConfig(plugin:PluginModel):void {
            _model = plugin as DisplayPluginModel;
            if (plugin.config) {
                log.debug("config object received with html " + plugin.config.html + ", stylesheet " + plugin.config.stylesheet);
                _styleSheetFile = plugin.config.stylesheet;
				if(plugin.config.callType != "URL")
                	_html = plugin.config.html;
                _useCloseButton = plugin.config.closeButton;
                _closeImageUrl = plugin.config.closeImage;
                _dropShadow = plugin.config.dropShadow;
            }
        }

        /**
         * Sets the Flowplayer interface. The interface is immediately ready to use, all
         * other plugins have been loaded an initialized also.
         * @param player
         */
        public function onLoad(player:Flowplayer):void {
            log.info("set player");
            _player = player;
            if (_styleSheetFile || _closeImageUrl) {
                loadResources(_styleSheetFile, _closeImageUrl);
            } else {
                createContentView(null, null);
                if (_useCloseButton) {
                    createCloseButton();
                }
                _model.dispatchOnLoad();
            }
        }

        /**
         * Sets the HTML content.
         * @param htmlText
         */
        [External]
        public function set html(htmlText:String):void {
            log.debug("set hetml()");
            _contentView.html = htmlText;
        }

        public function get html():String {
            log.debug("get hetml()");
            return _contentView.html;
        }

        /**
         * Appends HTML text to the content.
         * @param htmlText
         * @return the new text after append
         */
        [External]
        public function append(htmlText:String):String {
            log.debug("apped()");
            return _contentView.append(htmlText);
        }

        /**
         * Loads a new stylesheet and changes the style from the loaded sheet.
         */
        [External]
        public function loadStylesheet(styleSheetFile:String):void {
            if (! styleSheetFile) return;
            log.info("loading stylesheet from " + styleSheetFile);
            loadResources(styleSheetFile);
        }

        /**
         * Sets style properties.
         */
        public function css(styleProps:Object = null):Object {
            var result:Object = _contentView.css(styleProps);
            return result;
        }

        public function get style():FlowStyleSheet {
            return _contentView ? _contentView.style : null;
        }

        public function set style(value:FlowStyleSheet):void {
            Assert.notNull(_contentView, "content view not created yet");
            _contentView.style = value;
        }

        private function loadResources(styleSheetFile:String = null, imageFile:String = null):void {
            var loader:ResourceLoader = _player.createLoader();
            if (styleSheetFile) {
                log.debug("loading stylesheet from file " + _styleSheetFile);
            }
            if (imageFile) {
                log.debug("loading closeImage from file " + _closeImageUrl);
            }
            if (styleSheetFile) {
                loader.addTextResourceUrl(styleSheetFile);
            }
            if (imageFile) {
                loader.addBinaryResourceUrl(imageFile);
            }
            loader.load(null, onResourcesLoaded);
        }

        private function onResourcesLoaded(loader:ResourceLoader):void {
            if (_contentView) {
                if (_styleSheetFile) {
                    _contentView.style = createStyleSheet(loader.getContent(_styleSheetFile) as String);
                }
            } else {
                createContentView(_styleSheetFile ? loader.getContent(_styleSheetFile) as String : null, _closeImageUrl ? loader.getContent(_closeImageUrl) as DisplayObject : null);
            }
            if (_useCloseButton) {
                closeImage = loader.getContent(_closeImageUrl) as DisplayObject;
            }
            _model.dispatchOnLoad();
        }

        private function createStyleSheet(cssText:String = null):FlowStyleSheet {
            var styleSheet:FlowStyleSheet = new FlowStyleSheet("#content", cssText);
            // all root style properties come in config root (backgroundImage, backgroundGradient, borderRadius etc)
            addRules(styleSheet, _model.config);
            // style rules for the textField come inside a style node
            addRules(styleSheet, _model.config.style);
            return styleSheet;
        }

        private function addRules(styleSheet:FlowStyleSheet, rules:Object):void {
            var rootStyleProps:Object;
            for (var styleName:String in rules) {
                log.debug("adding additional style rule for " + styleName);
                if (FlowStyleSheet.isRootStyleProperty(styleName)) {
                    if (! rootStyleProps) {
                        rootStyleProps = new Object();
                    }
                    log.debug("setting root style property " + styleName + " to value " + rules[styleName]);
                    rootStyleProps[styleName] = rules[styleName];
                } else {
                    styleSheet.setStyle(styleName, rules[styleName]);
                }
            }
            styleSheet.addToRootStyle(rootStyleProps);
        }

        private function createContentView(cssText:String = null, closeImage:DisplayObject = null):void {
            log.debug("creating content view");
            _contentView = new ContentView(_player);
            log.debug("callign onResize");
            onResize(); // make it correct size before adding to display list (avoids unnecessary re-arrangement)
            log.debug("setting stylesheet " + cssText);
            _contentView.style = createStyleSheet(cssText);
            log.debug("setting html");
            _contentView.html = _html;
            log.debug("adding to display list");

            // distance, angle, color, alpha, blurX, blurY, strength,quality,inner,knockout
            if (_dropShadow) {
                _contentView.filters = [new DropShadowFilter(3, 45, 0x777777, 0.8, 15, 15, 2, 3)];
            }

            addChild(_contentView);
        }

        public override function set alpha(value:Number):void {
            super.alpha = value;
            if (!_contentView) return;
            _contentView.alpha = value;
        }

        private function onMouseOver(event:MouseEvent):void {
            if (!_model) return;
            if (_contentView.redrawing) return;
            _model.dispatch(PluginEventType.PLUGIN_EVENT, "onMouseOver");
        }

        private function onMouseOut(event:MouseEvent):void {
            if (!_model) return;
            _model.dispatch(PluginEventType.PLUGIN_EVENT, "onMouseOut");
        }

        private function onClick(event:MouseEvent):void {
            if (!_model) return;
            _model.dispatch(PluginEventType.PLUGIN_EVENT, "onClick");
        }

        public function getDefaultConfig():Object {
            return { top: 10, left: '50%', width: '95%', height: 50, opacity: 0.9, borderRadius: 10, backgroundGradient: 'low' };
        }

        public function animate(styleProps:Object):Object {
            return _contentView.animate(styleProps);
        }

        public function onBeforeCss(styleProps:Object = null):void {
        }

        public function onBeforeAnimate(styleProps:Object):void {
        }

        private function set closeImage(image:DisplayObject):void {
            if (_closeButton) {
                removeChild(_closeButton);
            }
            createCloseButton(image);
        }

        private function arrangeCloseButton():void {
            if (_closeButton && style) {
                _closeButton.x = width - _closeButton.width - 1 - style.borderRadius / 5;
                _closeButton.y = 1 + style.borderRadius / 5;
                setChildIndex(_closeButton, numChildren - 1);
            }
        }

        private function createCloseButton(icon:DisplayObject = null):void {
            _closeButton = new CloseButton(icon);
            addChild(_closeButton);
            _closeButton.addEventListener(MouseEvent.CLICK, onCloseClicked);
        }

        private function onCloseClicked(event:MouseEvent):void {
            this.removeListeners();
            _originalAlpha = this.alpha;
            _player.animationEngine.fadeOut(this, 500, onFadeOut);
        }

        private function onFadeOut():void {
            log.debug("faded out");
            //
            // restore original alpha value
            _model.alpha = _originalAlpha;
            _model.getDisplayObject().alpha = _originalAlpha;
            // we need to update the properties to the registry, so that animations happen correctly after this
            _player.pluginRegistry.updateDisplayProperties(_model);

            this.addListeners();
        }
    }
}
