!--------------------------------------------------------------------------------------------------!
!   CP2K: A general program to perform molecular dynamics simulations                              !
!   Copyright 2000-2021 CP2K developers group <https://cp2k.org>                                   !
!                                                                                                  !
!   SPDX-License-Identifier: GPL-2.0-or-later                                                      !
!--------------------------------------------------------------------------------------------------!

! **************************************************************************************************
!> \brief Contains routines for contraction without dgemms. PLEASE DO NOT MODIFY.
!> \notes Contains specific routines for contraction. The compiler flag
!>        -D__MAX_CONTR defines the maximum angular momentum up to which
!>        specialized code will be used. Default setting is d-functions.
!>        Increasing -D__MAX_CONTR produces faster code but might overburden
!>        the optimization capabilities of some poor compilers.
!>        This file contains specific code up to g-functions. If you need more
!>        look at cp2k/tools/hfx_tools/contraction/
!> \par History
!>      07.2009 created [Manuel Guidon]
!> \author Manuel Guidon
! **************************************************************************************************

MODULE hfx_contraction_methods

!** This defines the default behaviour
#ifndef __MAX_CONTR
#define __MAX_CONTR 2
#endif

   USE kinds, ONLY: dp
#include "../base/base_uses.f90"

   IMPLICIT NONE

   PRIVATE
   PUBLIC :: contract

CONTAINS

! **************************************************************************************************
!> \brief ...
!> \param ncoa ...
!> \param ncob ...
!> \param ncoc ...
!> \param ncod ...
!> \param nsoa ...
!> \param nsob ...
!> \param nsoc ...
!> \param nsod ...
!> \param n_a ...
!> \param n_b ...
!> \param n_c ...
!> \param n_d ...
!> \param nl_a ...
!> \param nl_b ...
!> \param nl_c ...
!> \param nl_d ...
!> \param work ...
!> \param sphi_a ...
!> \param sphi_b ...
!> \param sphi_c ...
!> \param sphi_d ...
!> \param primitives ...
!> \param buffer1 ...
!> \param buffer2 ...
! **************************************************************************************************
   SUBROUTINE contract(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                       n_a, n_b, n_c, n_d, nl_a, nl_b, nl_c, nl_d, work, &
                       sphi_a, sphi_b, sphi_c, sphi_d, &
                       primitives, &
                       buffer1, buffer2)

      INTEGER, INTENT(IN)         :: ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                     n_a, n_b, n_c, n_d, nl_a, nl_b, nl_c, nl_d
      REAL(dp), DIMENSION(ncoa*ncob*ncoc*ncod), INTENT(IN) :: work
      REAL(dp), DIMENSION(ncoa, nsoa*nl_a), INTENT(IN)   :: sphi_a
      REAL(dp), DIMENSION(ncob, nsob*nl_b), INTENT(IN)   :: sphi_b
      REAL(dp), DIMENSION(ncoc, nsoc*nl_c), INTENT(IN)   :: sphi_c
      REAL(dp), DIMENSION(ncod, nsod*nl_d), INTENT(IN)   :: sphi_d

      REAL(dp), DIMENSION(nsoa*nl_a, nsob*nl_b, nsoc*nl_c, nsod*nl_d) :: primitives
      REAL(dp), DIMENSION(ncoa*ncob*ncoc*ncod)  :: buffer1, buffer2

#if !defined(__LIBINT)
      MARK_USED(ncoa)
      MARK_USED(ncob)
      MARK_USED(ncoc)
      MARK_USED(ncod)
      MARK_USED(nsoa)
      MARK_USED(nsob)
      MARK_USED(nsoc)
      MARK_USED(nsod)
      MARK_USED(n_a)
      MARK_USED(n_b)
      MARK_USED(n_c)
      MARK_USED(n_d)
      MARK_USED(nl_a)
      MARK_USED(nl_b)
      MARK_USED(nl_c)
      MARK_USED(nl_d)
      MARK_USED(sphi_a)
      MARK_USED(sphi_b)
      MARK_USED(sphi_c)
      MARK_USED(sphi_d)
      MARK_USED(work)
      MARK_USED(primitives)
      MARK_USED(buffer1)
      MARK_USED(buffer2)
      CPABORT("libint not compiled in")
#else
      SELECT CASE (n_a)
      CASE (0)
         SELECT CASE (n_b)
         CASE (0)
            SELECT CASE (n_c)
            CASE (0)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 0 || __MAX_CONTR == 0
                  CALL contract_ssss(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 1 || __MAX_CONTR == 1
                  CALL contract_sssp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_sssd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_sssf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_sssg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (1)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 1 || __MAX_CONTR == 1
                  CALL contract_ssps(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 1 || __MAX_CONTR == 1
                  CALL contract_sspp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_sspd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_sspf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_sspg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (2)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_ssds(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_ssdp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_ssdd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_ssdf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_ssdg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (3)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_ssfs(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_ssfp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_ssfd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_ssff(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_ssfg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (4)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_ssgs(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_ssgp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_ssgd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_ssgf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_ssgg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE DEFAULT
               CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                     work, nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, &
                                     sphi_b, &
                                     sphi_c, &
                                     sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
            END SELECT
         CASE (1)
            SELECT CASE (n_c)
            CASE (0)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 1 || __MAX_CONTR == 1
                  CALL contract_spss(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 1 || __MAX_CONTR == 1
                  CALL contract_spsp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_spsd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_spsf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_spsg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (1)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 1 || __MAX_CONTR == 1
                  CALL contract_spps(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 1 || __MAX_CONTR == 1
                  CALL contract_sppp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_sppd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_sppf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_sppg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (2)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_spds(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_spdp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_spdd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_spdf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_spdg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (3)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_spfs(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_spfp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_spfd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_spff(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_spfg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (4)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_spgs(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_spgp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_spgd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_spgf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_spgg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE DEFAULT
               CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                     work, nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, &
                                     sphi_b, &
                                     sphi_c, &
                                     sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
            END SELECT
         CASE (2)
            SELECT CASE (n_c)
            CASE (0)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_sdss(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_sdsp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_sdsd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_sdsf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_sdsg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (1)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_sdps(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_sdpp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_sdpd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_sdpf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_sdpg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (2)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_sdds(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_sddp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_sddd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_sddf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_sddg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (3)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_sdfs(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_sdfp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_sdfd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_sdff(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_sdfg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (4)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_sdgs(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_sdgp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_sdgd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_sdgf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_sdgg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE DEFAULT
               CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                     work, nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, &
                                     sphi_b, &
                                     sphi_c, &
                                     sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
            END SELECT
         CASE (3)
            SELECT CASE (n_c)
            CASE (0)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_sfss(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_sfsp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_sfsd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_sfsf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_sfsg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (1)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_sfps(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_sfpp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_sfpd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_sfpf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_sfpg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (2)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_sfds(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_sfdp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_sfdd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_sfdf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_sfdg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (3)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_sffs(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_sffp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_sffd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_sfff(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_sffg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (4)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_sfgs(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_sfgp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_sfgd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_sfgf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_sfgg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE DEFAULT
               CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                     work, nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, &
                                     sphi_b, &
                                     sphi_c, &
                                     sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
            END SELECT
         CASE (4)
            SELECT CASE (n_c)
            CASE (0)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_sgss(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_sgsp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_sgsd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_sgsf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_sgsg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (1)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_sgps(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_sgpp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_sgpd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_sgpf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_sgpg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (2)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_sgds(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_sgdp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_sgdd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_sgdf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_sgdg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (3)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_sgfs(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_sgfp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_sgfd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_sgff(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_sgfg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (4)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_sggs(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_sggp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_sggd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_sggf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_sggg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE DEFAULT
               CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                     work, nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, &
                                     sphi_b, &
                                     sphi_c, &
                                     sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
            END SELECT
         CASE DEFAULT
            CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                  work, nl_a, nl_b, nl_c, nl_d, &
                                  sphi_a, &
                                  sphi_b, &
                                  sphi_c, &
                                  sphi_d, &
                                  primitives, &
                                  buffer1, buffer2)
         END SELECT
      CASE (1)
         SELECT CASE (n_b)
         CASE (0)
            SELECT CASE (n_c)
            CASE (0)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 1 || __MAX_CONTR == 1
                  CALL contract_psss(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 1 || __MAX_CONTR == 1
                  CALL contract_pssp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_pssd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_pssf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_pssg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (1)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 1 || __MAX_CONTR == 1
                  CALL contract_psps(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 1 || __MAX_CONTR == 1
                  CALL contract_pspp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_pspd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_pspf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_pspg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (2)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_psds(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_psdp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_psdd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_psdf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_psdg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (3)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_psfs(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_psfp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_psfd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_psff(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_psfg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (4)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_psgs(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_psgp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_psgd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_psgf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_psgg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE DEFAULT
               CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                     work, nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, &
                                     sphi_b, &
                                     sphi_c, &
                                     sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
            END SELECT
         CASE (1)
            SELECT CASE (n_c)
            CASE (0)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 1 || __MAX_CONTR == 1
                  CALL contract_ppss(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 1 || __MAX_CONTR == 1
                  CALL contract_ppsp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_ppsd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_ppsf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_ppsg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (1)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 1 || __MAX_CONTR == 1
                  CALL contract_ppps(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 1 || __MAX_CONTR == 1
                  CALL contract_pppp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_pppd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_pppf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_pppg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (2)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_ppds(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_ppdp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_ppdd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_ppdf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_ppdg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (3)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_ppfs(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_ppfp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_ppfd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_ppff(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_ppfg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (4)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_ppgs(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_ppgp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_ppgd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_ppgf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_ppgg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE DEFAULT
               CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                     work, nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, &
                                     sphi_b, &
                                     sphi_c, &
                                     sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
            END SELECT
         CASE (2)
            SELECT CASE (n_c)
            CASE (0)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_pdss(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_pdsp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_pdsd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_pdsf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_pdsg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (1)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_pdps(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_pdpp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_pdpd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_pdpf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_pdpg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (2)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_pdds(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_pddp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_pddd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_pddf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_pddg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (3)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_pdfs(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_pdfp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_pdfd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_pdff(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_pdfg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (4)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_pdgs(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_pdgp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_pdgd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_pdgf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_pdgg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE DEFAULT
               CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                     work, nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, &
                                     sphi_b, &
                                     sphi_c, &
                                     sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
            END SELECT
         CASE (3)
            SELECT CASE (n_c)
            CASE (0)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_pfss(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_pfsp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_pfsd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_pfsf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_pfsg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (1)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_pfps(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_pfpp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_pfpd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_pfpf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_pfpg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (2)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_pfds(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_pfdp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_pfdd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_pfdf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_pfdg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (3)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_pffs(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_pffp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_pffd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_pfff(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_pffg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (4)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_pfgs(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_pfgp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_pfgd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_pfgf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_pfgg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE DEFAULT
               CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                     work, nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, &
                                     sphi_b, &
                                     sphi_c, &
                                     sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
            END SELECT
         CASE (4)
            SELECT CASE (n_c)
            CASE (0)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_pgss(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_pgsp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_pgsd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_pgsf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_pgsg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (1)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_pgps(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_pgpp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_pgpd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_pgpf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_pgpg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (2)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_pgds(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_pgdp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_pgdd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_pgdf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_pgdg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (3)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_pgfs(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_pgfp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_pgfd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_pgff(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_pgfg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (4)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_pggs(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_pggp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_pggd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_pggf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_pggg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE DEFAULT
               CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                     work, nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, &
                                     sphi_b, &
                                     sphi_c, &
                                     sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
            END SELECT
         CASE DEFAULT
            CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                  work, nl_a, nl_b, nl_c, nl_d, &
                                  sphi_a, &
                                  sphi_b, &
                                  sphi_c, &
                                  sphi_d, &
                                  primitives, &
                                  buffer1, buffer2)
         END SELECT
      CASE (2)
         SELECT CASE (n_b)
         CASE (0)
            SELECT CASE (n_c)
            CASE (0)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_dsss(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_dssp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_dssd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_dssf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_dssg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (1)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_dsps(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_dspp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_dspd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_dspf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_dspg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (2)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_dsds(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_dsdp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_dsdd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_dsdf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_dsdg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (3)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_dsfs(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_dsfp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_dsfd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_dsff(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_dsfg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (4)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_dsgs(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_dsgp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_dsgd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_dsgf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_dsgg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE DEFAULT
               CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                     work, nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, &
                                     sphi_b, &
                                     sphi_c, &
                                     sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
            END SELECT
         CASE (1)
            SELECT CASE (n_c)
            CASE (0)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_dpss(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_dpsp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_dpsd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_dpsf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_dpsg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (1)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_dpps(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_dppp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_dppd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_dppf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_dppg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (2)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_dpds(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_dpdp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_dpdd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_dpdf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_dpdg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (3)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_dpfs(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_dpfp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_dpfd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_dpff(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_dpfg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (4)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_dpgs(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_dpgp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_dpgd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_dpgf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_dpgg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE DEFAULT
               CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                     work, nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, &
                                     sphi_b, &
                                     sphi_c, &
                                     sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
            END SELECT
         CASE (2)
            SELECT CASE (n_c)
            CASE (0)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_ddss(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_ddsp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_ddsd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_ddsf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_ddsg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (1)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_ddps(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_ddpp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_ddpd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_ddpf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_ddpg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (2)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_ddds(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_dddp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
                  CALL contract_dddd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_dddf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_dddg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (3)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_ddfs(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_ddfp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_ddfd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_ddff(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_ddfg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (4)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_ddgs(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_ddgp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_ddgd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_ddgf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_ddgg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE DEFAULT
               CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                     work, nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, &
                                     sphi_b, &
                                     sphi_c, &
                                     sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
            END SELECT
         CASE (3)
            SELECT CASE (n_c)
            CASE (0)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_dfss(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_dfsp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_dfsd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_dfsf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_dfsg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (1)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_dfps(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_dfpp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_dfpd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_dfpf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_dfpg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (2)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_dfds(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_dfdp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_dfdd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_dfdf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_dfdg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (3)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_dffs(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_dffp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_dffd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_dfff(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_dffg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (4)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_dfgs(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_dfgp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_dfgd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_dfgf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_dfgg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE DEFAULT
               CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                     work, nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, &
                                     sphi_b, &
                                     sphi_c, &
                                     sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
            END SELECT
         CASE (4)
            SELECT CASE (n_c)
            CASE (0)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_dgss(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_dgsp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_dgsd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_dgsf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_dgsg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (1)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_dgps(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_dgpp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_dgpd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_dgpf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_dgpg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (2)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_dgds(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_dgdp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_dgdd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_dgdf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_dgdg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (3)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_dgfs(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_dgfp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_dgfd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_dgff(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_dgfg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (4)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_dggs(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_dggp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_dggd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_dggf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_dggg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE DEFAULT
               CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                     work, nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, &
                                     sphi_b, &
                                     sphi_c, &
                                     sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
            END SELECT
         CASE DEFAULT
            CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                  work, nl_a, nl_b, nl_c, nl_d, &
                                  sphi_a, &
                                  sphi_b, &
                                  sphi_c, &
                                  sphi_d, &
                                  primitives, &
                                  buffer1, buffer2)
         END SELECT
      CASE (3)
         SELECT CASE (n_b)
         CASE (0)
            SELECT CASE (n_c)
            CASE (0)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_fsss(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_fssp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_fssd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_fssf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_fssg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (1)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_fsps(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_fspp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_fspd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_fspf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_fspg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (2)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_fsds(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_fsdp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_fsdd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_fsdf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_fsdg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (3)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_fsfs(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_fsfp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_fsfd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_fsff(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_fsfg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (4)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_fsgs(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_fsgp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_fsgd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_fsgf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_fsgg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE DEFAULT
               CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                     work, nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, &
                                     sphi_b, &
                                     sphi_c, &
                                     sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
            END SELECT
         CASE (1)
            SELECT CASE (n_c)
            CASE (0)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_fpss(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_fpsp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_fpsd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_fpsf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_fpsg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (1)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_fpps(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_fppp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_fppd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_fppf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_fppg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (2)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_fpds(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_fpdp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_fpdd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_fpdf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_fpdg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (3)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_fpfs(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_fpfp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_fpfd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_fpff(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_fpfg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (4)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_fpgs(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_fpgp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_fpgd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_fpgf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_fpgg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE DEFAULT
               CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                     work, nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, &
                                     sphi_b, &
                                     sphi_c, &
                                     sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
            END SELECT
         CASE (2)
            SELECT CASE (n_c)
            CASE (0)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_fdss(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_fdsp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_fdsd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_fdsf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_fdsg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (1)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_fdps(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_fdpp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_fdpd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_fdpf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_fdpg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (2)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_fdds(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_fddp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_fddd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_fddf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_fddg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (3)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_fdfs(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_fdfp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_fdfd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_fdff(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_fdfg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (4)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_fdgs(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_fdgp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_fdgd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_fdgf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_fdgg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE DEFAULT
               CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                     work, nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, &
                                     sphi_b, &
                                     sphi_c, &
                                     sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
            END SELECT
         CASE (3)
            SELECT CASE (n_c)
            CASE (0)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_ffss(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_ffsp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_ffsd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_ffsf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_ffsg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (1)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_ffps(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_ffpp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_ffpd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_ffpf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_ffpg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (2)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_ffds(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_ffdp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_ffdd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_ffdf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_ffdg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (3)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_fffs(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_fffp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_fffd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
                  CALL contract_ffff(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_fffg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (4)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_ffgs(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_ffgp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_ffgd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_ffgf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_ffgg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE DEFAULT
               CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                     work, nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, &
                                     sphi_b, &
                                     sphi_c, &
                                     sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
            END SELECT
         CASE (4)
            SELECT CASE (n_c)
            CASE (0)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_fgss(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_fgsp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_fgsd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_fgsf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_fgsg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (1)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_fgps(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_fgpp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_fgpd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_fgpf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_fgpg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (2)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_fgds(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_fgdp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_fgdd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_fgdf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_fgdg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (3)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_fgfs(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_fgfp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_fgfd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_fgff(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_fgfg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (4)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_fggs(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_fggp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_fggd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_fggf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_fggg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE DEFAULT
               CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                     work, nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, &
                                     sphi_b, &
                                     sphi_c, &
                                     sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
            END SELECT
         CASE DEFAULT
            CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                  work, nl_a, nl_b, nl_c, nl_d, &
                                  sphi_a, &
                                  sphi_b, &
                                  sphi_c, &
                                  sphi_d, &
                                  primitives, &
                                  buffer1, buffer2)
         END SELECT
      CASE (4)
         SELECT CASE (n_b)
         CASE (0)
            SELECT CASE (n_c)
            CASE (0)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gsss(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gssp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gssd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gssf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gssg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (1)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gsps(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gspp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gspd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gspf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gspg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (2)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gsds(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gsdp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gsdd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gsdf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gsdg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (3)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gsfs(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gsfp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gsfd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gsff(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gsfg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (4)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gsgs(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gsgp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gsgd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gsgf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gsgg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE DEFAULT
               CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                     work, nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, &
                                     sphi_b, &
                                     sphi_c, &
                                     sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
            END SELECT
         CASE (1)
            SELECT CASE (n_c)
            CASE (0)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gpss(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gpsp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gpsd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gpsf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gpsg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (1)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gpps(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gppp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gppd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gppf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gppg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (2)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gpds(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gpdp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gpdd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gpdf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gpdg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (3)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gpfs(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gpfp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gpfd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gpff(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gpfg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (4)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gpgs(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gpgp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gpgd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gpgf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gpgg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE DEFAULT
               CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                     work, nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, &
                                     sphi_b, &
                                     sphi_c, &
                                     sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
            END SELECT
         CASE (2)
            SELECT CASE (n_c)
            CASE (0)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gdss(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gdsp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gdsd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gdsf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gdsg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (1)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gdps(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gdpp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gdpd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gdpf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gdpg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (2)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gdds(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gddp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gddd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gddf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gddg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (3)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gdfs(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gdfp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gdfd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gdff(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gdfg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (4)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gdgs(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gdgp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gdgd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gdgf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gdgg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE DEFAULT
               CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                     work, nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, &
                                     sphi_b, &
                                     sphi_c, &
                                     sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
            END SELECT
         CASE (3)
            SELECT CASE (n_c)
            CASE (0)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gfss(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gfsp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gfsd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gfsf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gfsg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (1)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gfps(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gfpp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gfpd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gfpf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gfpg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (2)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gfds(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gfdp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gfdd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gfdf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gfdg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (3)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gffs(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gffp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gffd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gfff(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gffg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (4)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gfgs(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gfgp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gfgd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gfgf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gfgg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE DEFAULT
               CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                     work, nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, &
                                     sphi_b, &
                                     sphi_c, &
                                     sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
            END SELECT
         CASE (4)
            SELECT CASE (n_c)
            CASE (0)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_ggss(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_ggsp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_ggsd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_ggsf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_ggsg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (1)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_ggps(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_ggpp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_ggpd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_ggpf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_ggpg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (2)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_ggds(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_ggdp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_ggdd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_ggdf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_ggdg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (3)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_ggfs(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_ggfp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_ggfd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_ggff(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_ggfg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE (4)
               SELECT CASE (n_d)
               CASE (0)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gggs(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (1)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gggp(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (2)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gggd(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (3)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gggf(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE (4)
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
                  CALL contract_gggg(work, &
                                     nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, sphi_b, sphi_c, sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
#else
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
#endif
               CASE DEFAULT
                  CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                        work, nl_a, nl_b, nl_c, nl_d, &
                                        sphi_a, &
                                        sphi_b, &
                                        sphi_c, &
                                        sphi_d, &
                                        primitives, &
                                        buffer1, buffer2)
               END SELECT
            CASE DEFAULT
               CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                     work, nl_a, nl_b, nl_c, nl_d, &
                                     sphi_a, &
                                     sphi_b, &
                                     sphi_c, &
                                     sphi_d, &
                                     primitives, &
                                     buffer1, buffer2)
            END SELECT
         CASE DEFAULT
            CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                                  work, nl_a, nl_b, nl_c, nl_d, &
                                  sphi_a, &
                                  sphi_b, &
                                  sphi_c, &
                                  sphi_d, &
                                  primitives, &
                                  buffer1, buffer2)
         END SELECT
      CASE DEFAULT
         CALL contract_generic(ncoa, ncob, ncoc, ncod, nsoa, nsob, nsoc, nsod, &
                               work, nl_a, nl_b, nl_c, nl_d, &
                               sphi_a, &
                               sphi_b, &
                               sphi_c, &
                               sphi_d, &
                               primitives, &
                               buffer1, buffer2)
      END SELECT
#endif
   END SUBROUTINE contract
#if defined (__LIBINT)

#if __MAX_CONTR > 0 || __MAX_CONTR == 0
! **************************************************************************************************
!> \brief ...
!> \param work ...
!> \param nl_a ...
!> \param nl_b ...
!> \param nl_c ...
!> \param nl_d ...
!> \param sphi_a ...
!> \param sphi_b ...
!> \param sphi_c ...
!> \param sphi_d ...
!> \param primitives ...
!> \param buffer1 ...
!> \param buffer2 ...
! **************************************************************************************************
   SUBROUTINE contract_ssss(work, &
                            nl_a, nl_b, nl_c, nl_d, &
                            sphi_a, sphi_b, sphi_c, sphi_d, &
                            primitives, &
                            buffer1, buffer2)

      REAL(dp), DIMENSION(1*1*1*1), INTENT(IN)           :: work
      INTEGER                                            :: nl_a, nl_b, nl_c, nl_d
      REAL(dp), DIMENSION(1, 1*nl_a), INTENT(IN)         :: sphi_a
      REAL(dp), DIMENSION(1, 1*nl_b), INTENT(IN)         :: sphi_b
      REAL(dp), DIMENSION(1, 1*nl_c), INTENT(IN)         :: sphi_c
      REAL(dp), DIMENSION(1, 1*nl_d), INTENT(IN)         :: sphi_d
      REAL(dp), &
         DIMENSION(1*nl_a, 1*nl_b, 1*nl_c, 1*nl_d)       :: primitives
      REAL(dp), DIMENSION(1*1*1*1)                       :: buffer1, buffer2

      INTEGER                                            :: i, i1, i2, i3, ia, ib, ic, id, imax, &
                                                            kmax, s_offset_a1, s_offset_b1, &
                                                            s_offset_c1, s_offset_d1

      s_offset_a1 = 0
      DO ia = 1, nl_a
         s_offset_b1 = 0
         DO ib = 1, nl_b
            s_offset_c1 = 0
            DO ic = 1, nl_c
               s_offset_d1 = 0
               DO id = 1, nl_d

                  buffer1 = 0.0_dp
                  imax = 1*1*1
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + work(1 + (i - 1)*kmax)*sphi_a(1, 1 + s_offset_a1)
                  END DO

                  buffer2 = 0.0_dp
                  imax = 1*1*1
                  kmax = 1
                  DO i = 1, imax
                     buffer2(i + imax*(1 - 1)) = buffer2(i + imax*(1 - 1)) + buffer1(1 + (i - 1)*kmax)*sphi_b(1, 1 + s_offset_b1)
                  END DO

                  buffer1 = 0.0_dp
                  imax = 1*1*1
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 1 + s_offset_c1)
                  END DO

                  imax = 1*1*1
                  kmax = 1
                  i = 0
                  DO i1 = 1, 1
                  DO i2 = 1, 1
                  DO i3 = 1, 1
                     i = i + 1
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) &
                        + buffer1(1 + (i - 1)*kmax)*sphi_d(1, 1 + s_offset_d1)
                  END DO
                  END DO
                  END DO
                  s_offset_d1 = s_offset_d1 + 1
               END DO
               s_offset_c1 = s_offset_c1 + 1
            END DO
            s_offset_b1 = s_offset_b1 + 1
         END DO
         s_offset_a1 = s_offset_a1 + 1
      END DO
   END SUBROUTINE contract_ssss
#endif
#if __MAX_CONTR > 1 || __MAX_CONTR == 1
! **************************************************************************************************
!> \brief ...
!> \param work ...
!> \param nl_a ...
!> \param nl_b ...
!> \param nl_c ...
!> \param nl_d ...
!> \param sphi_a ...
!> \param sphi_b ...
!> \param sphi_c ...
!> \param sphi_d ...
!> \param primitives ...
!> \param buffer1 ...
!> \param buffer2 ...
! **************************************************************************************************
   SUBROUTINE contract_sssp(work, &
                            nl_a, nl_b, nl_c, nl_d, &
                            sphi_a, sphi_b, sphi_c, sphi_d, &
                            primitives, &
                            buffer1, buffer2)
      REAL(dp), DIMENSION(1*1*1*3), INTENT(IN)           :: work
      INTEGER                                            :: nl_a, nl_b, nl_c, nl_d
      REAL(dp), DIMENSION(1, 1*nl_a), INTENT(IN)         :: sphi_a
      REAL(dp), DIMENSION(1, 1*nl_b), INTENT(IN)         :: sphi_b
      REAL(dp), DIMENSION(1, 1*nl_c), INTENT(IN)         :: sphi_c
      REAL(dp), DIMENSION(3, 3*nl_d), INTENT(IN)         :: sphi_d
      REAL(dp), &
         DIMENSION(1*nl_a, 1*nl_b, 1*nl_c, 3*nl_d)       :: primitives
      REAL(dp), DIMENSION(1*1*1*3)                       :: buffer1, buffer2

      INTEGER                                            :: i, i1, i2, i3, ia, ib, ic, id, imax, &
                                                            kmax, s_offset_a1, s_offset_b1, &
                                                            s_offset_c1, s_offset_d1

      s_offset_a1 = 0
      DO ia = 1, nl_a
         s_offset_b1 = 0
         DO ib = 1, nl_b
            s_offset_c1 = 0
            DO ic = 1, nl_c
               s_offset_d1 = 0
               DO id = 1, nl_d

                  buffer1 = 0.0_dp
                  imax = 1*1*3
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + work(1 + (i - 1)*kmax)*sphi_a(1, 1 + s_offset_a1)
                  END DO

                  buffer2 = 0.0_dp
                  imax = 1*1*3
                  kmax = 1
                  DO i = 1, imax
                     buffer2(i + imax*(1 - 1)) = buffer2(i + imax*(1 - 1)) + buffer1(1 + (i - 1)*kmax)*sphi_b(1, 1 + s_offset_b1)
                  END DO

                  buffer1 = 0.0_dp
                  imax = 1*1*3
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 1 + s_offset_c1)
                  END DO

                  imax = 1*1*1
                  kmax = 3
                  i = 0
                  DO i1 = 1, 1
                  DO i2 = 1, 1
                  DO i3 = 1, 1
                     i = i + 1
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) &
                        + buffer1(1 + (i - 1)*kmax)*sphi_d(1, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) &
                        + buffer1(2 + (i - 1)*kmax)*sphi_d(2, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) &
                        + buffer1(3 + (i - 1)*kmax)*sphi_d(3, 2 + s_offset_d1)
                  END DO
                  END DO
                  END DO
                  s_offset_d1 = s_offset_d1 + 3
               END DO
               s_offset_c1 = s_offset_c1 + 1
            END DO
            s_offset_b1 = s_offset_b1 + 1
         END DO
         s_offset_a1 = s_offset_a1 + 1
      END DO
   END SUBROUTINE contract_sssp
#endif
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
! **************************************************************************************************
!> \brief ...
!> \param work ...
!> \param nl_a ...
!> \param nl_b ...
!> \param nl_c ...
!> \param nl_d ...
!> \param sphi_a ...
!> \param sphi_b ...
!> \param sphi_c ...
!> \param sphi_d ...
!> \param primitives ...
!> \param buffer1 ...
!> \param buffer2 ...
! **************************************************************************************************
   SUBROUTINE contract_sssd(work, &
                            nl_a, nl_b, nl_c, nl_d, &
                            sphi_a, sphi_b, sphi_c, sphi_d, &
                            primitives, &
                            buffer1, buffer2)
      REAL(dp), DIMENSION(1*1*1*6), INTENT(IN)           :: work
      INTEGER                                            :: nl_a, nl_b, nl_c, nl_d
      REAL(dp), DIMENSION(1, 1*nl_a), INTENT(IN)         :: sphi_a
      REAL(dp), DIMENSION(1, 1*nl_b), INTENT(IN)         :: sphi_b
      REAL(dp), DIMENSION(1, 1*nl_c), INTENT(IN)         :: sphi_c
      REAL(dp), DIMENSION(6, 5*nl_d), INTENT(IN)         :: sphi_d
      REAL(dp), &
         DIMENSION(1*nl_a, 1*nl_b, 1*nl_c, 5*nl_d)       :: primitives
      REAL(dp), DIMENSION(1*1*1*6)                       :: buffer1, buffer2

      INTEGER                                            :: i, i1, i2, i3, ia, ib, ic, id, imax, &
                                                            kmax, s_offset_a1, s_offset_b1, &
                                                            s_offset_c1, s_offset_d1

      s_offset_a1 = 0
      DO ia = 1, nl_a
         s_offset_b1 = 0
         DO ib = 1, nl_b
            s_offset_c1 = 0
            DO ic = 1, nl_c
               s_offset_d1 = 0
               DO id = 1, nl_d
                  buffer1 = 0.0_dp
                  imax = 1*1*6
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + work(1 + (i - 1)*kmax)*sphi_a(1, 1 + s_offset_a1)
                  END DO
                  buffer2 = 0.0_dp
                  imax = 1*1*6
                  kmax = 1
                  DO i = 1, imax
                     buffer2(i + imax*(1 - 1)) = buffer2(i + imax*(1 - 1)) + buffer1(1 + (i - 1)*kmax)*sphi_b(1, 1 + s_offset_b1)
                  END DO
                  buffer1 = 0.0_dp
                  imax = 1*1*6
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 1 + s_offset_c1)
                  END DO

                  imax = 1*1*1
                  kmax = 6
                  i = 0
                  DO i1 = 1, 1
                  DO i2 = 1, 1
                  DO i3 = 1, 1
                     i = i + 1
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) + &
                        buffer1(2 + (i - 1)*kmax)*sphi_d(2, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(3 + (i - 1)*kmax)*sphi_d(3, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) + &
                        buffer1(5 + (i - 1)*kmax)*sphi_d(5, 2 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(6 + (i - 1)*kmax)*sphi_d(6, 3 + s_offset_d1)
                  END DO
                  END DO
                  END DO
                  s_offset_d1 = s_offset_d1 + 5
               END DO
               s_offset_c1 = s_offset_c1 + 1
            END DO
            s_offset_b1 = s_offset_b1 + 1
         END DO
         s_offset_a1 = s_offset_a1 + 1
      END DO
   END SUBROUTINE contract_sssd
#endif
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
! **************************************************************************************************
!> \brief ...
!> \param work ...
!> \param nl_a ...
!> \param nl_b ...
!> \param nl_c ...
!> \param nl_d ...
!> \param sphi_a ...
!> \param sphi_b ...
!> \param sphi_c ...
!> \param sphi_d ...
!> \param primitives ...
!> \param buffer1 ...
!> \param buffer2 ...
! **************************************************************************************************
   SUBROUTINE contract_sssf(work, &
                            nl_a, nl_b, nl_c, nl_d, &
                            sphi_a, sphi_b, sphi_c, sphi_d, &
                            primitives, &
                            buffer1, buffer2)
      REAL(dp), DIMENSION(1*1*1*10), INTENT(IN)          :: work
      INTEGER                                            :: nl_a, nl_b, nl_c, nl_d
      REAL(dp), DIMENSION(1, 1*nl_a), INTENT(IN)         :: sphi_a
      REAL(dp), DIMENSION(1, 1*nl_b), INTENT(IN)         :: sphi_b
      REAL(dp), DIMENSION(1, 1*nl_c), INTENT(IN)         :: sphi_c
      REAL(dp), DIMENSION(10, 7*nl_d), INTENT(IN)        :: sphi_d
      REAL(dp), &
         DIMENSION(1*nl_a, 1*nl_b, 1*nl_c, 7*nl_d)       :: primitives
      REAL(dp), DIMENSION(1*1*1*10)                      :: buffer1, buffer2

      INTEGER                                            :: i, i1, i2, i3, ia, ib, ic, id, imax, &
                                                            kmax, s_offset_a1, s_offset_b1, &
                                                            s_offset_c1, s_offset_d1

      s_offset_a1 = 0
      DO ia = 1, nl_a
         s_offset_b1 = 0
         DO ib = 1, nl_b
            s_offset_c1 = 0
            DO ic = 1, nl_c
               s_offset_d1 = 0
               DO id = 1, nl_d
                  buffer1 = 0.0_dp
                  imax = 1*1*10
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + work(1 + (i - 1)*kmax)*sphi_a(1, 1 + s_offset_a1)
                  END DO
                  buffer2 = 0.0_dp
                  imax = 1*1*10
                  kmax = 1
                  DO i = 1, imax
                     buffer2(i + imax*(1 - 1)) = buffer2(i + imax*(1 - 1)) + buffer1(1 + (i - 1)*kmax)*sphi_b(1, 1 + s_offset_b1)
                  END DO
                  buffer1 = 0.0_dp
                  imax = 1*1*10
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 1 + s_offset_c1)
                  END DO
                  imax = 1*1*1
                  kmax = 10
                  i = 0
                  DO i1 = 1, 1
                  DO i2 = 1, 1
                  DO i3 = 1, 1
                     i = i + 1
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) + &
                        buffer1(2 + (i - 1)*kmax)*sphi_d(2, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(2 + (i - 1)*kmax)*sphi_d(2, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(3 + (i - 1)*kmax)*sphi_d(3, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) + &
                        buffer1(3 + (i - 1)*kmax)*sphi_d(3, 6 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) + &
                        buffer1(5 + (i - 1)*kmax)*sphi_d(5, 2 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(6 + (i - 1)*kmax)*sphi_d(6, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) + &
                        buffer1(7 + (i - 1)*kmax)*sphi_d(7, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(7 + (i - 1)*kmax)*sphi_d(7, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(8 + (i - 1)*kmax)*sphi_d(8, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) + &
                        buffer1(8 + (i - 1)*kmax)*sphi_d(8, 6 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(9 + (i - 1)*kmax)*sphi_d(9, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(10 + (i - 1)*kmax)*sphi_d(10, 4 + s_offset_d1)
                  END DO
                  END DO
                  END DO
                  s_offset_d1 = s_offset_d1 + 7
               END DO
               s_offset_c1 = s_offset_c1 + 1
            END DO
            s_offset_b1 = s_offset_b1 + 1
         END DO
         s_offset_a1 = s_offset_a1 + 1
      END DO
   END SUBROUTINE contract_sssf
#endif
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
! **************************************************************************************************
!> \brief ...
!> \param work ...
!> \param nl_a ...
!> \param nl_b ...
!> \param nl_c ...
!> \param nl_d ...
!> \param sphi_a ...
!> \param sphi_b ...
!> \param sphi_c ...
!> \param sphi_d ...
!> \param primitives ...
!> \param buffer1 ...
!> \param buffer2 ...
! **************************************************************************************************
   SUBROUTINE contract_sssg(work, &
                            nl_a, nl_b, nl_c, nl_d, &
                            sphi_a, sphi_b, sphi_c, sphi_d, &
                            primitives, &
                            buffer1, buffer2)
      REAL(dp), DIMENSION(1*1*1*15), INTENT(IN)          :: work
      INTEGER                                            :: nl_a, nl_b, nl_c, nl_d
      REAL(dp), DIMENSION(1, 1*nl_a), INTENT(IN)         :: sphi_a
      REAL(dp), DIMENSION(1, 1*nl_b), INTENT(IN)         :: sphi_b
      REAL(dp), DIMENSION(1, 1*nl_c), INTENT(IN)         :: sphi_c
      REAL(dp), DIMENSION(15, 9*nl_d), INTENT(IN)        :: sphi_d
      REAL(dp), &
         DIMENSION(1*nl_a, 1*nl_b, 1*nl_c, 9*nl_d)       :: primitives
      REAL(dp), DIMENSION(1*1*1*15)                      :: buffer1, buffer2

      INTEGER                                            :: i, i1, i2, i3, ia, ib, ic, id, imax, &
                                                            kmax, s_offset_a1, s_offset_b1, &
                                                            s_offset_c1, s_offset_d1

      s_offset_a1 = 0
      DO ia = 1, nl_a
         s_offset_b1 = 0
         DO ib = 1, nl_b
            s_offset_c1 = 0
            DO ic = 1, nl_c
               s_offset_d1 = 0
               DO id = 1, nl_d
                  buffer1 = 0.0_dp
                  imax = 1*1*15
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + work(1 + (i - 1)*kmax)*sphi_a(1, 1 + s_offset_a1)
                  END DO
                  buffer2 = 0.0_dp
                  imax = 1*1*15
                  kmax = 1
                  DO i = 1, imax
                     buffer2(i + imax*(1 - 1)) = buffer2(i + imax*(1 - 1)) + buffer1(1 + (i - 1)*kmax)*sphi_b(1, 1 + s_offset_b1)
                  END DO
                  buffer1 = 0.0_dp
                  imax = 1*1*15
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 1 + s_offset_c1)
                  END DO
                  imax = 1*1*1
                  kmax = 15
                  i = 0
                  DO i1 = 1, 1
                  DO i2 = 1, 1
                  DO i3 = 1, 1
                     i = i + 1
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 9 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) + &
                        buffer1(2 + (i - 1)*kmax)*sphi_d(2, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(2 + (i - 1)*kmax)*sphi_d(2, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) + &
                        buffer1(3 + (i - 1)*kmax)*sphi_d(3, 6 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 8) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 8) + &
                        buffer1(3 + (i - 1)*kmax)*sphi_d(3, 8 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 9 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) + &
                        buffer1(5 + (i - 1)*kmax)*sphi_d(5, 2 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(5 + (i - 1)*kmax)*sphi_d(5, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(6 + (i - 1)*kmax)*sphi_d(6, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(6 + (i - 1)*kmax)*sphi_d(6, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) + &
                        buffer1(7 + (i - 1)*kmax)*sphi_d(7, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(7 + (i - 1)*kmax)*sphi_d(7, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) + &
                        buffer1(8 + (i - 1)*kmax)*sphi_d(8, 6 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 8) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 8) + &
                        buffer1(8 + (i - 1)*kmax)*sphi_d(8, 8 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(9 + (i - 1)*kmax)*sphi_d(9, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) + &
                        buffer1(10 + (i - 1)*kmax)*sphi_d(10, 6 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(11 + (i - 1)*kmax)*sphi_d(11, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(11 + (i - 1)*kmax)*sphi_d(11, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) + &
                        buffer1(11 + (i - 1)*kmax)*sphi_d(11, 9 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) + &
                        buffer1(12 + (i - 1)*kmax)*sphi_d(12, 2 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(12 + (i - 1)*kmax)*sphi_d(12, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(13 + (i - 1)*kmax)*sphi_d(13, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(13 + (i - 1)*kmax)*sphi_d(13, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(14 + (i - 1)*kmax)*sphi_d(14, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(15 + (i - 1)*kmax)*sphi_d(15, 5 + s_offset_d1)
                  END DO
                  END DO
                  END DO
                  s_offset_d1 = s_offset_d1 + 9
               END DO
               s_offset_c1 = s_offset_c1 + 1
            END DO
            s_offset_b1 = s_offset_b1 + 1
         END DO
         s_offset_a1 = s_offset_a1 + 1
      END DO
   END SUBROUTINE contract_sssg
#endif
#if __MAX_CONTR > 1 || __MAX_CONTR == 1
! **************************************************************************************************
!> \brief ...
!> \param work ...
!> \param nl_a ...
!> \param nl_b ...
!> \param nl_c ...
!> \param nl_d ...
!> \param sphi_a ...
!> \param sphi_b ...
!> \param sphi_c ...
!> \param sphi_d ...
!> \param primitives ...
!> \param buffer1 ...
!> \param buffer2 ...
! **************************************************************************************************
   SUBROUTINE contract_ssps(work, &
                            nl_a, nl_b, nl_c, nl_d, &
                            sphi_a, sphi_b, sphi_c, sphi_d, &
                            primitives, &
                            buffer1, buffer2)
      REAL(dp), DIMENSION(1*1*3*1), INTENT(IN)           :: work
      INTEGER                                            :: nl_a, nl_b, nl_c, nl_d
      REAL(dp), DIMENSION(1, 1*nl_a), INTENT(IN)         :: sphi_a
      REAL(dp), DIMENSION(1, 1*nl_b), INTENT(IN)         :: sphi_b
      REAL(dp), DIMENSION(3, 3*nl_c), INTENT(IN)         :: sphi_c
      REAL(dp), DIMENSION(1, 1*nl_d), INTENT(IN)         :: sphi_d
      REAL(dp), &
         DIMENSION(1*nl_a, 1*nl_b, 3*nl_c, 1*nl_d)       :: primitives
      REAL(dp), DIMENSION(1*1*3*1)                       :: buffer1, buffer2

      INTEGER                                            :: i, i1, i2, i3, ia, ib, ic, id, imax, &
                                                            kmax, s_offset_a1, s_offset_b1, &
                                                            s_offset_c1, s_offset_d1

      s_offset_a1 = 0
      DO ia = 1, nl_a
         s_offset_b1 = 0
         DO ib = 1, nl_b
            s_offset_c1 = 0
            DO ic = 1, nl_c
               s_offset_d1 = 0
               DO id = 1, nl_d
                  buffer1 = 0.0_dp
                  imax = 1*3*1
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + work(1 + (i - 1)*kmax)*sphi_a(1, 1 + s_offset_a1)
                  END DO
                  buffer2 = 0.0_dp
                  imax = 1*3*1
                  kmax = 1
                  DO i = 1, imax
                     buffer2(i + imax*(1 - 1)) = buffer2(i + imax*(1 - 1)) + buffer1(1 + (i - 1)*kmax)*sphi_b(1, 1 + s_offset_b1)
                  END DO
                  buffer1 = 0.0_dp
                  imax = 1*1*1
                  kmax = 3
                  DO i = 1, imax
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 3 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 1 + s_offset_c1)
                     buffer1(i + imax*(2 - 1)) = buffer1(i + imax*(2 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 2 + s_offset_c1)
                  END DO
                  imax = 1*1*3
                  kmax = 1
                  i = 0
                  DO i1 = 1, 3
                  DO i2 = 1, 1
                  DO i3 = 1, 1
                     i = i + 1
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) &
                        + buffer1(1 + (i - 1)*kmax)*sphi_d(1, 1 + s_offset_d1)
                  END DO
                  END DO
                  END DO
                  s_offset_d1 = s_offset_d1 + 1
               END DO
               s_offset_c1 = s_offset_c1 + 3
            END DO
            s_offset_b1 = s_offset_b1 + 1
         END DO
         s_offset_a1 = s_offset_a1 + 1
      END DO
   END SUBROUTINE contract_ssps
#endif
#if __MAX_CONTR > 1 || __MAX_CONTR == 1
! **************************************************************************************************
!> \brief ...
!> \param work ...
!> \param nl_a ...
!> \param nl_b ...
!> \param nl_c ...
!> \param nl_d ...
!> \param sphi_a ...
!> \param sphi_b ...
!> \param sphi_c ...
!> \param sphi_d ...
!> \param primitives ...
!> \param buffer1 ...
!> \param buffer2 ...
! **************************************************************************************************
   SUBROUTINE contract_sspp(work, &
                            nl_a, nl_b, nl_c, nl_d, &
                            sphi_a, sphi_b, sphi_c, sphi_d, &
                            primitives, &
                            buffer1, buffer2)
      REAL(dp), DIMENSION(1*1*3*3), INTENT(IN)           :: work
      INTEGER                                            :: nl_a, nl_b, nl_c, nl_d
      REAL(dp), DIMENSION(1, 1*nl_a), INTENT(IN)         :: sphi_a
      REAL(dp), DIMENSION(1, 1*nl_b), INTENT(IN)         :: sphi_b
      REAL(dp), DIMENSION(3, 3*nl_c), INTENT(IN)         :: sphi_c
      REAL(dp), DIMENSION(3, 3*nl_d), INTENT(IN)         :: sphi_d
      REAL(dp), &
         DIMENSION(1*nl_a, 1*nl_b, 3*nl_c, 3*nl_d)       :: primitives
      REAL(dp), DIMENSION(1*1*3*3)                       :: buffer1, buffer2

      INTEGER                                            :: i, i1, i2, i3, ia, ib, ic, id, imax, &
                                                            kmax, s_offset_a1, s_offset_b1, &
                                                            s_offset_c1, s_offset_d1

      s_offset_a1 = 0
      DO ia = 1, nl_a
         s_offset_b1 = 0
         DO ib = 1, nl_b
            s_offset_c1 = 0
            DO ic = 1, nl_c
               s_offset_d1 = 0
               DO id = 1, nl_d
                  buffer1 = 0.0_dp
                  imax = 1*3*3
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + work(1 + (i - 1)*kmax)*sphi_a(1, 1 + s_offset_a1)
                  END DO
                  buffer2 = 0.0_dp
                  imax = 1*3*3
                  kmax = 1
                  DO i = 1, imax
                     buffer2(i + imax*(1 - 1)) = buffer2(i + imax*(1 - 1)) + buffer1(1 + (i - 1)*kmax)*sphi_b(1, 1 + s_offset_b1)
                  END DO
                  buffer1 = 0.0_dp
                  imax = 1*1*3
                  kmax = 3
                  DO i = 1, imax
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 3 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 1 + s_offset_c1)
                     buffer1(i + imax*(2 - 1)) = buffer1(i + imax*(2 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 2 + s_offset_c1)
                  END DO
                  imax = 1*1*3
                  kmax = 3
                  i = 0
                  DO i1 = 1, 3
                  DO i2 = 1, 1
                  DO i3 = 1, 1
                     i = i + 1
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) &
                        + buffer1(1 + (i - 1)*kmax)*sphi_d(1, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) &
                        + buffer1(2 + (i - 1)*kmax)*sphi_d(2, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) &
                        + buffer1(3 + (i - 1)*kmax)*sphi_d(3, 2 + s_offset_d1)
                  END DO
                  END DO
                  END DO
                  s_offset_d1 = s_offset_d1 + 3
               END DO
               s_offset_c1 = s_offset_c1 + 3
            END DO
            s_offset_b1 = s_offset_b1 + 1
         END DO
         s_offset_a1 = s_offset_a1 + 1
      END DO
   END SUBROUTINE contract_sspp
#endif
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
! **************************************************************************************************
!> \brief ...
!> \param work ...
!> \param nl_a ...
!> \param nl_b ...
!> \param nl_c ...
!> \param nl_d ...
!> \param sphi_a ...
!> \param sphi_b ...
!> \param sphi_c ...
!> \param sphi_d ...
!> \param primitives ...
!> \param buffer1 ...
!> \param buffer2 ...
! **************************************************************************************************
   SUBROUTINE contract_sspd(work, &
                            nl_a, nl_b, nl_c, nl_d, &
                            sphi_a, sphi_b, sphi_c, sphi_d, &
                            primitives, &
                            buffer1, buffer2)
      REAL(dp), DIMENSION(1*1*3*6), INTENT(IN)           :: work
      INTEGER                                            :: nl_a, nl_b, nl_c, nl_d
      REAL(dp), DIMENSION(1, 1*nl_a), INTENT(IN)         :: sphi_a
      REAL(dp), DIMENSION(1, 1*nl_b), INTENT(IN)         :: sphi_b
      REAL(dp), DIMENSION(3, 3*nl_c), INTENT(IN)         :: sphi_c
      REAL(dp), DIMENSION(6, 5*nl_d), INTENT(IN)         :: sphi_d
      REAL(dp), &
         DIMENSION(1*nl_a, 1*nl_b, 3*nl_c, 5*nl_d)       :: primitives
      REAL(dp), DIMENSION(1*1*3*6)                       :: buffer1, buffer2

      INTEGER                                            :: i, i1, i2, i3, ia, ib, ic, id, imax, &
                                                            kmax, s_offset_a1, s_offset_b1, &
                                                            s_offset_c1, s_offset_d1

      s_offset_a1 = 0
      DO ia = 1, nl_a
         s_offset_b1 = 0
         DO ib = 1, nl_b
            s_offset_c1 = 0
            DO ic = 1, nl_c
               s_offset_d1 = 0
               DO id = 1, nl_d
                  buffer1 = 0.0_dp
                  imax = 1*3*6
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + work(1 + (i - 1)*kmax)*sphi_a(1, 1 + s_offset_a1)
                  END DO
                  buffer2 = 0.0_dp
                  imax = 1*3*6
                  kmax = 1
                  DO i = 1, imax
                     buffer2(i + imax*(1 - 1)) = buffer2(i + imax*(1 - 1)) + buffer1(1 + (i - 1)*kmax)*sphi_b(1, 1 + s_offset_b1)
                  END DO
                  buffer1 = 0.0_dp
                  imax = 1*1*6
                  kmax = 3
                  DO i = 1, imax
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 3 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 1 + s_offset_c1)
                     buffer1(i + imax*(2 - 1)) = buffer1(i + imax*(2 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 2 + s_offset_c1)
                  END DO
                  imax = 1*1*3
                  kmax = 6
                  i = 0
                  DO i1 = 1, 3
                  DO i2 = 1, 1
                  DO i3 = 1, 1
                     i = i + 1
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) + &
                        buffer1(2 + (i - 1)*kmax)*sphi_d(2, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(3 + (i - 1)*kmax)*sphi_d(3, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) + &
                        buffer1(5 + (i - 1)*kmax)*sphi_d(5, 2 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(6 + (i - 1)*kmax)*sphi_d(6, 3 + s_offset_d1)
                  END DO
                  END DO
                  END DO
                  s_offset_d1 = s_offset_d1 + 5
               END DO
               s_offset_c1 = s_offset_c1 + 3
            END DO
            s_offset_b1 = s_offset_b1 + 1
         END DO
         s_offset_a1 = s_offset_a1 + 1
      END DO
   END SUBROUTINE contract_sspd
#endif
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
! **************************************************************************************************
!> \brief ...
!> \param work ...
!> \param nl_a ...
!> \param nl_b ...
!> \param nl_c ...
!> \param nl_d ...
!> \param sphi_a ...
!> \param sphi_b ...
!> \param sphi_c ...
!> \param sphi_d ...
!> \param primitives ...
!> \param buffer1 ...
!> \param buffer2 ...
! **************************************************************************************************
   SUBROUTINE contract_sspf(work, &
                            nl_a, nl_b, nl_c, nl_d, &
                            sphi_a, sphi_b, sphi_c, sphi_d, &
                            primitives, &
                            buffer1, buffer2)
      REAL(dp), DIMENSION(1*1*3*10), INTENT(IN)          :: work
      INTEGER                                            :: nl_a, nl_b, nl_c, nl_d
      REAL(dp), DIMENSION(1, 1*nl_a), INTENT(IN)         :: sphi_a
      REAL(dp), DIMENSION(1, 1*nl_b), INTENT(IN)         :: sphi_b
      REAL(dp), DIMENSION(3, 3*nl_c), INTENT(IN)         :: sphi_c
      REAL(dp), DIMENSION(10, 7*nl_d), INTENT(IN)        :: sphi_d
      REAL(dp), &
         DIMENSION(1*nl_a, 1*nl_b, 3*nl_c, 7*nl_d)       :: primitives
      REAL(dp), DIMENSION(1*1*3*10)                      :: buffer1, buffer2

      INTEGER                                            :: i, i1, i2, i3, ia, ib, ic, id, imax, &
                                                            kmax, s_offset_a1, s_offset_b1, &
                                                            s_offset_c1, s_offset_d1

      s_offset_a1 = 0
      DO ia = 1, nl_a
         s_offset_b1 = 0
         DO ib = 1, nl_b
            s_offset_c1 = 0
            DO ic = 1, nl_c
               s_offset_d1 = 0
               DO id = 1, nl_d
                  buffer1 = 0.0_dp
                  imax = 1*3*10
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + work(1 + (i - 1)*kmax)*sphi_a(1, 1 + s_offset_a1)
                  END DO
                  buffer2 = 0.0_dp
                  imax = 1*3*10
                  kmax = 1
                  DO i = 1, imax
                     buffer2(i + imax*(1 - 1)) = buffer2(i + imax*(1 - 1)) + buffer1(1 + (i - 1)*kmax)*sphi_b(1, 1 + s_offset_b1)
                  END DO
                  buffer1 = 0.0_dp
                  imax = 1*1*10
                  kmax = 3
                  DO i = 1, imax
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 3 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 1 + s_offset_c1)
                     buffer1(i + imax*(2 - 1)) = buffer1(i + imax*(2 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 2 + s_offset_c1)
                  END DO
                  imax = 1*1*3
                  kmax = 10
                  i = 0
                  DO i1 = 1, 3
                  DO i2 = 1, 1
                  DO i3 = 1, 1
                     i = i + 1
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) + &
                        buffer1(2 + (i - 1)*kmax)*sphi_d(2, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(2 + (i - 1)*kmax)*sphi_d(2, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(3 + (i - 1)*kmax)*sphi_d(3, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) + &
                        buffer1(3 + (i - 1)*kmax)*sphi_d(3, 6 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) + &
                        buffer1(5 + (i - 1)*kmax)*sphi_d(5, 2 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(6 + (i - 1)*kmax)*sphi_d(6, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) + &
                        buffer1(7 + (i - 1)*kmax)*sphi_d(7, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(7 + (i - 1)*kmax)*sphi_d(7, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(8 + (i - 1)*kmax)*sphi_d(8, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) + &
                        buffer1(8 + (i - 1)*kmax)*sphi_d(8, 6 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(9 + (i - 1)*kmax)*sphi_d(9, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(10 + (i - 1)*kmax)*sphi_d(10, 4 + s_offset_d1)
                  END DO
                  END DO
                  END DO
                  s_offset_d1 = s_offset_d1 + 7
               END DO
               s_offset_c1 = s_offset_c1 + 3
            END DO
            s_offset_b1 = s_offset_b1 + 1
         END DO
         s_offset_a1 = s_offset_a1 + 1
      END DO
   END SUBROUTINE contract_sspf
#endif
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
! **************************************************************************************************
!> \brief ...
!> \param work ...
!> \param nl_a ...
!> \param nl_b ...
!> \param nl_c ...
!> \param nl_d ...
!> \param sphi_a ...
!> \param sphi_b ...
!> \param sphi_c ...
!> \param sphi_d ...
!> \param primitives ...
!> \param buffer1 ...
!> \param buffer2 ...
! **************************************************************************************************
   SUBROUTINE contract_sspg(work, &
                            nl_a, nl_b, nl_c, nl_d, &
                            sphi_a, sphi_b, sphi_c, sphi_d, &
                            primitives, &
                            buffer1, buffer2)
      REAL(dp), DIMENSION(1*1*3*15), INTENT(IN)          :: work
      INTEGER                                            :: nl_a, nl_b, nl_c, nl_d
      REAL(dp), DIMENSION(1, 1*nl_a), INTENT(IN)         :: sphi_a
      REAL(dp), DIMENSION(1, 1*nl_b), INTENT(IN)         :: sphi_b
      REAL(dp), DIMENSION(3, 3*nl_c), INTENT(IN)         :: sphi_c
      REAL(dp), DIMENSION(15, 9*nl_d), INTENT(IN)        :: sphi_d
      REAL(dp), &
         DIMENSION(1*nl_a, 1*nl_b, 3*nl_c, 9*nl_d)       :: primitives
      REAL(dp), DIMENSION(1*1*3*15)                      :: buffer1, buffer2

      INTEGER                                            :: i, i1, i2, i3, ia, ib, ic, id, imax, &
                                                            kmax, s_offset_a1, s_offset_b1, &
                                                            s_offset_c1, s_offset_d1

      s_offset_a1 = 0
      DO ia = 1, nl_a
         s_offset_b1 = 0
         DO ib = 1, nl_b
            s_offset_c1 = 0
            DO ic = 1, nl_c
               s_offset_d1 = 0
               DO id = 1, nl_d
                  buffer1 = 0.0_dp
                  imax = 1*3*15
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + work(1 + (i - 1)*kmax)*sphi_a(1, 1 + s_offset_a1)
                  END DO
                  buffer2 = 0.0_dp
                  imax = 1*3*15
                  kmax = 1
                  DO i = 1, imax
                     buffer2(i + imax*(1 - 1)) = buffer2(i + imax*(1 - 1)) + buffer1(1 + (i - 1)*kmax)*sphi_b(1, 1 + s_offset_b1)
                  END DO
                  buffer1 = 0.0_dp
                  imax = 1*1*15
                  kmax = 3
                  DO i = 1, imax
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 3 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 1 + s_offset_c1)
                     buffer1(i + imax*(2 - 1)) = buffer1(i + imax*(2 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 2 + s_offset_c1)
                  END DO
                  imax = 1*1*3
                  kmax = 15
                  i = 0
                  DO i1 = 1, 3
                  DO i2 = 1, 1
                  DO i3 = 1, 1
                     i = i + 1
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 9 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) + &
                        buffer1(2 + (i - 1)*kmax)*sphi_d(2, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(2 + (i - 1)*kmax)*sphi_d(2, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) + &
                        buffer1(3 + (i - 1)*kmax)*sphi_d(3, 6 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 8) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 8) + &
                        buffer1(3 + (i - 1)*kmax)*sphi_d(3, 8 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 9 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) + &
                        buffer1(5 + (i - 1)*kmax)*sphi_d(5, 2 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(5 + (i - 1)*kmax)*sphi_d(5, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(6 + (i - 1)*kmax)*sphi_d(6, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(6 + (i - 1)*kmax)*sphi_d(6, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) + &
                        buffer1(7 + (i - 1)*kmax)*sphi_d(7, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(7 + (i - 1)*kmax)*sphi_d(7, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) + &
                        buffer1(8 + (i - 1)*kmax)*sphi_d(8, 6 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 8) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 8) + &
                        buffer1(8 + (i - 1)*kmax)*sphi_d(8, 8 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(9 + (i - 1)*kmax)*sphi_d(9, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) + &
                        buffer1(10 + (i - 1)*kmax)*sphi_d(10, 6 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(11 + (i - 1)*kmax)*sphi_d(11, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(11 + (i - 1)*kmax)*sphi_d(11, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) + &
                        buffer1(11 + (i - 1)*kmax)*sphi_d(11, 9 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) + &
                        buffer1(12 + (i - 1)*kmax)*sphi_d(12, 2 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(12 + (i - 1)*kmax)*sphi_d(12, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(13 + (i - 1)*kmax)*sphi_d(13, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(13 + (i - 1)*kmax)*sphi_d(13, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(14 + (i - 1)*kmax)*sphi_d(14, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(15 + (i - 1)*kmax)*sphi_d(15, 5 + s_offset_d1)
                  END DO
                  END DO
                  END DO
                  s_offset_d1 = s_offset_d1 + 9
               END DO
               s_offset_c1 = s_offset_c1 + 3
            END DO
            s_offset_b1 = s_offset_b1 + 1
         END DO
         s_offset_a1 = s_offset_a1 + 1
      END DO
   END SUBROUTINE contract_sspg
#endif
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
! **************************************************************************************************
!> \brief ...
!> \param work ...
!> \param nl_a ...
!> \param nl_b ...
!> \param nl_c ...
!> \param nl_d ...
!> \param sphi_a ...
!> \param sphi_b ...
!> \param sphi_c ...
!> \param sphi_d ...
!> \param primitives ...
!> \param buffer1 ...
!> \param buffer2 ...
! **************************************************************************************************
   SUBROUTINE contract_ssds(work, &
                            nl_a, nl_b, nl_c, nl_d, &
                            sphi_a, sphi_b, sphi_c, sphi_d, &
                            primitives, &
                            buffer1, buffer2)
      REAL(dp), DIMENSION(1*1*6*1), INTENT(IN)           :: work
      INTEGER                                            :: nl_a, nl_b, nl_c, nl_d
      REAL(dp), DIMENSION(1, 1*nl_a), INTENT(IN)         :: sphi_a
      REAL(dp), DIMENSION(1, 1*nl_b), INTENT(IN)         :: sphi_b
      REAL(dp), DIMENSION(6, 5*nl_c), INTENT(IN)         :: sphi_c
      REAL(dp), DIMENSION(1, 1*nl_d), INTENT(IN)         :: sphi_d
      REAL(dp), &
         DIMENSION(1*nl_a, 1*nl_b, 5*nl_c, 1*nl_d)       :: primitives
      REAL(dp), DIMENSION(1*1*6*1)                       :: buffer1, buffer2

      INTEGER                                            :: i, i1, i2, i3, ia, ib, ic, id, imax, &
                                                            kmax, s_offset_a1, s_offset_b1, &
                                                            s_offset_c1, s_offset_d1

      s_offset_a1 = 0
      DO ia = 1, nl_a
         s_offset_b1 = 0
         DO ib = 1, nl_b
            s_offset_c1 = 0
            DO ic = 1, nl_c
               s_offset_d1 = 0
               DO id = 1, nl_d
                  buffer1 = 0.0_dp
                  imax = 1*6*1
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + work(1 + (i - 1)*kmax)*sphi_a(1, 1 + s_offset_a1)
                  END DO
                  buffer2 = 0.0_dp
                  imax = 1*6*1
                  kmax = 1
                  DO i = 1, imax
                     buffer2(i + imax*(1 - 1)) = buffer2(i + imax*(1 - 1)) + buffer1(1 + (i - 1)*kmax)*sphi_b(1, 1 + s_offset_b1)
                  END DO
                  buffer1 = 0.0_dp
                  imax = 1*1*1
                  kmax = 6
                  DO i = 1, imax
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 3 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 5 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 1 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 4 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 3 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 5 + s_offset_c1)
                     buffer1(i + imax*(2 - 1)) = buffer1(i + imax*(2 - 1)) + buffer2(5 + (i - 1)*kmax)*sphi_c(5, 2 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(6 + (i - 1)*kmax)*sphi_c(6, 3 + s_offset_c1)
                  END DO
                  imax = 1*1*5
                  kmax = 1
                  i = 0
                  DO i1 = 1, 5
                  DO i2 = 1, 1
                  DO i3 = 1, 1
                     i = i + 1
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) &
                        + buffer1(1 + (i - 1)*kmax)*sphi_d(1, 1 + s_offset_d1)
                  END DO
                  END DO
                  END DO
                  s_offset_d1 = s_offset_d1 + 1
               END DO
               s_offset_c1 = s_offset_c1 + 5
            END DO
            s_offset_b1 = s_offset_b1 + 1
         END DO
         s_offset_a1 = s_offset_a1 + 1
      END DO
   END SUBROUTINE contract_ssds
#endif
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
! **************************************************************************************************
!> \brief ...
!> \param work ...
!> \param nl_a ...
!> \param nl_b ...
!> \param nl_c ...
!> \param nl_d ...
!> \param sphi_a ...
!> \param sphi_b ...
!> \param sphi_c ...
!> \param sphi_d ...
!> \param primitives ...
!> \param buffer1 ...
!> \param buffer2 ...
! **************************************************************************************************
   SUBROUTINE contract_ssdp(work, &
                            nl_a, nl_b, nl_c, nl_d, &
                            sphi_a, sphi_b, sphi_c, sphi_d, &
                            primitives, &
                            buffer1, buffer2)
      REAL(dp), DIMENSION(1*1*6*3), INTENT(IN)           :: work
      INTEGER                                            :: nl_a, nl_b, nl_c, nl_d
      REAL(dp), DIMENSION(1, 1*nl_a), INTENT(IN)         :: sphi_a
      REAL(dp), DIMENSION(1, 1*nl_b), INTENT(IN)         :: sphi_b
      REAL(dp), DIMENSION(6, 5*nl_c), INTENT(IN)         :: sphi_c
      REAL(dp), DIMENSION(3, 3*nl_d), INTENT(IN)         :: sphi_d
      REAL(dp), &
         DIMENSION(1*nl_a, 1*nl_b, 5*nl_c, 3*nl_d)       :: primitives
      REAL(dp), DIMENSION(1*1*6*3)                       :: buffer1, buffer2

      INTEGER                                            :: i, i1, i2, i3, ia, ib, ic, id, imax, &
                                                            kmax, s_offset_a1, s_offset_b1, &
                                                            s_offset_c1, s_offset_d1

      s_offset_a1 = 0
      DO ia = 1, nl_a
         s_offset_b1 = 0
         DO ib = 1, nl_b
            s_offset_c1 = 0
            DO ic = 1, nl_c
               s_offset_d1 = 0
               DO id = 1, nl_d
                  buffer1 = 0.0_dp
                  imax = 1*6*3
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + work(1 + (i - 1)*kmax)*sphi_a(1, 1 + s_offset_a1)
                  END DO
                  buffer2 = 0.0_dp
                  imax = 1*6*3
                  kmax = 1
                  DO i = 1, imax
                     buffer2(i + imax*(1 - 1)) = buffer2(i + imax*(1 - 1)) + buffer1(1 + (i - 1)*kmax)*sphi_b(1, 1 + s_offset_b1)
                  END DO
                  buffer1 = 0.0_dp
                  imax = 1*1*3
                  kmax = 6
                  DO i = 1, imax
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 3 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 5 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 1 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 4 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 3 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 5 + s_offset_c1)
                     buffer1(i + imax*(2 - 1)) = buffer1(i + imax*(2 - 1)) + buffer2(5 + (i - 1)*kmax)*sphi_c(5, 2 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(6 + (i - 1)*kmax)*sphi_c(6, 3 + s_offset_c1)
                  END DO
                  imax = 1*1*5
                  kmax = 3
                  i = 0
                  DO i1 = 1, 5
                  DO i2 = 1, 1
                  DO i3 = 1, 1
                     i = i + 1
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) &
                        + buffer1(1 + (i - 1)*kmax)*sphi_d(1, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) &
                        + buffer1(2 + (i - 1)*kmax)*sphi_d(2, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) &
                        + buffer1(3 + (i - 1)*kmax)*sphi_d(3, 2 + s_offset_d1)
                  END DO
                  END DO
                  END DO
                  s_offset_d1 = s_offset_d1 + 3
               END DO
               s_offset_c1 = s_offset_c1 + 5
            END DO
            s_offset_b1 = s_offset_b1 + 1
         END DO
         s_offset_a1 = s_offset_a1 + 1
      END DO
   END SUBROUTINE contract_ssdp
#endif
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
! **************************************************************************************************
!> \brief ...
!> \param work ...
!> \param nl_a ...
!> \param nl_b ...
!> \param nl_c ...
!> \param nl_d ...
!> \param sphi_a ...
!> \param sphi_b ...
!> \param sphi_c ...
!> \param sphi_d ...
!> \param primitives ...
!> \param buffer1 ...
!> \param buffer2 ...
! **************************************************************************************************
   SUBROUTINE contract_ssdd(work, &
                            nl_a, nl_b, nl_c, nl_d, &
                            sphi_a, sphi_b, sphi_c, sphi_d, &
                            primitives, &
                            buffer1, buffer2)
      REAL(dp), DIMENSION(1*1*6*6), INTENT(IN)           :: work
      INTEGER                                            :: nl_a, nl_b, nl_c, nl_d
      REAL(dp), DIMENSION(1, 1*nl_a), INTENT(IN)         :: sphi_a
      REAL(dp), DIMENSION(1, 1*nl_b), INTENT(IN)         :: sphi_b
      REAL(dp), DIMENSION(6, 5*nl_c), INTENT(IN)         :: sphi_c
      REAL(dp), DIMENSION(6, 5*nl_d), INTENT(IN)         :: sphi_d
      REAL(dp), &
         DIMENSION(1*nl_a, 1*nl_b, 5*nl_c, 5*nl_d)       :: primitives
      REAL(dp), DIMENSION(1*1*6*6)                       :: buffer1, buffer2

      INTEGER                                            :: i, i1, i2, i3, ia, ib, ic, id, imax, &
                                                            kmax, s_offset_a1, s_offset_b1, &
                                                            s_offset_c1, s_offset_d1

      s_offset_a1 = 0
      DO ia = 1, nl_a
         s_offset_b1 = 0
         DO ib = 1, nl_b
            s_offset_c1 = 0
            DO ic = 1, nl_c
               s_offset_d1 = 0
               DO id = 1, nl_d
                  buffer1 = 0.0_dp
                  imax = 1*6*6
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + work(1 + (i - 1)*kmax)*sphi_a(1, 1 + s_offset_a1)
                  END DO
                  buffer2 = 0.0_dp
                  imax = 1*6*6
                  kmax = 1
                  DO i = 1, imax
                     buffer2(i + imax*(1 - 1)) = buffer2(i + imax*(1 - 1)) + buffer1(1 + (i - 1)*kmax)*sphi_b(1, 1 + s_offset_b1)
                  END DO
                  buffer1 = 0.0_dp
                  imax = 1*1*6
                  kmax = 6
                  DO i = 1, imax
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 3 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 5 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 1 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 4 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 3 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 5 + s_offset_c1)
                     buffer1(i + imax*(2 - 1)) = buffer1(i + imax*(2 - 1)) + buffer2(5 + (i - 1)*kmax)*sphi_c(5, 2 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(6 + (i - 1)*kmax)*sphi_c(6, 3 + s_offset_c1)
                  END DO
                  imax = 1*1*5
                  kmax = 6
                  i = 0
                  DO i1 = 1, 5
                  DO i2 = 1, 1
                  DO i3 = 1, 1
                     i = i + 1
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) + &
                        buffer1(2 + (i - 1)*kmax)*sphi_d(2, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(3 + (i - 1)*kmax)*sphi_d(3, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) + &
                        buffer1(5 + (i - 1)*kmax)*sphi_d(5, 2 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(6 + (i - 1)*kmax)*sphi_d(6, 3 + s_offset_d1)
                  END DO
                  END DO
                  END DO
                  s_offset_d1 = s_offset_d1 + 5
               END DO
               s_offset_c1 = s_offset_c1 + 5
            END DO
            s_offset_b1 = s_offset_b1 + 1
         END DO
         s_offset_a1 = s_offset_a1 + 1
      END DO
   END SUBROUTINE contract_ssdd
#endif
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
! **************************************************************************************************
!> \brief ...
!> \param work ...
!> \param nl_a ...
!> \param nl_b ...
!> \param nl_c ...
!> \param nl_d ...
!> \param sphi_a ...
!> \param sphi_b ...
!> \param sphi_c ...
!> \param sphi_d ...
!> \param primitives ...
!> \param buffer1 ...
!> \param buffer2 ...
! **************************************************************************************************
   SUBROUTINE contract_ssdf(work, &
                            nl_a, nl_b, nl_c, nl_d, &
                            sphi_a, sphi_b, sphi_c, sphi_d, &
                            primitives, &
                            buffer1, buffer2)
      REAL(dp), DIMENSION(1*1*6*10), INTENT(IN)          :: work
      INTEGER                                            :: nl_a, nl_b, nl_c, nl_d
      REAL(dp), DIMENSION(1, 1*nl_a), INTENT(IN)         :: sphi_a
      REAL(dp), DIMENSION(1, 1*nl_b), INTENT(IN)         :: sphi_b
      REAL(dp), DIMENSION(6, 5*nl_c), INTENT(IN)         :: sphi_c
      REAL(dp), DIMENSION(10, 7*nl_d), INTENT(IN)        :: sphi_d
      REAL(dp), &
         DIMENSION(1*nl_a, 1*nl_b, 5*nl_c, 7*nl_d)       :: primitives
      REAL(dp), DIMENSION(1*1*6*10)                      :: buffer1, buffer2

      INTEGER                                            :: i, i1, i2, i3, ia, ib, ic, id, imax, &
                                                            kmax, s_offset_a1, s_offset_b1, &
                                                            s_offset_c1, s_offset_d1

      s_offset_a1 = 0
      DO ia = 1, nl_a
         s_offset_b1 = 0
         DO ib = 1, nl_b
            s_offset_c1 = 0
            DO ic = 1, nl_c
               s_offset_d1 = 0
               DO id = 1, nl_d
                  buffer1 = 0.0_dp
                  imax = 1*6*10
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + work(1 + (i - 1)*kmax)*sphi_a(1, 1 + s_offset_a1)
                  END DO
                  buffer2 = 0.0_dp
                  imax = 1*6*10
                  kmax = 1
                  DO i = 1, imax
                     buffer2(i + imax*(1 - 1)) = buffer2(i + imax*(1 - 1)) + buffer1(1 + (i - 1)*kmax)*sphi_b(1, 1 + s_offset_b1)
                  END DO
                  buffer1 = 0.0_dp
                  imax = 1*1*10
                  kmax = 6
                  DO i = 1, imax
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 3 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 5 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 1 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 4 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 3 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 5 + s_offset_c1)
                     buffer1(i + imax*(2 - 1)) = buffer1(i + imax*(2 - 1)) + buffer2(5 + (i - 1)*kmax)*sphi_c(5, 2 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(6 + (i - 1)*kmax)*sphi_c(6, 3 + s_offset_c1)
                  END DO
                  imax = 1*1*5
                  kmax = 10
                  i = 0
                  DO i1 = 1, 5
                  DO i2 = 1, 1
                  DO i3 = 1, 1
                     i = i + 1
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) + &
                        buffer1(2 + (i - 1)*kmax)*sphi_d(2, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(2 + (i - 1)*kmax)*sphi_d(2, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(3 + (i - 1)*kmax)*sphi_d(3, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) + &
                        buffer1(3 + (i - 1)*kmax)*sphi_d(3, 6 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) + &
                        buffer1(5 + (i - 1)*kmax)*sphi_d(5, 2 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(6 + (i - 1)*kmax)*sphi_d(6, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) + &
                        buffer1(7 + (i - 1)*kmax)*sphi_d(7, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(7 + (i - 1)*kmax)*sphi_d(7, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(8 + (i - 1)*kmax)*sphi_d(8, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) + &
                        buffer1(8 + (i - 1)*kmax)*sphi_d(8, 6 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(9 + (i - 1)*kmax)*sphi_d(9, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(10 + (i - 1)*kmax)*sphi_d(10, 4 + s_offset_d1)
                  END DO
                  END DO
                  END DO
                  s_offset_d1 = s_offset_d1 + 7
               END DO
               s_offset_c1 = s_offset_c1 + 5
            END DO
            s_offset_b1 = s_offset_b1 + 1
         END DO
         s_offset_a1 = s_offset_a1 + 1
      END DO
   END SUBROUTINE contract_ssdf
#endif
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
! **************************************************************************************************
!> \brief ...
!> \param work ...
!> \param nl_a ...
!> \param nl_b ...
!> \param nl_c ...
!> \param nl_d ...
!> \param sphi_a ...
!> \param sphi_b ...
!> \param sphi_c ...
!> \param sphi_d ...
!> \param primitives ...
!> \param buffer1 ...
!> \param buffer2 ...
! **************************************************************************************************
   SUBROUTINE contract_ssdg(work, &
                            nl_a, nl_b, nl_c, nl_d, &
                            sphi_a, sphi_b, sphi_c, sphi_d, &
                            primitives, &
                            buffer1, buffer2)
      REAL(dp), DIMENSION(1*1*6*15), INTENT(IN)          :: work
      INTEGER                                            :: nl_a, nl_b, nl_c, nl_d
      REAL(dp), DIMENSION(1, 1*nl_a), INTENT(IN)         :: sphi_a
      REAL(dp), DIMENSION(1, 1*nl_b), INTENT(IN)         :: sphi_b
      REAL(dp), DIMENSION(6, 5*nl_c), INTENT(IN)         :: sphi_c
      REAL(dp), DIMENSION(15, 9*nl_d), INTENT(IN)        :: sphi_d
      REAL(dp), &
         DIMENSION(1*nl_a, 1*nl_b, 5*nl_c, 9*nl_d)       :: primitives
      REAL(dp), DIMENSION(1*1*6*15)                      :: buffer1, buffer2

      INTEGER                                            :: i, i1, i2, i3, ia, ib, ic, id, imax, &
                                                            kmax, s_offset_a1, s_offset_b1, &
                                                            s_offset_c1, s_offset_d1

      s_offset_a1 = 0
      DO ia = 1, nl_a
         s_offset_b1 = 0
         DO ib = 1, nl_b
            s_offset_c1 = 0
            DO ic = 1, nl_c
               s_offset_d1 = 0
               DO id = 1, nl_d
                  buffer1 = 0.0_dp
                  imax = 1*6*15
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + work(1 + (i - 1)*kmax)*sphi_a(1, 1 + s_offset_a1)
                  END DO
                  buffer2 = 0.0_dp
                  imax = 1*6*15
                  kmax = 1
                  DO i = 1, imax
                     buffer2(i + imax*(1 - 1)) = buffer2(i + imax*(1 - 1)) + buffer1(1 + (i - 1)*kmax)*sphi_b(1, 1 + s_offset_b1)
                  END DO
                  buffer1 = 0.0_dp
                  imax = 1*1*15
                  kmax = 6
                  DO i = 1, imax
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 3 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 5 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 1 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 4 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 3 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 5 + s_offset_c1)
                     buffer1(i + imax*(2 - 1)) = buffer1(i + imax*(2 - 1)) + buffer2(5 + (i - 1)*kmax)*sphi_c(5, 2 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(6 + (i - 1)*kmax)*sphi_c(6, 3 + s_offset_c1)
                  END DO
                  imax = 1*1*5
                  kmax = 15
                  i = 0
                  DO i1 = 1, 5
                  DO i2 = 1, 1
                  DO i3 = 1, 1
                     i = i + 1
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 9 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) + &
                        buffer1(2 + (i - 1)*kmax)*sphi_d(2, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(2 + (i - 1)*kmax)*sphi_d(2, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) + &
                        buffer1(3 + (i - 1)*kmax)*sphi_d(3, 6 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 8) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 8) + &
                        buffer1(3 + (i - 1)*kmax)*sphi_d(3, 8 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 9 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) + &
                        buffer1(5 + (i - 1)*kmax)*sphi_d(5, 2 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(5 + (i - 1)*kmax)*sphi_d(5, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(6 + (i - 1)*kmax)*sphi_d(6, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(6 + (i - 1)*kmax)*sphi_d(6, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) + &
                        buffer1(7 + (i - 1)*kmax)*sphi_d(7, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(7 + (i - 1)*kmax)*sphi_d(7, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) + &
                        buffer1(8 + (i - 1)*kmax)*sphi_d(8, 6 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 8) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 8) + &
                        buffer1(8 + (i - 1)*kmax)*sphi_d(8, 8 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(9 + (i - 1)*kmax)*sphi_d(9, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) + &
                        buffer1(10 + (i - 1)*kmax)*sphi_d(10, 6 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(11 + (i - 1)*kmax)*sphi_d(11, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(11 + (i - 1)*kmax)*sphi_d(11, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) + &
                        buffer1(11 + (i - 1)*kmax)*sphi_d(11, 9 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) + &
                        buffer1(12 + (i - 1)*kmax)*sphi_d(12, 2 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(12 + (i - 1)*kmax)*sphi_d(12, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(13 + (i - 1)*kmax)*sphi_d(13, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(13 + (i - 1)*kmax)*sphi_d(13, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(14 + (i - 1)*kmax)*sphi_d(14, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(15 + (i - 1)*kmax)*sphi_d(15, 5 + s_offset_d1)
                  END DO
                  END DO
                  END DO
                  s_offset_d1 = s_offset_d1 + 9
               END DO
               s_offset_c1 = s_offset_c1 + 5
            END DO
            s_offset_b1 = s_offset_b1 + 1
         END DO
         s_offset_a1 = s_offset_a1 + 1
      END DO
   END SUBROUTINE contract_ssdg
#endif
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
! **************************************************************************************************
!> \brief ...
!> \param work ...
!> \param nl_a ...
!> \param nl_b ...
!> \param nl_c ...
!> \param nl_d ...
!> \param sphi_a ...
!> \param sphi_b ...
!> \param sphi_c ...
!> \param sphi_d ...
!> \param primitives ...
!> \param buffer1 ...
!> \param buffer2 ...
! **************************************************************************************************
   SUBROUTINE contract_ssfs(work, &
                            nl_a, nl_b, nl_c, nl_d, &
                            sphi_a, sphi_b, sphi_c, sphi_d, &
                            primitives, &
                            buffer1, buffer2)
      REAL(dp), DIMENSION(1*1*10*1), INTENT(IN)          :: work
      INTEGER                                            :: nl_a, nl_b, nl_c, nl_d
      REAL(dp), DIMENSION(1, 1*nl_a), INTENT(IN)         :: sphi_a
      REAL(dp), DIMENSION(1, 1*nl_b), INTENT(IN)         :: sphi_b
      REAL(dp), DIMENSION(10, 7*nl_c), INTENT(IN)        :: sphi_c
      REAL(dp), DIMENSION(1, 1*nl_d), INTENT(IN)         :: sphi_d
      REAL(dp), &
         DIMENSION(1*nl_a, 1*nl_b, 7*nl_c, 1*nl_d)       :: primitives
      REAL(dp), DIMENSION(1*1*10*1)                      :: buffer1, buffer2

      INTEGER                                            :: i, i1, i2, i3, ia, ib, ic, id, imax, &
                                                            kmax, s_offset_a1, s_offset_b1, &
                                                            s_offset_c1, s_offset_d1

      s_offset_a1 = 0
      DO ia = 1, nl_a
         s_offset_b1 = 0
         DO ib = 1, nl_b
            s_offset_c1 = 0
            DO ic = 1, nl_c
               s_offset_d1 = 0
               DO id = 1, nl_d
                  buffer1 = 0.0_dp
                  imax = 1*10*1
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + work(1 + (i - 1)*kmax)*sphi_a(1, 1 + s_offset_a1)
                  END DO
                  buffer2 = 0.0_dp
                  imax = 1*10*1
                  kmax = 1
                  DO i = 1, imax
                     buffer2(i + imax*(1 - 1)) = buffer2(i + imax*(1 - 1)) + buffer1(1 + (i - 1)*kmax)*sphi_b(1, 1 + s_offset_b1)
                  END DO
                  buffer1 = 0.0_dp
                  imax = 1*1*1
                  kmax = 10
                  DO i = 1, imax
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 7 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 1 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 3 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 4 + s_offset_c1)
                     buffer1(i + imax*(6 - 1)) = buffer1(i + imax*(6 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 6 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 7 + s_offset_c1)
                     buffer1(i + imax*(2 - 1)) = buffer1(i + imax*(2 - 1)) + buffer2(5 + (i - 1)*kmax)*sphi_c(5, 2 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(6 + (i - 1)*kmax)*sphi_c(6, 5 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(7 + (i - 1)*kmax)*sphi_c(7, 1 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(7 + (i - 1)*kmax)*sphi_c(7, 3 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(8 + (i - 1)*kmax)*sphi_c(8, 4 + s_offset_c1)
                     buffer1(i + imax*(6 - 1)) = buffer1(i + imax*(6 - 1)) + buffer2(8 + (i - 1)*kmax)*sphi_c(8, 6 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(9 + (i - 1)*kmax)*sphi_c(9, 3 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(10 + (i - 1)*kmax)*sphi_c(10, 4 + s_offset_c1)
                  END DO
                  imax = 1*1*7
                  kmax = 1
                  i = 0
                  DO i1 = 1, 7
                  DO i2 = 1, 1
                  DO i3 = 1, 1
                     i = i + 1
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) &
                        + buffer1(1 + (i - 1)*kmax)*sphi_d(1, 1 + s_offset_d1)
                  END DO
                  END DO
                  END DO
                  s_offset_d1 = s_offset_d1 + 1
               END DO
               s_offset_c1 = s_offset_c1 + 7
            END DO
            s_offset_b1 = s_offset_b1 + 1
         END DO
         s_offset_a1 = s_offset_a1 + 1
      END DO
   END SUBROUTINE contract_ssfs
#endif
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
! **************************************************************************************************
!> \brief ...
!> \param work ...
!> \param nl_a ...
!> \param nl_b ...
!> \param nl_c ...
!> \param nl_d ...
!> \param sphi_a ...
!> \param sphi_b ...
!> \param sphi_c ...
!> \param sphi_d ...
!> \param primitives ...
!> \param buffer1 ...
!> \param buffer2 ...
! **************************************************************************************************
   SUBROUTINE contract_ssfp(work, &
                            nl_a, nl_b, nl_c, nl_d, &
                            sphi_a, sphi_b, sphi_c, sphi_d, &
                            primitives, &
                            buffer1, buffer2)
      REAL(dp), DIMENSION(1*1*10*3), INTENT(IN)          :: work
      INTEGER                                            :: nl_a, nl_b, nl_c, nl_d
      REAL(dp), DIMENSION(1, 1*nl_a), INTENT(IN)         :: sphi_a
      REAL(dp), DIMENSION(1, 1*nl_b), INTENT(IN)         :: sphi_b
      REAL(dp), DIMENSION(10, 7*nl_c), INTENT(IN)        :: sphi_c
      REAL(dp), DIMENSION(3, 3*nl_d), INTENT(IN)         :: sphi_d
      REAL(dp), &
         DIMENSION(1*nl_a, 1*nl_b, 7*nl_c, 3*nl_d)       :: primitives
      REAL(dp), DIMENSION(1*1*10*3)                      :: buffer1, buffer2

      INTEGER                                            :: i, i1, i2, i3, ia, ib, ic, id, imax, &
                                                            kmax, s_offset_a1, s_offset_b1, &
                                                            s_offset_c1, s_offset_d1

      s_offset_a1 = 0
      DO ia = 1, nl_a
         s_offset_b1 = 0
         DO ib = 1, nl_b
            s_offset_c1 = 0
            DO ic = 1, nl_c
               s_offset_d1 = 0
               DO id = 1, nl_d
                  buffer1 = 0.0_dp
                  imax = 1*10*3
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + work(1 + (i - 1)*kmax)*sphi_a(1, 1 + s_offset_a1)
                  END DO
                  buffer2 = 0.0_dp
                  imax = 1*10*3
                  kmax = 1
                  DO i = 1, imax
                     buffer2(i + imax*(1 - 1)) = buffer2(i + imax*(1 - 1)) + buffer1(1 + (i - 1)*kmax)*sphi_b(1, 1 + s_offset_b1)
                  END DO
                  buffer1 = 0.0_dp
                  imax = 1*1*3
                  kmax = 10
                  DO i = 1, imax
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 7 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 1 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 3 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 4 + s_offset_c1)
                     buffer1(i + imax*(6 - 1)) = buffer1(i + imax*(6 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 6 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 7 + s_offset_c1)
                     buffer1(i + imax*(2 - 1)) = buffer1(i + imax*(2 - 1)) + buffer2(5 + (i - 1)*kmax)*sphi_c(5, 2 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(6 + (i - 1)*kmax)*sphi_c(6, 5 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(7 + (i - 1)*kmax)*sphi_c(7, 1 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(7 + (i - 1)*kmax)*sphi_c(7, 3 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(8 + (i - 1)*kmax)*sphi_c(8, 4 + s_offset_c1)
                     buffer1(i + imax*(6 - 1)) = buffer1(i + imax*(6 - 1)) + buffer2(8 + (i - 1)*kmax)*sphi_c(8, 6 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(9 + (i - 1)*kmax)*sphi_c(9, 3 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(10 + (i - 1)*kmax)*sphi_c(10, 4 + s_offset_c1)
                  END DO
                  imax = 1*1*7
                  kmax = 3
                  i = 0
                  DO i1 = 1, 7
                  DO i2 = 1, 1
                  DO i3 = 1, 1
                     i = i + 1
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) &
                        + buffer1(1 + (i - 1)*kmax)*sphi_d(1, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) &
                        + buffer1(2 + (i - 1)*kmax)*sphi_d(2, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) &
                        + buffer1(3 + (i - 1)*kmax)*sphi_d(3, 2 + s_offset_d1)
                  END DO
                  END DO
                  END DO
                  s_offset_d1 = s_offset_d1 + 3
               END DO
               s_offset_c1 = s_offset_c1 + 7
            END DO
            s_offset_b1 = s_offset_b1 + 1
         END DO
         s_offset_a1 = s_offset_a1 + 1
      END DO
   END SUBROUTINE contract_ssfp
#endif
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
! **************************************************************************************************
!> \brief ...
!> \param work ...
!> \param nl_a ...
!> \param nl_b ...
!> \param nl_c ...
!> \param nl_d ...
!> \param sphi_a ...
!> \param sphi_b ...
!> \param sphi_c ...
!> \param sphi_d ...
!> \param primitives ...
!> \param buffer1 ...
!> \param buffer2 ...
! **************************************************************************************************
   SUBROUTINE contract_ssfd(work, &
                            nl_a, nl_b, nl_c, nl_d, &
                            sphi_a, sphi_b, sphi_c, sphi_d, &
                            primitives, &
                            buffer1, buffer2)
      REAL(dp), DIMENSION(1*1*10*6), INTENT(IN)          :: work
      INTEGER                                            :: nl_a, nl_b, nl_c, nl_d
      REAL(dp), DIMENSION(1, 1*nl_a), INTENT(IN)         :: sphi_a
      REAL(dp), DIMENSION(1, 1*nl_b), INTENT(IN)         :: sphi_b
      REAL(dp), DIMENSION(10, 7*nl_c), INTENT(IN)        :: sphi_c
      REAL(dp), DIMENSION(6, 5*nl_d), INTENT(IN)         :: sphi_d
      REAL(dp), &
         DIMENSION(1*nl_a, 1*nl_b, 7*nl_c, 5*nl_d)       :: primitives
      REAL(dp), DIMENSION(1*1*10*6)                      :: buffer1, buffer2

      INTEGER                                            :: i, i1, i2, i3, ia, ib, ic, id, imax, &
                                                            kmax, s_offset_a1, s_offset_b1, &
                                                            s_offset_c1, s_offset_d1

      s_offset_a1 = 0
      DO ia = 1, nl_a
         s_offset_b1 = 0
         DO ib = 1, nl_b
            s_offset_c1 = 0
            DO ic = 1, nl_c
               s_offset_d1 = 0
               DO id = 1, nl_d
                  buffer1 = 0.0_dp
                  imax = 1*10*6
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + work(1 + (i - 1)*kmax)*sphi_a(1, 1 + s_offset_a1)
                  END DO
                  buffer2 = 0.0_dp
                  imax = 1*10*6
                  kmax = 1
                  DO i = 1, imax
                     buffer2(i + imax*(1 - 1)) = buffer2(i + imax*(1 - 1)) + buffer1(1 + (i - 1)*kmax)*sphi_b(1, 1 + s_offset_b1)
                  END DO
                  buffer1 = 0.0_dp
                  imax = 1*1*6
                  kmax = 10
                  DO i = 1, imax
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 7 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 1 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 3 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 4 + s_offset_c1)
                     buffer1(i + imax*(6 - 1)) = buffer1(i + imax*(6 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 6 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 7 + s_offset_c1)
                     buffer1(i + imax*(2 - 1)) = buffer1(i + imax*(2 - 1)) + buffer2(5 + (i - 1)*kmax)*sphi_c(5, 2 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(6 + (i - 1)*kmax)*sphi_c(6, 5 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(7 + (i - 1)*kmax)*sphi_c(7, 1 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(7 + (i - 1)*kmax)*sphi_c(7, 3 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(8 + (i - 1)*kmax)*sphi_c(8, 4 + s_offset_c1)
                     buffer1(i + imax*(6 - 1)) = buffer1(i + imax*(6 - 1)) + buffer2(8 + (i - 1)*kmax)*sphi_c(8, 6 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(9 + (i - 1)*kmax)*sphi_c(9, 3 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(10 + (i - 1)*kmax)*sphi_c(10, 4 + s_offset_c1)
                  END DO
                  imax = 1*1*7
                  kmax = 6
                  i = 0
                  DO i1 = 1, 7
                  DO i2 = 1, 1
                  DO i3 = 1, 1
                     i = i + 1
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) + &
                        buffer1(2 + (i - 1)*kmax)*sphi_d(2, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(3 + (i - 1)*kmax)*sphi_d(3, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) + &
                        buffer1(5 + (i - 1)*kmax)*sphi_d(5, 2 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(6 + (i - 1)*kmax)*sphi_d(6, 3 + s_offset_d1)
                  END DO
                  END DO
                  END DO
                  s_offset_d1 = s_offset_d1 + 5
               END DO
               s_offset_c1 = s_offset_c1 + 7
            END DO
            s_offset_b1 = s_offset_b1 + 1
         END DO
         s_offset_a1 = s_offset_a1 + 1
      END DO
   END SUBROUTINE contract_ssfd
#endif
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
! **************************************************************************************************
!> \brief ...
!> \param work ...
!> \param nl_a ...
!> \param nl_b ...
!> \param nl_c ...
!> \param nl_d ...
!> \param sphi_a ...
!> \param sphi_b ...
!> \param sphi_c ...
!> \param sphi_d ...
!> \param primitives ...
!> \param buffer1 ...
!> \param buffer2 ...
! **************************************************************************************************
   SUBROUTINE contract_ssff(work, &
                            nl_a, nl_b, nl_c, nl_d, &
                            sphi_a, sphi_b, sphi_c, sphi_d, &
                            primitives, &
                            buffer1, buffer2)
      REAL(dp), DIMENSION(1*1*10*10), INTENT(IN)         :: work
      INTEGER                                            :: nl_a, nl_b, nl_c, nl_d
      REAL(dp), DIMENSION(1, 1*nl_a), INTENT(IN)         :: sphi_a
      REAL(dp), DIMENSION(1, 1*nl_b), INTENT(IN)         :: sphi_b
      REAL(dp), DIMENSION(10, 7*nl_c), INTENT(IN)        :: sphi_c
      REAL(dp), DIMENSION(10, 7*nl_d), INTENT(IN)        :: sphi_d
      REAL(dp), &
         DIMENSION(1*nl_a, 1*nl_b, 7*nl_c, 7*nl_d)       :: primitives
      REAL(dp), DIMENSION(1*1*10*10)                     :: buffer1, buffer2

      INTEGER                                            :: i, i1, i2, i3, ia, ib, ic, id, imax, &
                                                            kmax, s_offset_a1, s_offset_b1, &
                                                            s_offset_c1, s_offset_d1

      s_offset_a1 = 0
      DO ia = 1, nl_a
         s_offset_b1 = 0
         DO ib = 1, nl_b
            s_offset_c1 = 0
            DO ic = 1, nl_c
               s_offset_d1 = 0
               DO id = 1, nl_d
                  buffer1 = 0.0_dp
                  imax = 1*10*10
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + work(1 + (i - 1)*kmax)*sphi_a(1, 1 + s_offset_a1)
                  END DO
                  buffer2 = 0.0_dp
                  imax = 1*10*10
                  kmax = 1
                  DO i = 1, imax
                     buffer2(i + imax*(1 - 1)) = buffer2(i + imax*(1 - 1)) + buffer1(1 + (i - 1)*kmax)*sphi_b(1, 1 + s_offset_b1)
                  END DO
                  buffer1 = 0.0_dp
                  imax = 1*1*10
                  kmax = 10
                  DO i = 1, imax
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 7 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 1 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 3 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 4 + s_offset_c1)
                     buffer1(i + imax*(6 - 1)) = buffer1(i + imax*(6 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 6 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 7 + s_offset_c1)
                     buffer1(i + imax*(2 - 1)) = buffer1(i + imax*(2 - 1)) + buffer2(5 + (i - 1)*kmax)*sphi_c(5, 2 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(6 + (i - 1)*kmax)*sphi_c(6, 5 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(7 + (i - 1)*kmax)*sphi_c(7, 1 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(7 + (i - 1)*kmax)*sphi_c(7, 3 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(8 + (i - 1)*kmax)*sphi_c(8, 4 + s_offset_c1)
                     buffer1(i + imax*(6 - 1)) = buffer1(i + imax*(6 - 1)) + buffer2(8 + (i - 1)*kmax)*sphi_c(8, 6 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(9 + (i - 1)*kmax)*sphi_c(9, 3 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(10 + (i - 1)*kmax)*sphi_c(10, 4 + s_offset_c1)
                  END DO
                  imax = 1*1*7
                  kmax = 10
                  i = 0
                  DO i1 = 1, 7
                  DO i2 = 1, 1
                  DO i3 = 1, 1
                     i = i + 1
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) + &
                        buffer1(2 + (i - 1)*kmax)*sphi_d(2, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(2 + (i - 1)*kmax)*sphi_d(2, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(3 + (i - 1)*kmax)*sphi_d(3, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) + &
                        buffer1(3 + (i - 1)*kmax)*sphi_d(3, 6 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) + &
                        buffer1(5 + (i - 1)*kmax)*sphi_d(5, 2 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(6 + (i - 1)*kmax)*sphi_d(6, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) + &
                        buffer1(7 + (i - 1)*kmax)*sphi_d(7, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(7 + (i - 1)*kmax)*sphi_d(7, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(8 + (i - 1)*kmax)*sphi_d(8, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) + &
                        buffer1(8 + (i - 1)*kmax)*sphi_d(8, 6 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(9 + (i - 1)*kmax)*sphi_d(9, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(10 + (i - 1)*kmax)*sphi_d(10, 4 + s_offset_d1)
                  END DO
                  END DO
                  END DO
                  s_offset_d1 = s_offset_d1 + 7
               END DO
               s_offset_c1 = s_offset_c1 + 7
            END DO
            s_offset_b1 = s_offset_b1 + 1
         END DO
         s_offset_a1 = s_offset_a1 + 1
      END DO
   END SUBROUTINE contract_ssff
#endif
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
! **************************************************************************************************
!> \brief ...
!> \param work ...
!> \param nl_a ...
!> \param nl_b ...
!> \param nl_c ...
!> \param nl_d ...
!> \param sphi_a ...
!> \param sphi_b ...
!> \param sphi_c ...
!> \param sphi_d ...
!> \param primitives ...
!> \param buffer1 ...
!> \param buffer2 ...
! **************************************************************************************************
   SUBROUTINE contract_ssfg(work, &
                            nl_a, nl_b, nl_c, nl_d, &
                            sphi_a, sphi_b, sphi_c, sphi_d, &
                            primitives, &
                            buffer1, buffer2)
      REAL(dp), DIMENSION(1*1*10*15), INTENT(IN)         :: work
      INTEGER                                            :: nl_a, nl_b, nl_c, nl_d
      REAL(dp), DIMENSION(1, 1*nl_a), INTENT(IN)         :: sphi_a
      REAL(dp), DIMENSION(1, 1*nl_b), INTENT(IN)         :: sphi_b
      REAL(dp), DIMENSION(10, 7*nl_c), INTENT(IN)        :: sphi_c
      REAL(dp), DIMENSION(15, 9*nl_d), INTENT(IN)        :: sphi_d
      REAL(dp), &
         DIMENSION(1*nl_a, 1*nl_b, 7*nl_c, 9*nl_d)       :: primitives
      REAL(dp), DIMENSION(1*1*10*15)                     :: buffer1, buffer2

      INTEGER                                            :: i, i1, i2, i3, ia, ib, ic, id, imax, &
                                                            kmax, s_offset_a1, s_offset_b1, &
                                                            s_offset_c1, s_offset_d1

      s_offset_a1 = 0
      DO ia = 1, nl_a
         s_offset_b1 = 0
         DO ib = 1, nl_b
            s_offset_c1 = 0
            DO ic = 1, nl_c
               s_offset_d1 = 0
               DO id = 1, nl_d
                  buffer1 = 0.0_dp
                  imax = 1*10*15
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + work(1 + (i - 1)*kmax)*sphi_a(1, 1 + s_offset_a1)
                  END DO
                  buffer2 = 0.0_dp
                  imax = 1*10*15
                  kmax = 1
                  DO i = 1, imax
                     buffer2(i + imax*(1 - 1)) = buffer2(i + imax*(1 - 1)) + buffer1(1 + (i - 1)*kmax)*sphi_b(1, 1 + s_offset_b1)
                  END DO
                  buffer1 = 0.0_dp
                  imax = 1*1*15
                  kmax = 10
                  DO i = 1, imax
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 7 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 1 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 3 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 4 + s_offset_c1)
                     buffer1(i + imax*(6 - 1)) = buffer1(i + imax*(6 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 6 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 7 + s_offset_c1)
                     buffer1(i + imax*(2 - 1)) = buffer1(i + imax*(2 - 1)) + buffer2(5 + (i - 1)*kmax)*sphi_c(5, 2 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(6 + (i - 1)*kmax)*sphi_c(6, 5 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(7 + (i - 1)*kmax)*sphi_c(7, 1 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(7 + (i - 1)*kmax)*sphi_c(7, 3 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(8 + (i - 1)*kmax)*sphi_c(8, 4 + s_offset_c1)
                     buffer1(i + imax*(6 - 1)) = buffer1(i + imax*(6 - 1)) + buffer2(8 + (i - 1)*kmax)*sphi_c(8, 6 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(9 + (i - 1)*kmax)*sphi_c(9, 3 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(10 + (i - 1)*kmax)*sphi_c(10, 4 + s_offset_c1)
                  END DO
                  imax = 1*1*7
                  kmax = 15
                  i = 0
                  DO i1 = 1, 7
                  DO i2 = 1, 1
                  DO i3 = 1, 1
                     i = i + 1
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 9 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) + &
                        buffer1(2 + (i - 1)*kmax)*sphi_d(2, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(2 + (i - 1)*kmax)*sphi_d(2, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) + &
                        buffer1(3 + (i - 1)*kmax)*sphi_d(3, 6 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 8) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 8) + &
                        buffer1(3 + (i - 1)*kmax)*sphi_d(3, 8 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 9 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) + &
                        buffer1(5 + (i - 1)*kmax)*sphi_d(5, 2 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(5 + (i - 1)*kmax)*sphi_d(5, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(6 + (i - 1)*kmax)*sphi_d(6, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(6 + (i - 1)*kmax)*sphi_d(6, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) + &
                        buffer1(7 + (i - 1)*kmax)*sphi_d(7, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(7 + (i - 1)*kmax)*sphi_d(7, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) + &
                        buffer1(8 + (i - 1)*kmax)*sphi_d(8, 6 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 8) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 8) + &
                        buffer1(8 + (i - 1)*kmax)*sphi_d(8, 8 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(9 + (i - 1)*kmax)*sphi_d(9, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) + &
                        buffer1(10 + (i - 1)*kmax)*sphi_d(10, 6 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(11 + (i - 1)*kmax)*sphi_d(11, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(11 + (i - 1)*kmax)*sphi_d(11, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) + &
                        buffer1(11 + (i - 1)*kmax)*sphi_d(11, 9 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) + &
                        buffer1(12 + (i - 1)*kmax)*sphi_d(12, 2 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(12 + (i - 1)*kmax)*sphi_d(12, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(13 + (i - 1)*kmax)*sphi_d(13, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(13 + (i - 1)*kmax)*sphi_d(13, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(14 + (i - 1)*kmax)*sphi_d(14, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(15 + (i - 1)*kmax)*sphi_d(15, 5 + s_offset_d1)
                  END DO
                  END DO
                  END DO
                  s_offset_d1 = s_offset_d1 + 9
               END DO
               s_offset_c1 = s_offset_c1 + 7
            END DO
            s_offset_b1 = s_offset_b1 + 1
         END DO
         s_offset_a1 = s_offset_a1 + 1
      END DO
   END SUBROUTINE contract_ssfg
#endif
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
! **************************************************************************************************
!> \brief ...
!> \param work ...
!> \param nl_a ...
!> \param nl_b ...
!> \param nl_c ...
!> \param nl_d ...
!> \param sphi_a ...
!> \param sphi_b ...
!> \param sphi_c ...
!> \param sphi_d ...
!> \param primitives ...
!> \param buffer1 ...
!> \param buffer2 ...
! **************************************************************************************************
   SUBROUTINE contract_ssgs(work, &
                            nl_a, nl_b, nl_c, nl_d, &
                            sphi_a, sphi_b, sphi_c, sphi_d, &
                            primitives, &
                            buffer1, buffer2)
      REAL(dp), DIMENSION(1*1*15*1), INTENT(IN)          :: work
      INTEGER                                            :: nl_a, nl_b, nl_c, nl_d
      REAL(dp), DIMENSION(1, 1*nl_a), INTENT(IN)         :: sphi_a
      REAL(dp), DIMENSION(1, 1*nl_b), INTENT(IN)         :: sphi_b
      REAL(dp), DIMENSION(15, 9*nl_c), INTENT(IN)        :: sphi_c
      REAL(dp), DIMENSION(1, 1*nl_d), INTENT(IN)         :: sphi_d
      REAL(dp), &
         DIMENSION(1*nl_a, 1*nl_b, 9*nl_c, 1*nl_d)       :: primitives
      REAL(dp), DIMENSION(1*1*15*1)                      :: buffer1, buffer2

      INTEGER                                            :: i, i1, i2, i3, ia, ib, ic, id, imax, &
                                                            kmax, s_offset_a1, s_offset_b1, &
                                                            s_offset_c1, s_offset_d1

      s_offset_a1 = 0
      DO ia = 1, nl_a
         s_offset_b1 = 0
         DO ib = 1, nl_b
            s_offset_c1 = 0
            DO ic = 1, nl_c
               s_offset_d1 = 0
               DO id = 1, nl_d
                  buffer1 = 0.0_dp
                  imax = 1*15*1
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + work(1 + (i - 1)*kmax)*sphi_a(1, 1 + s_offset_a1)
                  END DO
                  buffer2 = 0.0_dp
                  imax = 1*15*1
                  kmax = 1
                  DO i = 1, imax
                     buffer2(i + imax*(1 - 1)) = buffer2(i + imax*(1 - 1)) + buffer1(1 + (i - 1)*kmax)*sphi_b(1, 1 + s_offset_b1)
                  END DO
                  buffer1 = 0.0_dp
                  imax = 1*1*1
                  kmax = 15
                  DO i = 1, imax
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 7 + s_offset_c1)
                     buffer1(i + imax*(9 - 1)) = buffer1(i + imax*(9 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 9 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 1 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 3 + s_offset_c1)
                     buffer1(i + imax*(6 - 1)) = buffer1(i + imax*(6 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 6 + s_offset_c1)
                     buffer1(i + imax*(8 - 1)) = buffer1(i + imax*(8 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 8 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 5 + s_offset_c1)
                     buffer1(i + imax*(9 - 1)) = buffer1(i + imax*(9 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 9 + s_offset_c1)
                     buffer1(i + imax*(2 - 1)) = buffer1(i + imax*(2 - 1)) + buffer2(5 + (i - 1)*kmax)*sphi_c(5, 2 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(5 + (i - 1)*kmax)*sphi_c(5, 4 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(6 + (i - 1)*kmax)*sphi_c(6, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(6 + (i - 1)*kmax)*sphi_c(6, 7 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(7 + (i - 1)*kmax)*sphi_c(7, 1 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(7 + (i - 1)*kmax)*sphi_c(7, 3 + s_offset_c1)
                     buffer1(i + imax*(6 - 1)) = buffer1(i + imax*(6 - 1)) + buffer2(8 + (i - 1)*kmax)*sphi_c(8, 6 + s_offset_c1)
                     buffer1(i + imax*(8 - 1)) = buffer1(i + imax*(8 - 1)) + buffer2(8 + (i - 1)*kmax)*sphi_c(8, 8 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(9 + (i - 1)*kmax)*sphi_c(9, 3 + s_offset_c1)
                     buffer1(i + imax*(6 - 1)) = buffer1(i + imax*(6 - 1)) + buffer2(10 + (i - 1)*kmax)*sphi_c(10, 6 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(11 + (i - 1)*kmax)*sphi_c(11, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(11 + (i - 1)*kmax)*sphi_c(11, 7 + s_offset_c1)
                     buffer1(i + imax*(9 - 1)) = buffer1(i + imax*(9 - 1)) + buffer2(11 + (i - 1)*kmax)*sphi_c(11, 9 + s_offset_c1)
                     buffer1(i + imax*(2 - 1)) = buffer1(i + imax*(2 - 1)) + buffer2(12 + (i - 1)*kmax)*sphi_c(12, 2 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(12 + (i - 1)*kmax)*sphi_c(12, 4 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(13 + (i - 1)*kmax)*sphi_c(13, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(13 + (i - 1)*kmax)*sphi_c(13, 7 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(14 + (i - 1)*kmax)*sphi_c(14, 4 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(15 + (i - 1)*kmax)*sphi_c(15, 5 + s_offset_c1)
                  END DO
                  imax = 1*1*9
                  kmax = 1
                  i = 0
                  DO i1 = 1, 9
                  DO i2 = 1, 1
                  DO i3 = 1, 1
                     i = i + 1
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) &
                        + buffer1(1 + (i - 1)*kmax)*sphi_d(1, 1 + s_offset_d1)
                  END DO
                  END DO
                  END DO
                  s_offset_d1 = s_offset_d1 + 1
               END DO
               s_offset_c1 = s_offset_c1 + 9
            END DO
            s_offset_b1 = s_offset_b1 + 1
         END DO
         s_offset_a1 = s_offset_a1 + 1
      END DO
   END SUBROUTINE contract_ssgs
#endif
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
! **************************************************************************************************
!> \brief ...
!> \param work ...
!> \param nl_a ...
!> \param nl_b ...
!> \param nl_c ...
!> \param nl_d ...
!> \param sphi_a ...
!> \param sphi_b ...
!> \param sphi_c ...
!> \param sphi_d ...
!> \param primitives ...
!> \param buffer1 ...
!> \param buffer2 ...
! **************************************************************************************************
   SUBROUTINE contract_ssgp(work, &
                            nl_a, nl_b, nl_c, nl_d, &
                            sphi_a, sphi_b, sphi_c, sphi_d, &
                            primitives, &
                            buffer1, buffer2)
      REAL(dp), DIMENSION(1*1*15*3), INTENT(IN)          :: work
      INTEGER                                            :: nl_a, nl_b, nl_c, nl_d
      REAL(dp), DIMENSION(1, 1*nl_a), INTENT(IN)         :: sphi_a
      REAL(dp), DIMENSION(1, 1*nl_b), INTENT(IN)         :: sphi_b
      REAL(dp), DIMENSION(15, 9*nl_c), INTENT(IN)        :: sphi_c
      REAL(dp), DIMENSION(3, 3*nl_d), INTENT(IN)         :: sphi_d
      REAL(dp), &
         DIMENSION(1*nl_a, 1*nl_b, 9*nl_c, 3*nl_d)       :: primitives
      REAL(dp), DIMENSION(1*1*15*3)                      :: buffer1, buffer2

      INTEGER                                            :: i, i1, i2, i3, ia, ib, ic, id, imax, &
                                                            kmax, s_offset_a1, s_offset_b1, &
                                                            s_offset_c1, s_offset_d1

      s_offset_a1 = 0
      DO ia = 1, nl_a
         s_offset_b1 = 0
         DO ib = 1, nl_b
            s_offset_c1 = 0
            DO ic = 1, nl_c
               s_offset_d1 = 0
               DO id = 1, nl_d
                  buffer1 = 0.0_dp
                  imax = 1*15*3
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + work(1 + (i - 1)*kmax)*sphi_a(1, 1 + s_offset_a1)
                  END DO
                  buffer2 = 0.0_dp
                  imax = 1*15*3
                  kmax = 1
                  DO i = 1, imax
                     buffer2(i + imax*(1 - 1)) = buffer2(i + imax*(1 - 1)) + buffer1(1 + (i - 1)*kmax)*sphi_b(1, 1 + s_offset_b1)
                  END DO
                  buffer1 = 0.0_dp
                  imax = 1*1*3
                  kmax = 15
                  DO i = 1, imax
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 7 + s_offset_c1)
                     buffer1(i + imax*(9 - 1)) = buffer1(i + imax*(9 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 9 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 1 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 3 + s_offset_c1)
                     buffer1(i + imax*(6 - 1)) = buffer1(i + imax*(6 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 6 + s_offset_c1)
                     buffer1(i + imax*(8 - 1)) = buffer1(i + imax*(8 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 8 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 5 + s_offset_c1)
                     buffer1(i + imax*(9 - 1)) = buffer1(i + imax*(9 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 9 + s_offset_c1)
                     buffer1(i + imax*(2 - 1)) = buffer1(i + imax*(2 - 1)) + buffer2(5 + (i - 1)*kmax)*sphi_c(5, 2 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(5 + (i - 1)*kmax)*sphi_c(5, 4 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(6 + (i - 1)*kmax)*sphi_c(6, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(6 + (i - 1)*kmax)*sphi_c(6, 7 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(7 + (i - 1)*kmax)*sphi_c(7, 1 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(7 + (i - 1)*kmax)*sphi_c(7, 3 + s_offset_c1)
                     buffer1(i + imax*(6 - 1)) = buffer1(i + imax*(6 - 1)) + buffer2(8 + (i - 1)*kmax)*sphi_c(8, 6 + s_offset_c1)
                     buffer1(i + imax*(8 - 1)) = buffer1(i + imax*(8 - 1)) + buffer2(8 + (i - 1)*kmax)*sphi_c(8, 8 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(9 + (i - 1)*kmax)*sphi_c(9, 3 + s_offset_c1)
                     buffer1(i + imax*(6 - 1)) = buffer1(i + imax*(6 - 1)) + buffer2(10 + (i - 1)*kmax)*sphi_c(10, 6 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(11 + (i - 1)*kmax)*sphi_c(11, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(11 + (i - 1)*kmax)*sphi_c(11, 7 + s_offset_c1)
                     buffer1(i + imax*(9 - 1)) = buffer1(i + imax*(9 - 1)) + buffer2(11 + (i - 1)*kmax)*sphi_c(11, 9 + s_offset_c1)
                     buffer1(i + imax*(2 - 1)) = buffer1(i + imax*(2 - 1)) + buffer2(12 + (i - 1)*kmax)*sphi_c(12, 2 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(12 + (i - 1)*kmax)*sphi_c(12, 4 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(13 + (i - 1)*kmax)*sphi_c(13, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(13 + (i - 1)*kmax)*sphi_c(13, 7 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(14 + (i - 1)*kmax)*sphi_c(14, 4 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(15 + (i - 1)*kmax)*sphi_c(15, 5 + s_offset_c1)
                  END DO
                  imax = 1*1*9
                  kmax = 3
                  i = 0
                  DO i1 = 1, 9
                  DO i2 = 1, 1
                  DO i3 = 1, 1
                     i = i + 1
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) &
                        + buffer1(1 + (i - 1)*kmax)*sphi_d(1, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) &
                        + buffer1(2 + (i - 1)*kmax)*sphi_d(2, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) &
                        + buffer1(3 + (i - 1)*kmax)*sphi_d(3, 2 + s_offset_d1)
                  END DO
                  END DO
                  END DO
                  s_offset_d1 = s_offset_d1 + 3
               END DO
               s_offset_c1 = s_offset_c1 + 9
            END DO
            s_offset_b1 = s_offset_b1 + 1
         END DO
         s_offset_a1 = s_offset_a1 + 1
      END DO
   END SUBROUTINE contract_ssgp
#endif
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
! **************************************************************************************************
!> \brief ...
!> \param work ...
!> \param nl_a ...
!> \param nl_b ...
!> \param nl_c ...
!> \param nl_d ...
!> \param sphi_a ...
!> \param sphi_b ...
!> \param sphi_c ...
!> \param sphi_d ...
!> \param primitives ...
!> \param buffer1 ...
!> \param buffer2 ...
! **************************************************************************************************
   SUBROUTINE contract_ssgd(work, &
                            nl_a, nl_b, nl_c, nl_d, &
                            sphi_a, sphi_b, sphi_c, sphi_d, &
                            primitives, &
                            buffer1, buffer2)
      REAL(dp), DIMENSION(1*1*15*6), INTENT(IN)          :: work
      INTEGER                                            :: nl_a, nl_b, nl_c, nl_d
      REAL(dp), DIMENSION(1, 1*nl_a), INTENT(IN)         :: sphi_a
      REAL(dp), DIMENSION(1, 1*nl_b), INTENT(IN)         :: sphi_b
      REAL(dp), DIMENSION(15, 9*nl_c), INTENT(IN)        :: sphi_c
      REAL(dp), DIMENSION(6, 5*nl_d), INTENT(IN)         :: sphi_d
      REAL(dp), &
         DIMENSION(1*nl_a, 1*nl_b, 9*nl_c, 5*nl_d)       :: primitives
      REAL(dp), DIMENSION(1*1*15*6)                      :: buffer1, buffer2

      INTEGER                                            :: i, i1, i2, i3, ia, ib, ic, id, imax, &
                                                            kmax, s_offset_a1, s_offset_b1, &
                                                            s_offset_c1, s_offset_d1

      s_offset_a1 = 0
      DO ia = 1, nl_a
         s_offset_b1 = 0
         DO ib = 1, nl_b
            s_offset_c1 = 0
            DO ic = 1, nl_c
               s_offset_d1 = 0
               DO id = 1, nl_d
                  buffer1 = 0.0_dp
                  imax = 1*15*6
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + work(1 + (i - 1)*kmax)*sphi_a(1, 1 + s_offset_a1)
                  END DO
                  buffer2 = 0.0_dp
                  imax = 1*15*6
                  kmax = 1
                  DO i = 1, imax
                     buffer2(i + imax*(1 - 1)) = buffer2(i + imax*(1 - 1)) + buffer1(1 + (i - 1)*kmax)*sphi_b(1, 1 + s_offset_b1)
                  END DO
                  buffer1 = 0.0_dp
                  imax = 1*1*6
                  kmax = 15
                  DO i = 1, imax
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 7 + s_offset_c1)
                     buffer1(i + imax*(9 - 1)) = buffer1(i + imax*(9 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 9 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 1 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 3 + s_offset_c1)
                     buffer1(i + imax*(6 - 1)) = buffer1(i + imax*(6 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 6 + s_offset_c1)
                     buffer1(i + imax*(8 - 1)) = buffer1(i + imax*(8 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 8 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 5 + s_offset_c1)
                     buffer1(i + imax*(9 - 1)) = buffer1(i + imax*(9 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 9 + s_offset_c1)
                     buffer1(i + imax*(2 - 1)) = buffer1(i + imax*(2 - 1)) + buffer2(5 + (i - 1)*kmax)*sphi_c(5, 2 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(5 + (i - 1)*kmax)*sphi_c(5, 4 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(6 + (i - 1)*kmax)*sphi_c(6, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(6 + (i - 1)*kmax)*sphi_c(6, 7 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(7 + (i - 1)*kmax)*sphi_c(7, 1 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(7 + (i - 1)*kmax)*sphi_c(7, 3 + s_offset_c1)
                     buffer1(i + imax*(6 - 1)) = buffer1(i + imax*(6 - 1)) + buffer2(8 + (i - 1)*kmax)*sphi_c(8, 6 + s_offset_c1)
                     buffer1(i + imax*(8 - 1)) = buffer1(i + imax*(8 - 1)) + buffer2(8 + (i - 1)*kmax)*sphi_c(8, 8 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(9 + (i - 1)*kmax)*sphi_c(9, 3 + s_offset_c1)
                     buffer1(i + imax*(6 - 1)) = buffer1(i + imax*(6 - 1)) + buffer2(10 + (i - 1)*kmax)*sphi_c(10, 6 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(11 + (i - 1)*kmax)*sphi_c(11, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(11 + (i - 1)*kmax)*sphi_c(11, 7 + s_offset_c1)
                     buffer1(i + imax*(9 - 1)) = buffer1(i + imax*(9 - 1)) + buffer2(11 + (i - 1)*kmax)*sphi_c(11, 9 + s_offset_c1)
                     buffer1(i + imax*(2 - 1)) = buffer1(i + imax*(2 - 1)) + buffer2(12 + (i - 1)*kmax)*sphi_c(12, 2 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(12 + (i - 1)*kmax)*sphi_c(12, 4 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(13 + (i - 1)*kmax)*sphi_c(13, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(13 + (i - 1)*kmax)*sphi_c(13, 7 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(14 + (i - 1)*kmax)*sphi_c(14, 4 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(15 + (i - 1)*kmax)*sphi_c(15, 5 + s_offset_c1)
                  END DO
                  imax = 1*1*9
                  kmax = 6
                  i = 0
                  DO i1 = 1, 9
                  DO i2 = 1, 1
                  DO i3 = 1, 1
                     i = i + 1
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) + &
                        buffer1(2 + (i - 1)*kmax)*sphi_d(2, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(3 + (i - 1)*kmax)*sphi_d(3, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) + &
                        buffer1(5 + (i - 1)*kmax)*sphi_d(5, 2 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(6 + (i - 1)*kmax)*sphi_d(6, 3 + s_offset_d1)
                  END DO
                  END DO
                  END DO
                  s_offset_d1 = s_offset_d1 + 5
               END DO
               s_offset_c1 = s_offset_c1 + 9
            END DO
            s_offset_b1 = s_offset_b1 + 1
         END DO
         s_offset_a1 = s_offset_a1 + 1
      END DO
   END SUBROUTINE contract_ssgd
#endif
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
! **************************************************************************************************
!> \brief ...
!> \param work ...
!> \param nl_a ...
!> \param nl_b ...
!> \param nl_c ...
!> \param nl_d ...
!> \param sphi_a ...
!> \param sphi_b ...
!> \param sphi_c ...
!> \param sphi_d ...
!> \param primitives ...
!> \param buffer1 ...
!> \param buffer2 ...
! **************************************************************************************************
   SUBROUTINE contract_ssgf(work, &
                            nl_a, nl_b, nl_c, nl_d, &
                            sphi_a, sphi_b, sphi_c, sphi_d, &
                            primitives, &
                            buffer1, buffer2)
      REAL(dp), DIMENSION(1*1*15*10), INTENT(IN)         :: work
      INTEGER                                            :: nl_a, nl_b, nl_c, nl_d
      REAL(dp), DIMENSION(1, 1*nl_a), INTENT(IN)         :: sphi_a
      REAL(dp), DIMENSION(1, 1*nl_b), INTENT(IN)         :: sphi_b
      REAL(dp), DIMENSION(15, 9*nl_c), INTENT(IN)        :: sphi_c
      REAL(dp), DIMENSION(10, 7*nl_d), INTENT(IN)        :: sphi_d
      REAL(dp), &
         DIMENSION(1*nl_a, 1*nl_b, 9*nl_c, 7*nl_d)       :: primitives
      REAL(dp), DIMENSION(1*1*15*10)                     :: buffer1, buffer2

      INTEGER                                            :: i, i1, i2, i3, ia, ib, ic, id, imax, &
                                                            kmax, s_offset_a1, s_offset_b1, &
                                                            s_offset_c1, s_offset_d1

      s_offset_a1 = 0
      DO ia = 1, nl_a
         s_offset_b1 = 0
         DO ib = 1, nl_b
            s_offset_c1 = 0
            DO ic = 1, nl_c
               s_offset_d1 = 0
               DO id = 1, nl_d
                  buffer1 = 0.0_dp
                  imax = 1*15*10
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + work(1 + (i - 1)*kmax)*sphi_a(1, 1 + s_offset_a1)
                  END DO
                  buffer2 = 0.0_dp
                  imax = 1*15*10
                  kmax = 1
                  DO i = 1, imax
                     buffer2(i + imax*(1 - 1)) = buffer2(i + imax*(1 - 1)) + buffer1(1 + (i - 1)*kmax)*sphi_b(1, 1 + s_offset_b1)
                  END DO
                  buffer1 = 0.0_dp
                  imax = 1*1*10
                  kmax = 15
                  DO i = 1, imax
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 7 + s_offset_c1)
                     buffer1(i + imax*(9 - 1)) = buffer1(i + imax*(9 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 9 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 1 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 3 + s_offset_c1)
                     buffer1(i + imax*(6 - 1)) = buffer1(i + imax*(6 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 6 + s_offset_c1)
                     buffer1(i + imax*(8 - 1)) = buffer1(i + imax*(8 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 8 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 5 + s_offset_c1)
                     buffer1(i + imax*(9 - 1)) = buffer1(i + imax*(9 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 9 + s_offset_c1)
                     buffer1(i + imax*(2 - 1)) = buffer1(i + imax*(2 - 1)) + buffer2(5 + (i - 1)*kmax)*sphi_c(5, 2 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(5 + (i - 1)*kmax)*sphi_c(5, 4 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(6 + (i - 1)*kmax)*sphi_c(6, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(6 + (i - 1)*kmax)*sphi_c(6, 7 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(7 + (i - 1)*kmax)*sphi_c(7, 1 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(7 + (i - 1)*kmax)*sphi_c(7, 3 + s_offset_c1)
                     buffer1(i + imax*(6 - 1)) = buffer1(i + imax*(6 - 1)) + buffer2(8 + (i - 1)*kmax)*sphi_c(8, 6 + s_offset_c1)
                     buffer1(i + imax*(8 - 1)) = buffer1(i + imax*(8 - 1)) + buffer2(8 + (i - 1)*kmax)*sphi_c(8, 8 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(9 + (i - 1)*kmax)*sphi_c(9, 3 + s_offset_c1)
                     buffer1(i + imax*(6 - 1)) = buffer1(i + imax*(6 - 1)) + buffer2(10 + (i - 1)*kmax)*sphi_c(10, 6 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(11 + (i - 1)*kmax)*sphi_c(11, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(11 + (i - 1)*kmax)*sphi_c(11, 7 + s_offset_c1)
                     buffer1(i + imax*(9 - 1)) = buffer1(i + imax*(9 - 1)) + buffer2(11 + (i - 1)*kmax)*sphi_c(11, 9 + s_offset_c1)
                     buffer1(i + imax*(2 - 1)) = buffer1(i + imax*(2 - 1)) + buffer2(12 + (i - 1)*kmax)*sphi_c(12, 2 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(12 + (i - 1)*kmax)*sphi_c(12, 4 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(13 + (i - 1)*kmax)*sphi_c(13, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(13 + (i - 1)*kmax)*sphi_c(13, 7 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(14 + (i - 1)*kmax)*sphi_c(14, 4 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(15 + (i - 1)*kmax)*sphi_c(15, 5 + s_offset_c1)
                  END DO
                  imax = 1*1*9
                  kmax = 10
                  i = 0
                  DO i1 = 1, 9
                  DO i2 = 1, 1
                  DO i3 = 1, 1
                     i = i + 1
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) + &
                        buffer1(2 + (i - 1)*kmax)*sphi_d(2, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(2 + (i - 1)*kmax)*sphi_d(2, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(3 + (i - 1)*kmax)*sphi_d(3, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) + &
                        buffer1(3 + (i - 1)*kmax)*sphi_d(3, 6 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) + &
                        buffer1(5 + (i - 1)*kmax)*sphi_d(5, 2 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(6 + (i - 1)*kmax)*sphi_d(6, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) + &
                        buffer1(7 + (i - 1)*kmax)*sphi_d(7, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(7 + (i - 1)*kmax)*sphi_d(7, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(8 + (i - 1)*kmax)*sphi_d(8, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) + &
                        buffer1(8 + (i - 1)*kmax)*sphi_d(8, 6 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(9 + (i - 1)*kmax)*sphi_d(9, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(10 + (i - 1)*kmax)*sphi_d(10, 4 + s_offset_d1)
                  END DO
                  END DO
                  END DO
                  s_offset_d1 = s_offset_d1 + 7
               END DO
               s_offset_c1 = s_offset_c1 + 9
            END DO
            s_offset_b1 = s_offset_b1 + 1
         END DO
         s_offset_a1 = s_offset_a1 + 1
      END DO
   END SUBROUTINE contract_ssgf
#endif
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
! **************************************************************************************************
!> \brief ...
!> \param work ...
!> \param nl_a ...
!> \param nl_b ...
!> \param nl_c ...
!> \param nl_d ...
!> \param sphi_a ...
!> \param sphi_b ...
!> \param sphi_c ...
!> \param sphi_d ...
!> \param primitives ...
!> \param buffer1 ...
!> \param buffer2 ...
! **************************************************************************************************
   SUBROUTINE contract_ssgg(work, &
                            nl_a, nl_b, nl_c, nl_d, &
                            sphi_a, sphi_b, sphi_c, sphi_d, &
                            primitives, &
                            buffer1, buffer2)
      REAL(dp), DIMENSION(1*1*15*15), INTENT(IN)         :: work
      INTEGER                                            :: nl_a, nl_b, nl_c, nl_d
      REAL(dp), DIMENSION(1, 1*nl_a), INTENT(IN)         :: sphi_a
      REAL(dp), DIMENSION(1, 1*nl_b), INTENT(IN)         :: sphi_b
      REAL(dp), DIMENSION(15, 9*nl_c), INTENT(IN)        :: sphi_c
      REAL(dp), DIMENSION(15, 9*nl_d), INTENT(IN)        :: sphi_d
      REAL(dp), &
         DIMENSION(1*nl_a, 1*nl_b, 9*nl_c, 9*nl_d)       :: primitives
      REAL(dp), DIMENSION(1*1*15*15)                     :: buffer1, buffer2

      INTEGER                                            :: i, i1, i2, i3, ia, ib, ic, id, imax, &
                                                            kmax, s_offset_a1, s_offset_b1, &
                                                            s_offset_c1, s_offset_d1

      s_offset_a1 = 0
      DO ia = 1, nl_a
         s_offset_b1 = 0
         DO ib = 1, nl_b
            s_offset_c1 = 0
            DO ic = 1, nl_c
               s_offset_d1 = 0
               DO id = 1, nl_d
                  buffer1 = 0.0_dp
                  imax = 1*15*15
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + work(1 + (i - 1)*kmax)*sphi_a(1, 1 + s_offset_a1)
                  END DO
                  buffer2 = 0.0_dp
                  imax = 1*15*15
                  kmax = 1
                  DO i = 1, imax
                     buffer2(i + imax*(1 - 1)) = buffer2(i + imax*(1 - 1)) + buffer1(1 + (i - 1)*kmax)*sphi_b(1, 1 + s_offset_b1)
                  END DO
                  buffer1 = 0.0_dp
                  imax = 1*1*15
                  kmax = 15
                  DO i = 1, imax
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 7 + s_offset_c1)
                     buffer1(i + imax*(9 - 1)) = buffer1(i + imax*(9 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 9 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 1 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 3 + s_offset_c1)
                     buffer1(i + imax*(6 - 1)) = buffer1(i + imax*(6 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 6 + s_offset_c1)
                     buffer1(i + imax*(8 - 1)) = buffer1(i + imax*(8 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 8 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 5 + s_offset_c1)
                     buffer1(i + imax*(9 - 1)) = buffer1(i + imax*(9 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 9 + s_offset_c1)
                     buffer1(i + imax*(2 - 1)) = buffer1(i + imax*(2 - 1)) + buffer2(5 + (i - 1)*kmax)*sphi_c(5, 2 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(5 + (i - 1)*kmax)*sphi_c(5, 4 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(6 + (i - 1)*kmax)*sphi_c(6, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(6 + (i - 1)*kmax)*sphi_c(6, 7 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(7 + (i - 1)*kmax)*sphi_c(7, 1 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(7 + (i - 1)*kmax)*sphi_c(7, 3 + s_offset_c1)
                     buffer1(i + imax*(6 - 1)) = buffer1(i + imax*(6 - 1)) + buffer2(8 + (i - 1)*kmax)*sphi_c(8, 6 + s_offset_c1)
                     buffer1(i + imax*(8 - 1)) = buffer1(i + imax*(8 - 1)) + buffer2(8 + (i - 1)*kmax)*sphi_c(8, 8 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(9 + (i - 1)*kmax)*sphi_c(9, 3 + s_offset_c1)
                     buffer1(i + imax*(6 - 1)) = buffer1(i + imax*(6 - 1)) + buffer2(10 + (i - 1)*kmax)*sphi_c(10, 6 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(11 + (i - 1)*kmax)*sphi_c(11, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(11 + (i - 1)*kmax)*sphi_c(11, 7 + s_offset_c1)
                     buffer1(i + imax*(9 - 1)) = buffer1(i + imax*(9 - 1)) + buffer2(11 + (i - 1)*kmax)*sphi_c(11, 9 + s_offset_c1)
                     buffer1(i + imax*(2 - 1)) = buffer1(i + imax*(2 - 1)) + buffer2(12 + (i - 1)*kmax)*sphi_c(12, 2 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(12 + (i - 1)*kmax)*sphi_c(12, 4 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(13 + (i - 1)*kmax)*sphi_c(13, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(13 + (i - 1)*kmax)*sphi_c(13, 7 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(14 + (i - 1)*kmax)*sphi_c(14, 4 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(15 + (i - 1)*kmax)*sphi_c(15, 5 + s_offset_c1)
                  END DO
                  imax = 1*1*9
                  kmax = 15
                  i = 0
                  DO i1 = 1, 9
                  DO i2 = 1, 1
                  DO i3 = 1, 1
                     i = i + 1
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 9 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) + &
                        buffer1(2 + (i - 1)*kmax)*sphi_d(2, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(2 + (i - 1)*kmax)*sphi_d(2, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) + &
                        buffer1(3 + (i - 1)*kmax)*sphi_d(3, 6 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 8) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 8) + &
                        buffer1(3 + (i - 1)*kmax)*sphi_d(3, 8 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 9 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) + &
                        buffer1(5 + (i - 1)*kmax)*sphi_d(5, 2 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(5 + (i - 1)*kmax)*sphi_d(5, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(6 + (i - 1)*kmax)*sphi_d(6, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(6 + (i - 1)*kmax)*sphi_d(6, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) + &
                        buffer1(7 + (i - 1)*kmax)*sphi_d(7, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(7 + (i - 1)*kmax)*sphi_d(7, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) + &
                        buffer1(8 + (i - 1)*kmax)*sphi_d(8, 6 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 8) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 8) + &
                        buffer1(8 + (i - 1)*kmax)*sphi_d(8, 8 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(9 + (i - 1)*kmax)*sphi_d(9, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) + &
                        buffer1(10 + (i - 1)*kmax)*sphi_d(10, 6 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(11 + (i - 1)*kmax)*sphi_d(11, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(11 + (i - 1)*kmax)*sphi_d(11, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) + &
                        buffer1(11 + (i - 1)*kmax)*sphi_d(11, 9 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) + &
                        buffer1(12 + (i - 1)*kmax)*sphi_d(12, 2 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(12 + (i - 1)*kmax)*sphi_d(12, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(13 + (i - 1)*kmax)*sphi_d(13, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(13 + (i - 1)*kmax)*sphi_d(13, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(14 + (i - 1)*kmax)*sphi_d(14, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(15 + (i - 1)*kmax)*sphi_d(15, 5 + s_offset_d1)
                  END DO
                  END DO
                  END DO
                  s_offset_d1 = s_offset_d1 + 9
               END DO
               s_offset_c1 = s_offset_c1 + 9
            END DO
            s_offset_b1 = s_offset_b1 + 1
         END DO
         s_offset_a1 = s_offset_a1 + 1
      END DO
   END SUBROUTINE contract_ssgg
#endif
#if __MAX_CONTR > 1 || __MAX_CONTR == 1
! **************************************************************************************************
!> \brief ...
!> \param work ...
!> \param nl_a ...
!> \param nl_b ...
!> \param nl_c ...
!> \param nl_d ...
!> \param sphi_a ...
!> \param sphi_b ...
!> \param sphi_c ...
!> \param sphi_d ...
!> \param primitives ...
!> \param buffer1 ...
!> \param buffer2 ...
! **************************************************************************************************
   SUBROUTINE contract_spss(work, &
                            nl_a, nl_b, nl_c, nl_d, &
                            sphi_a, sphi_b, sphi_c, sphi_d, &
                            primitives, &
                            buffer1, buffer2)
      REAL(dp), DIMENSION(1*3*1*1), INTENT(IN)           :: work
      INTEGER                                            :: nl_a, nl_b, nl_c, nl_d
      REAL(dp), DIMENSION(1, 1*nl_a), INTENT(IN)         :: sphi_a
      REAL(dp), DIMENSION(3, 3*nl_b), INTENT(IN)         :: sphi_b
      REAL(dp), DIMENSION(1, 1*nl_c), INTENT(IN)         :: sphi_c
      REAL(dp), DIMENSION(1, 1*nl_d), INTENT(IN)         :: sphi_d
      REAL(dp), &
         DIMENSION(1*nl_a, 3*nl_b, 1*nl_c, 1*nl_d)       :: primitives
      REAL(dp), DIMENSION(1*3*1*1)                       :: buffer1, buffer2

      INTEGER                                            :: i, i1, i2, i3, ia, ib, ic, id, imax, &
                                                            kmax, s_offset_a1, s_offset_b1, &
                                                            s_offset_c1, s_offset_d1

      s_offset_a1 = 0
      DO ia = 1, nl_a
         s_offset_b1 = 0
         DO ib = 1, nl_b
            s_offset_c1 = 0
            DO ic = 1, nl_c
               s_offset_d1 = 0
               DO id = 1, nl_d
                  buffer1 = 0.0_dp
                  imax = 3*1*1
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + work(1 + (i - 1)*kmax)*sphi_a(1, 1 + s_offset_a1)
                  END DO
                  buffer2 = 0.0_dp
                  imax = 1*1*1
                  kmax = 3
                  DO i = 1, imax
                     buffer2(i + imax*(3 - 1)) = buffer2(i + imax*(3 - 1)) + buffer1(1 + (i - 1)*kmax)*sphi_b(1, 3 + s_offset_b1)
                     buffer2(i + imax*(1 - 1)) = buffer2(i + imax*(1 - 1)) + buffer1(2 + (i - 1)*kmax)*sphi_b(2, 1 + s_offset_b1)
                     buffer2(i + imax*(2 - 1)) = buffer2(i + imax*(2 - 1)) + buffer1(3 + (i - 1)*kmax)*sphi_b(3, 2 + s_offset_b1)
                  END DO
                  buffer1 = 0.0_dp
                  imax = 1*3*1
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 1 + s_offset_c1)
                  END DO
                  imax = 1*3*1
                  kmax = 1
                  i = 0
                  DO i1 = 1, 1
                  DO i2 = 1, 3
                  DO i3 = 1, 1
                     i = i + 1
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) &
                        + buffer1(1 + (i - 1)*kmax)*sphi_d(1, 1 + s_offset_d1)
                  END DO
                  END DO
                  END DO
                  s_offset_d1 = s_offset_d1 + 1
               END DO
               s_offset_c1 = s_offset_c1 + 1
            END DO
            s_offset_b1 = s_offset_b1 + 3
         END DO
         s_offset_a1 = s_offset_a1 + 1
      END DO
   END SUBROUTINE contract_spss
#endif
#if __MAX_CONTR > 1 || __MAX_CONTR == 1
! **************************************************************************************************
!> \brief ...
!> \param work ...
!> \param nl_a ...
!> \param nl_b ...
!> \param nl_c ...
!> \param nl_d ...
!> \param sphi_a ...
!> \param sphi_b ...
!> \param sphi_c ...
!> \param sphi_d ...
!> \param primitives ...
!> \param buffer1 ...
!> \param buffer2 ...
! **************************************************************************************************
   SUBROUTINE contract_spsp(work, &
                            nl_a, nl_b, nl_c, nl_d, &
                            sphi_a, sphi_b, sphi_c, sphi_d, &
                            primitives, &
                            buffer1, buffer2)
      REAL(dp), DIMENSION(1*3*1*3), INTENT(IN)           :: work
      INTEGER                                            :: nl_a, nl_b, nl_c, nl_d
      REAL(dp), DIMENSION(1, 1*nl_a), INTENT(IN)         :: sphi_a
      REAL(dp), DIMENSION(3, 3*nl_b), INTENT(IN)         :: sphi_b
      REAL(dp), DIMENSION(1, 1*nl_c), INTENT(IN)         :: sphi_c
      REAL(dp), DIMENSION(3, 3*nl_d), INTENT(IN)         :: sphi_d
      REAL(dp), &
         DIMENSION(1*nl_a, 3*nl_b, 1*nl_c, 3*nl_d)       :: primitives
      REAL(dp), DIMENSION(1*3*1*3)                       :: buffer1, buffer2

      INTEGER                                            :: i, i1, i2, i3, ia, ib, ic, id, imax, &
                                                            kmax, s_offset_a1, s_offset_b1, &
                                                            s_offset_c1, s_offset_d1

      s_offset_a1 = 0
      DO ia = 1, nl_a
         s_offset_b1 = 0
         DO ib = 1, nl_b
            s_offset_c1 = 0
            DO ic = 1, nl_c
               s_offset_d1 = 0
               DO id = 1, nl_d
                  buffer1 = 0.0_dp
                  imax = 3*1*3
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + work(1 + (i - 1)*kmax)*sphi_a(1, 1 + s_offset_a1)
                  END DO
                  buffer2 = 0.0_dp
                  imax = 1*1*3
                  kmax = 3
                  DO i = 1, imax
                     buffer2(i + imax*(3 - 1)) = buffer2(i + imax*(3 - 1)) + buffer1(1 + (i - 1)*kmax)*sphi_b(1, 3 + s_offset_b1)
                     buffer2(i + imax*(1 - 1)) = buffer2(i + imax*(1 - 1)) + buffer1(2 + (i - 1)*kmax)*sphi_b(2, 1 + s_offset_b1)
                     buffer2(i + imax*(2 - 1)) = buffer2(i + imax*(2 - 1)) + buffer1(3 + (i - 1)*kmax)*sphi_b(3, 2 + s_offset_b1)
                  END DO
                  buffer1 = 0.0_dp
                  imax = 1*3*3
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 1 + s_offset_c1)
                  END DO
                  imax = 1*3*1
                  kmax = 3
                  i = 0
                  DO i1 = 1, 1
                  DO i2 = 1, 3
                  DO i3 = 1, 1
                     i = i + 1
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) &
                        + buffer1(1 + (i - 1)*kmax)*sphi_d(1, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) &
                        + buffer1(2 + (i - 1)*kmax)*sphi_d(2, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) &
                        + buffer1(3 + (i - 1)*kmax)*sphi_d(3, 2 + s_offset_d1)
                  END DO
                  END DO
                  END DO
                  s_offset_d1 = s_offset_d1 + 3
               END DO
               s_offset_c1 = s_offset_c1 + 1
            END DO
            s_offset_b1 = s_offset_b1 + 3
         END DO
         s_offset_a1 = s_offset_a1 + 1
      END DO
   END SUBROUTINE contract_spsp
#endif
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
! **************************************************************************************************
!> \brief ...
!> \param work ...
!> \param nl_a ...
!> \param nl_b ...
!> \param nl_c ...
!> \param nl_d ...
!> \param sphi_a ...
!> \param sphi_b ...
!> \param sphi_c ...
!> \param sphi_d ...
!> \param primitives ...
!> \param buffer1 ...
!> \param buffer2 ...
! **************************************************************************************************
   SUBROUTINE contract_spsd(work, &
                            nl_a, nl_b, nl_c, nl_d, &
                            sphi_a, sphi_b, sphi_c, sphi_d, &
                            primitives, &
                            buffer1, buffer2)
      REAL(dp), DIMENSION(1*3*1*6), INTENT(IN)           :: work
      INTEGER                                            :: nl_a, nl_b, nl_c, nl_d
      REAL(dp), DIMENSION(1, 1*nl_a), INTENT(IN)         :: sphi_a
      REAL(dp), DIMENSION(3, 3*nl_b), INTENT(IN)         :: sphi_b
      REAL(dp), DIMENSION(1, 1*nl_c), INTENT(IN)         :: sphi_c
      REAL(dp), DIMENSION(6, 5*nl_d), INTENT(IN)         :: sphi_d
      REAL(dp), &
         DIMENSION(1*nl_a, 3*nl_b, 1*nl_c, 5*nl_d)       :: primitives
      REAL(dp), DIMENSION(1*3*1*6)                       :: buffer1, buffer2

      INTEGER                                            :: i, i1, i2, i3, ia, ib, ic, id, imax, &
                                                            kmax, s_offset_a1, s_offset_b1, &
                                                            s_offset_c1, s_offset_d1

      s_offset_a1 = 0
      DO ia = 1, nl_a
         s_offset_b1 = 0
         DO ib = 1, nl_b
            s_offset_c1 = 0
            DO ic = 1, nl_c
               s_offset_d1 = 0
               DO id = 1, nl_d
                  buffer1 = 0.0_dp
                  imax = 3*1*6
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + work(1 + (i - 1)*kmax)*sphi_a(1, 1 + s_offset_a1)
                  END DO
                  buffer2 = 0.0_dp
                  imax = 1*1*6
                  kmax = 3
                  DO i = 1, imax
                     buffer2(i + imax*(3 - 1)) = buffer2(i + imax*(3 - 1)) + buffer1(1 + (i - 1)*kmax)*sphi_b(1, 3 + s_offset_b1)
                     buffer2(i + imax*(1 - 1)) = buffer2(i + imax*(1 - 1)) + buffer1(2 + (i - 1)*kmax)*sphi_b(2, 1 + s_offset_b1)
                     buffer2(i + imax*(2 - 1)) = buffer2(i + imax*(2 - 1)) + buffer1(3 + (i - 1)*kmax)*sphi_b(3, 2 + s_offset_b1)
                  END DO
                  buffer1 = 0.0_dp
                  imax = 1*3*6
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 1 + s_offset_c1)
                  END DO
                  imax = 1*3*1
                  kmax = 6
                  i = 0
                  DO i1 = 1, 1
                  DO i2 = 1, 3
                  DO i3 = 1, 1
                     i = i + 1
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) + &
                        buffer1(2 + (i - 1)*kmax)*sphi_d(2, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(3 + (i - 1)*kmax)*sphi_d(3, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) + &
                        buffer1(5 + (i - 1)*kmax)*sphi_d(5, 2 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(6 + (i - 1)*kmax)*sphi_d(6, 3 + s_offset_d1)
                  END DO
                  END DO
                  END DO
                  s_offset_d1 = s_offset_d1 + 5
               END DO
               s_offset_c1 = s_offset_c1 + 1
            END DO
            s_offset_b1 = s_offset_b1 + 3
         END DO
         s_offset_a1 = s_offset_a1 + 1
      END DO
   END SUBROUTINE contract_spsd
#endif
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
! **************************************************************************************************
!> \brief ...
!> \param work ...
!> \param nl_a ...
!> \param nl_b ...
!> \param nl_c ...
!> \param nl_d ...
!> \param sphi_a ...
!> \param sphi_b ...
!> \param sphi_c ...
!> \param sphi_d ...
!> \param primitives ...
!> \param buffer1 ...
!> \param buffer2 ...
! **************************************************************************************************
   SUBROUTINE contract_spsf(work, &
                            nl_a, nl_b, nl_c, nl_d, &
                            sphi_a, sphi_b, sphi_c, sphi_d, &
                            primitives, &
                            buffer1, buffer2)
      REAL(dp), DIMENSION(1*3*1*10), INTENT(IN)          :: work
      INTEGER                                            :: nl_a, nl_b, nl_c, nl_d
      REAL(dp), DIMENSION(1, 1*nl_a), INTENT(IN)         :: sphi_a
      REAL(dp), DIMENSION(3, 3*nl_b), INTENT(IN)         :: sphi_b
      REAL(dp), DIMENSION(1, 1*nl_c), INTENT(IN)         :: sphi_c
      REAL(dp), DIMENSION(10, 7*nl_d), INTENT(IN)        :: sphi_d
      REAL(dp), &
         DIMENSION(1*nl_a, 3*nl_b, 1*nl_c, 7*nl_d)       :: primitives
      REAL(dp), DIMENSION(1*3*1*10)                      :: buffer1, buffer2

      INTEGER                                            :: i, i1, i2, i3, ia, ib, ic, id, imax, &
                                                            kmax, s_offset_a1, s_offset_b1, &
                                                            s_offset_c1, s_offset_d1

      s_offset_a1 = 0
      DO ia = 1, nl_a
         s_offset_b1 = 0
         DO ib = 1, nl_b
            s_offset_c1 = 0
            DO ic = 1, nl_c
               s_offset_d1 = 0
               DO id = 1, nl_d
                  buffer1 = 0.0_dp
                  imax = 3*1*10
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + work(1 + (i - 1)*kmax)*sphi_a(1, 1 + s_offset_a1)
                  END DO
                  buffer2 = 0.0_dp
                  imax = 1*1*10
                  kmax = 3
                  DO i = 1, imax
                     buffer2(i + imax*(3 - 1)) = buffer2(i + imax*(3 - 1)) + buffer1(1 + (i - 1)*kmax)*sphi_b(1, 3 + s_offset_b1)
                     buffer2(i + imax*(1 - 1)) = buffer2(i + imax*(1 - 1)) + buffer1(2 + (i - 1)*kmax)*sphi_b(2, 1 + s_offset_b1)
                     buffer2(i + imax*(2 - 1)) = buffer2(i + imax*(2 - 1)) + buffer1(3 + (i - 1)*kmax)*sphi_b(3, 2 + s_offset_b1)
                  END DO
                  buffer1 = 0.0_dp
                  imax = 1*3*10
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 1 + s_offset_c1)
                  END DO
                  imax = 1*3*1
                  kmax = 10
                  i = 0
                  DO i1 = 1, 1
                  DO i2 = 1, 3
                  DO i3 = 1, 1
                     i = i + 1
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) + &
                        buffer1(2 + (i - 1)*kmax)*sphi_d(2, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(2 + (i - 1)*kmax)*sphi_d(2, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(3 + (i - 1)*kmax)*sphi_d(3, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) + &
                        buffer1(3 + (i - 1)*kmax)*sphi_d(3, 6 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) + &
                        buffer1(5 + (i - 1)*kmax)*sphi_d(5, 2 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(6 + (i - 1)*kmax)*sphi_d(6, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) + &
                        buffer1(7 + (i - 1)*kmax)*sphi_d(7, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(7 + (i - 1)*kmax)*sphi_d(7, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(8 + (i - 1)*kmax)*sphi_d(8, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) + &
                        buffer1(8 + (i - 1)*kmax)*sphi_d(8, 6 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(9 + (i - 1)*kmax)*sphi_d(9, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(10 + (i - 1)*kmax)*sphi_d(10, 4 + s_offset_d1)
                  END DO
                  END DO
                  END DO
                  s_offset_d1 = s_offset_d1 + 7
               END DO
               s_offset_c1 = s_offset_c1 + 1
            END DO
            s_offset_b1 = s_offset_b1 + 3
         END DO
         s_offset_a1 = s_offset_a1 + 1
      END DO
   END SUBROUTINE contract_spsf
#endif
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
! **************************************************************************************************
!> \brief ...
!> \param work ...
!> \param nl_a ...
!> \param nl_b ...
!> \param nl_c ...
!> \param nl_d ...
!> \param sphi_a ...
!> \param sphi_b ...
!> \param sphi_c ...
!> \param sphi_d ...
!> \param primitives ...
!> \param buffer1 ...
!> \param buffer2 ...
! **************************************************************************************************
   SUBROUTINE contract_spsg(work, &
                            nl_a, nl_b, nl_c, nl_d, &
                            sphi_a, sphi_b, sphi_c, sphi_d, &
                            primitives, &
                            buffer1, buffer2)
      REAL(dp), DIMENSION(1*3*1*15), INTENT(IN)          :: work
      INTEGER                                            :: nl_a, nl_b, nl_c, nl_d
      REAL(dp), DIMENSION(1, 1*nl_a), INTENT(IN)         :: sphi_a
      REAL(dp), DIMENSION(3, 3*nl_b), INTENT(IN)         :: sphi_b
      REAL(dp), DIMENSION(1, 1*nl_c), INTENT(IN)         :: sphi_c
      REAL(dp), DIMENSION(15, 9*nl_d), INTENT(IN)        :: sphi_d
      REAL(dp), &
         DIMENSION(1*nl_a, 3*nl_b, 1*nl_c, 9*nl_d)       :: primitives
      REAL(dp), DIMENSION(1*3*1*15)                      :: buffer1, buffer2

      INTEGER                                            :: i, i1, i2, i3, ia, ib, ic, id, imax, &
                                                            kmax, s_offset_a1, s_offset_b1, &
                                                            s_offset_c1, s_offset_d1

      s_offset_a1 = 0
      DO ia = 1, nl_a
         s_offset_b1 = 0
         DO ib = 1, nl_b
            s_offset_c1 = 0
            DO ic = 1, nl_c
               s_offset_d1 = 0
               DO id = 1, nl_d
                  buffer1 = 0.0_dp
                  imax = 3*1*15
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + work(1 + (i - 1)*kmax)*sphi_a(1, 1 + s_offset_a1)
                  END DO
                  buffer2 = 0.0_dp
                  imax = 1*1*15
                  kmax = 3
                  DO i = 1, imax
                     buffer2(i + imax*(3 - 1)) = buffer2(i + imax*(3 - 1)) + buffer1(1 + (i - 1)*kmax)*sphi_b(1, 3 + s_offset_b1)
                     buffer2(i + imax*(1 - 1)) = buffer2(i + imax*(1 - 1)) + buffer1(2 + (i - 1)*kmax)*sphi_b(2, 1 + s_offset_b1)
                     buffer2(i + imax*(2 - 1)) = buffer2(i + imax*(2 - 1)) + buffer1(3 + (i - 1)*kmax)*sphi_b(3, 2 + s_offset_b1)
                  END DO
                  buffer1 = 0.0_dp
                  imax = 1*3*15
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 1 + s_offset_c1)
                  END DO
                  imax = 1*3*1
                  kmax = 15
                  i = 0
                  DO i1 = 1, 1
                  DO i2 = 1, 3
                  DO i3 = 1, 1
                     i = i + 1
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 9 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) + &
                        buffer1(2 + (i - 1)*kmax)*sphi_d(2, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(2 + (i - 1)*kmax)*sphi_d(2, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) + &
                        buffer1(3 + (i - 1)*kmax)*sphi_d(3, 6 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 8) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 8) + &
                        buffer1(3 + (i - 1)*kmax)*sphi_d(3, 8 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 9 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) + &
                        buffer1(5 + (i - 1)*kmax)*sphi_d(5, 2 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(5 + (i - 1)*kmax)*sphi_d(5, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(6 + (i - 1)*kmax)*sphi_d(6, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(6 + (i - 1)*kmax)*sphi_d(6, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) + &
                        buffer1(7 + (i - 1)*kmax)*sphi_d(7, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(7 + (i - 1)*kmax)*sphi_d(7, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) + &
                        buffer1(8 + (i - 1)*kmax)*sphi_d(8, 6 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 8) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 8) + &
                        buffer1(8 + (i - 1)*kmax)*sphi_d(8, 8 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(9 + (i - 1)*kmax)*sphi_d(9, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) + &
                        buffer1(10 + (i - 1)*kmax)*sphi_d(10, 6 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(11 + (i - 1)*kmax)*sphi_d(11, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(11 + (i - 1)*kmax)*sphi_d(11, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) + &
                        buffer1(11 + (i - 1)*kmax)*sphi_d(11, 9 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) + &
                        buffer1(12 + (i - 1)*kmax)*sphi_d(12, 2 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(12 + (i - 1)*kmax)*sphi_d(12, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(13 + (i - 1)*kmax)*sphi_d(13, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(13 + (i - 1)*kmax)*sphi_d(13, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(14 + (i - 1)*kmax)*sphi_d(14, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(15 + (i - 1)*kmax)*sphi_d(15, 5 + s_offset_d1)
                  END DO
                  END DO
                  END DO
                  s_offset_d1 = s_offset_d1 + 9
               END DO
               s_offset_c1 = s_offset_c1 + 1
            END DO
            s_offset_b1 = s_offset_b1 + 3
         END DO
         s_offset_a1 = s_offset_a1 + 1
      END DO
   END SUBROUTINE contract_spsg
#endif
#if __MAX_CONTR > 1 || __MAX_CONTR == 1
! **************************************************************************************************
!> \brief ...
!> \param work ...
!> \param nl_a ...
!> \param nl_b ...
!> \param nl_c ...
!> \param nl_d ...
!> \param sphi_a ...
!> \param sphi_b ...
!> \param sphi_c ...
!> \param sphi_d ...
!> \param primitives ...
!> \param buffer1 ...
!> \param buffer2 ...
! **************************************************************************************************
   SUBROUTINE contract_spps(work, &
                            nl_a, nl_b, nl_c, nl_d, &
                            sphi_a, sphi_b, sphi_c, sphi_d, &
                            primitives, &
                            buffer1, buffer2)
      REAL(dp), DIMENSION(1*3*3*1), INTENT(IN)           :: work
      INTEGER                                            :: nl_a, nl_b, nl_c, nl_d
      REAL(dp), DIMENSION(1, 1*nl_a), INTENT(IN)         :: sphi_a
      REAL(dp), DIMENSION(3, 3*nl_b), INTENT(IN)         :: sphi_b
      REAL(dp), DIMENSION(3, 3*nl_c), INTENT(IN)         :: sphi_c
      REAL(dp), DIMENSION(1, 1*nl_d), INTENT(IN)         :: sphi_d
      REAL(dp), &
         DIMENSION(1*nl_a, 3*nl_b, 3*nl_c, 1*nl_d)       :: primitives
      REAL(dp), DIMENSION(1*3*3*1)                       :: buffer1, buffer2

      INTEGER                                            :: i, i1, i2, i3, ia, ib, ic, id, imax, &
                                                            kmax, s_offset_a1, s_offset_b1, &
                                                            s_offset_c1, s_offset_d1

      s_offset_a1 = 0
      DO ia = 1, nl_a
         s_offset_b1 = 0
         DO ib = 1, nl_b
            s_offset_c1 = 0
            DO ic = 1, nl_c
               s_offset_d1 = 0
               DO id = 1, nl_d
                  buffer1 = 0.0_dp
                  imax = 3*3*1
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + work(1 + (i - 1)*kmax)*sphi_a(1, 1 + s_offset_a1)
                  END DO
                  buffer2 = 0.0_dp
                  imax = 1*3*1
                  kmax = 3
                  DO i = 1, imax
                     buffer2(i + imax*(3 - 1)) = buffer2(i + imax*(3 - 1)) + buffer1(1 + (i - 1)*kmax)*sphi_b(1, 3 + s_offset_b1)
                     buffer2(i + imax*(1 - 1)) = buffer2(i + imax*(1 - 1)) + buffer1(2 + (i - 1)*kmax)*sphi_b(2, 1 + s_offset_b1)
                     buffer2(i + imax*(2 - 1)) = buffer2(i + imax*(2 - 1)) + buffer1(3 + (i - 1)*kmax)*sphi_b(3, 2 + s_offset_b1)
                  END DO
                  buffer1 = 0.0_dp
                  imax = 1*3*1
                  kmax = 3
                  DO i = 1, imax
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 3 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 1 + s_offset_c1)
                     buffer1(i + imax*(2 - 1)) = buffer1(i + imax*(2 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 2 + s_offset_c1)
                  END DO
                  imax = 1*3*3
                  kmax = 1
                  i = 0
                  DO i1 = 1, 3
                  DO i2 = 1, 3
                  DO i3 = 1, 1
                     i = i + 1
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) &
                        + buffer1(1 + (i - 1)*kmax)*sphi_d(1, 1 + s_offset_d1)
                  END DO
                  END DO
                  END DO
                  s_offset_d1 = s_offset_d1 + 1
               END DO
               s_offset_c1 = s_offset_c1 + 3
            END DO
            s_offset_b1 = s_offset_b1 + 3
         END DO
         s_offset_a1 = s_offset_a1 + 1
      END DO
   END SUBROUTINE contract_spps
#endif
#if __MAX_CONTR > 1 || __MAX_CONTR == 1
! **************************************************************************************************
!> \brief ...
!> \param work ...
!> \param nl_a ...
!> \param nl_b ...
!> \param nl_c ...
!> \param nl_d ...
!> \param sphi_a ...
!> \param sphi_b ...
!> \param sphi_c ...
!> \param sphi_d ...
!> \param primitives ...
!> \param buffer1 ...
!> \param buffer2 ...
! **************************************************************************************************
   SUBROUTINE contract_sppp(work, &
                            nl_a, nl_b, nl_c, nl_d, &
                            sphi_a, sphi_b, sphi_c, sphi_d, &
                            primitives, &
                            buffer1, buffer2)
      REAL(dp), DIMENSION(1*3*3*3), INTENT(IN)           :: work
      INTEGER                                            :: nl_a, nl_b, nl_c, nl_d
      REAL(dp), DIMENSION(1, 1*nl_a), INTENT(IN)         :: sphi_a
      REAL(dp), DIMENSION(3, 3*nl_b), INTENT(IN)         :: sphi_b
      REAL(dp), DIMENSION(3, 3*nl_c), INTENT(IN)         :: sphi_c
      REAL(dp), DIMENSION(3, 3*nl_d), INTENT(IN)         :: sphi_d
      REAL(dp), &
         DIMENSION(1*nl_a, 3*nl_b, 3*nl_c, 3*nl_d)       :: primitives
      REAL(dp), DIMENSION(1*3*3*3)                       :: buffer1, buffer2

      INTEGER                                            :: i, i1, i2, i3, ia, ib, ic, id, imax, &
                                                            kmax, s_offset_a1, s_offset_b1, &
                                                            s_offset_c1, s_offset_d1

      s_offset_a1 = 0
      DO ia = 1, nl_a
         s_offset_b1 = 0
         DO ib = 1, nl_b
            s_offset_c1 = 0
            DO ic = 1, nl_c
               s_offset_d1 = 0
               DO id = 1, nl_d
                  buffer1 = 0.0_dp
                  imax = 3*3*3
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + work(1 + (i - 1)*kmax)*sphi_a(1, 1 + s_offset_a1)
                  END DO
                  buffer2 = 0.0_dp
                  imax = 1*3*3
                  kmax = 3
                  DO i = 1, imax
                     buffer2(i + imax*(3 - 1)) = buffer2(i + imax*(3 - 1)) + buffer1(1 + (i - 1)*kmax)*sphi_b(1, 3 + s_offset_b1)
                     buffer2(i + imax*(1 - 1)) = buffer2(i + imax*(1 - 1)) + buffer1(2 + (i - 1)*kmax)*sphi_b(2, 1 + s_offset_b1)
                     buffer2(i + imax*(2 - 1)) = buffer2(i + imax*(2 - 1)) + buffer1(3 + (i - 1)*kmax)*sphi_b(3, 2 + s_offset_b1)
                  END DO
                  buffer1 = 0.0_dp
                  imax = 1*3*3
                  kmax = 3
                  DO i = 1, imax
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 3 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 1 + s_offset_c1)
                     buffer1(i + imax*(2 - 1)) = buffer1(i + imax*(2 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 2 + s_offset_c1)
                  END DO
                  imax = 1*3*3
                  kmax = 3
                  i = 0
                  DO i1 = 1, 3
                  DO i2 = 1, 3
                  DO i3 = 1, 1
                     i = i + 1
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) &
                        + buffer1(1 + (i - 1)*kmax)*sphi_d(1, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) &
                        + buffer1(2 + (i - 1)*kmax)*sphi_d(2, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) &
                        + buffer1(3 + (i - 1)*kmax)*sphi_d(3, 2 + s_offset_d1)
                  END DO
                  END DO
                  END DO
                  s_offset_d1 = s_offset_d1 + 3
               END DO
               s_offset_c1 = s_offset_c1 + 3
            END DO
            s_offset_b1 = s_offset_b1 + 3
         END DO
         s_offset_a1 = s_offset_a1 + 1
      END DO
   END SUBROUTINE contract_sppp
#endif
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
! **************************************************************************************************
!> \brief ...
!> \param work ...
!> \param nl_a ...
!> \param nl_b ...
!> \param nl_c ...
!> \param nl_d ...
!> \param sphi_a ...
!> \param sphi_b ...
!> \param sphi_c ...
!> \param sphi_d ...
!> \param primitives ...
!> \param buffer1 ...
!> \param buffer2 ...
! **************************************************************************************************
   SUBROUTINE contract_sppd(work, &
                            nl_a, nl_b, nl_c, nl_d, &
                            sphi_a, sphi_b, sphi_c, sphi_d, &
                            primitives, &
                            buffer1, buffer2)
      REAL(dp), DIMENSION(1*3*3*6), INTENT(IN)           :: work
      INTEGER                                            :: nl_a, nl_b, nl_c, nl_d
      REAL(dp), DIMENSION(1, 1*nl_a), INTENT(IN)         :: sphi_a
      REAL(dp), DIMENSION(3, 3*nl_b), INTENT(IN)         :: sphi_b
      REAL(dp), DIMENSION(3, 3*nl_c), INTENT(IN)         :: sphi_c
      REAL(dp), DIMENSION(6, 5*nl_d), INTENT(IN)         :: sphi_d
      REAL(dp), &
         DIMENSION(1*nl_a, 3*nl_b, 3*nl_c, 5*nl_d)       :: primitives
      REAL(dp), DIMENSION(1*3*3*6)                       :: buffer1, buffer2

      INTEGER                                            :: i, i1, i2, i3, ia, ib, ic, id, imax, &
                                                            kmax, s_offset_a1, s_offset_b1, &
                                                            s_offset_c1, s_offset_d1

      s_offset_a1 = 0
      DO ia = 1, nl_a
         s_offset_b1 = 0
         DO ib = 1, nl_b
            s_offset_c1 = 0
            DO ic = 1, nl_c
               s_offset_d1 = 0
               DO id = 1, nl_d
                  buffer1 = 0.0_dp
                  imax = 3*3*6
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + work(1 + (i - 1)*kmax)*sphi_a(1, 1 + s_offset_a1)
                  END DO
                  buffer2 = 0.0_dp
                  imax = 1*3*6
                  kmax = 3
                  DO i = 1, imax
                     buffer2(i + imax*(3 - 1)) = buffer2(i + imax*(3 - 1)) + buffer1(1 + (i - 1)*kmax)*sphi_b(1, 3 + s_offset_b1)
                     buffer2(i + imax*(1 - 1)) = buffer2(i + imax*(1 - 1)) + buffer1(2 + (i - 1)*kmax)*sphi_b(2, 1 + s_offset_b1)
                     buffer2(i + imax*(2 - 1)) = buffer2(i + imax*(2 - 1)) + buffer1(3 + (i - 1)*kmax)*sphi_b(3, 2 + s_offset_b1)
                  END DO
                  buffer1 = 0.0_dp
                  imax = 1*3*6
                  kmax = 3
                  DO i = 1, imax
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 3 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 1 + s_offset_c1)
                     buffer1(i + imax*(2 - 1)) = buffer1(i + imax*(2 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 2 + s_offset_c1)
                  END DO
                  imax = 1*3*3
                  kmax = 6
                  i = 0
                  DO i1 = 1, 3
                  DO i2 = 1, 3
                  DO i3 = 1, 1
                     i = i + 1
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) + &
                        buffer1(2 + (i - 1)*kmax)*sphi_d(2, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(3 + (i - 1)*kmax)*sphi_d(3, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) + &
                        buffer1(5 + (i - 1)*kmax)*sphi_d(5, 2 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(6 + (i - 1)*kmax)*sphi_d(6, 3 + s_offset_d1)
                  END DO
                  END DO
                  END DO
                  s_offset_d1 = s_offset_d1 + 5
               END DO
               s_offset_c1 = s_offset_c1 + 3
            END DO
            s_offset_b1 = s_offset_b1 + 3
         END DO
         s_offset_a1 = s_offset_a1 + 1
      END DO
   END SUBROUTINE contract_sppd
#endif
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
! **************************************************************************************************
!> \brief ...
!> \param work ...
!> \param nl_a ...
!> \param nl_b ...
!> \param nl_c ...
!> \param nl_d ...
!> \param sphi_a ...
!> \param sphi_b ...
!> \param sphi_c ...
!> \param sphi_d ...
!> \param primitives ...
!> \param buffer1 ...
!> \param buffer2 ...
! **************************************************************************************************
   SUBROUTINE contract_sppf(work, &
                            nl_a, nl_b, nl_c, nl_d, &
                            sphi_a, sphi_b, sphi_c, sphi_d, &
                            primitives, &
                            buffer1, buffer2)
      REAL(dp), DIMENSION(1*3*3*10), INTENT(IN)          :: work
      INTEGER                                            :: nl_a, nl_b, nl_c, nl_d
      REAL(dp), DIMENSION(1, 1*nl_a), INTENT(IN)         :: sphi_a
      REAL(dp), DIMENSION(3, 3*nl_b), INTENT(IN)         :: sphi_b
      REAL(dp), DIMENSION(3, 3*nl_c), INTENT(IN)         :: sphi_c
      REAL(dp), DIMENSION(10, 7*nl_d), INTENT(IN)        :: sphi_d
      REAL(dp), &
         DIMENSION(1*nl_a, 3*nl_b, 3*nl_c, 7*nl_d)       :: primitives
      REAL(dp), DIMENSION(1*3*3*10)                      :: buffer1, buffer2

      INTEGER                                            :: i, i1, i2, i3, ia, ib, ic, id, imax, &
                                                            kmax, s_offset_a1, s_offset_b1, &
                                                            s_offset_c1, s_offset_d1

      s_offset_a1 = 0
      DO ia = 1, nl_a
         s_offset_b1 = 0
         DO ib = 1, nl_b
            s_offset_c1 = 0
            DO ic = 1, nl_c
               s_offset_d1 = 0
               DO id = 1, nl_d
                  buffer1 = 0.0_dp
                  imax = 3*3*10
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + work(1 + (i - 1)*kmax)*sphi_a(1, 1 + s_offset_a1)
                  END DO
                  buffer2 = 0.0_dp
                  imax = 1*3*10
                  kmax = 3
                  DO i = 1, imax
                     buffer2(i + imax*(3 - 1)) = buffer2(i + imax*(3 - 1)) + buffer1(1 + (i - 1)*kmax)*sphi_b(1, 3 + s_offset_b1)
                     buffer2(i + imax*(1 - 1)) = buffer2(i + imax*(1 - 1)) + buffer1(2 + (i - 1)*kmax)*sphi_b(2, 1 + s_offset_b1)
                     buffer2(i + imax*(2 - 1)) = buffer2(i + imax*(2 - 1)) + buffer1(3 + (i - 1)*kmax)*sphi_b(3, 2 + s_offset_b1)
                  END DO
                  buffer1 = 0.0_dp
                  imax = 1*3*10
                  kmax = 3
                  DO i = 1, imax
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 3 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 1 + s_offset_c1)
                     buffer1(i + imax*(2 - 1)) = buffer1(i + imax*(2 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 2 + s_offset_c1)
                  END DO
                  imax = 1*3*3
                  kmax = 10
                  i = 0
                  DO i1 = 1, 3
                  DO i2 = 1, 3
                  DO i3 = 1, 1
                     i = i + 1
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) + &
                        buffer1(2 + (i - 1)*kmax)*sphi_d(2, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(2 + (i - 1)*kmax)*sphi_d(2, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(3 + (i - 1)*kmax)*sphi_d(3, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) + &
                        buffer1(3 + (i - 1)*kmax)*sphi_d(3, 6 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) + &
                        buffer1(5 + (i - 1)*kmax)*sphi_d(5, 2 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(6 + (i - 1)*kmax)*sphi_d(6, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) + &
                        buffer1(7 + (i - 1)*kmax)*sphi_d(7, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(7 + (i - 1)*kmax)*sphi_d(7, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(8 + (i - 1)*kmax)*sphi_d(8, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) + &
                        buffer1(8 + (i - 1)*kmax)*sphi_d(8, 6 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(9 + (i - 1)*kmax)*sphi_d(9, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(10 + (i - 1)*kmax)*sphi_d(10, 4 + s_offset_d1)
                  END DO
                  END DO
                  END DO
                  s_offset_d1 = s_offset_d1 + 7
               END DO
               s_offset_c1 = s_offset_c1 + 3
            END DO
            s_offset_b1 = s_offset_b1 + 3
         END DO
         s_offset_a1 = s_offset_a1 + 1
      END DO
   END SUBROUTINE contract_sppf
#endif
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
! **************************************************************************************************
!> \brief ...
!> \param work ...
!> \param nl_a ...
!> \param nl_b ...
!> \param nl_c ...
!> \param nl_d ...
!> \param sphi_a ...
!> \param sphi_b ...
!> \param sphi_c ...
!> \param sphi_d ...
!> \param primitives ...
!> \param buffer1 ...
!> \param buffer2 ...
! **************************************************************************************************
   SUBROUTINE contract_sppg(work, &
                            nl_a, nl_b, nl_c, nl_d, &
                            sphi_a, sphi_b, sphi_c, sphi_d, &
                            primitives, &
                            buffer1, buffer2)
      REAL(dp), DIMENSION(1*3*3*15), INTENT(IN)          :: work
      INTEGER                                            :: nl_a, nl_b, nl_c, nl_d
      REAL(dp), DIMENSION(1, 1*nl_a), INTENT(IN)         :: sphi_a
      REAL(dp), DIMENSION(3, 3*nl_b), INTENT(IN)         :: sphi_b
      REAL(dp), DIMENSION(3, 3*nl_c), INTENT(IN)         :: sphi_c
      REAL(dp), DIMENSION(15, 9*nl_d), INTENT(IN)        :: sphi_d
      REAL(dp), &
         DIMENSION(1*nl_a, 3*nl_b, 3*nl_c, 9*nl_d)       :: primitives
      REAL(dp), DIMENSION(1*3*3*15)                      :: buffer1, buffer2

      INTEGER                                            :: i, i1, i2, i3, ia, ib, ic, id, imax, &
                                                            kmax, s_offset_a1, s_offset_b1, &
                                                            s_offset_c1, s_offset_d1

      s_offset_a1 = 0
      DO ia = 1, nl_a
         s_offset_b1 = 0
         DO ib = 1, nl_b
            s_offset_c1 = 0
            DO ic = 1, nl_c
               s_offset_d1 = 0
               DO id = 1, nl_d
                  buffer1 = 0.0_dp
                  imax = 3*3*15
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + work(1 + (i - 1)*kmax)*sphi_a(1, 1 + s_offset_a1)
                  END DO
                  buffer2 = 0.0_dp
                  imax = 1*3*15
                  kmax = 3
                  DO i = 1, imax
                     buffer2(i + imax*(3 - 1)) = buffer2(i + imax*(3 - 1)) + buffer1(1 + (i - 1)*kmax)*sphi_b(1, 3 + s_offset_b1)
                     buffer2(i + imax*(1 - 1)) = buffer2(i + imax*(1 - 1)) + buffer1(2 + (i - 1)*kmax)*sphi_b(2, 1 + s_offset_b1)
                     buffer2(i + imax*(2 - 1)) = buffer2(i + imax*(2 - 1)) + buffer1(3 + (i - 1)*kmax)*sphi_b(3, 2 + s_offset_b1)
                  END DO
                  buffer1 = 0.0_dp
                  imax = 1*3*15
                  kmax = 3
                  DO i = 1, imax
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 3 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 1 + s_offset_c1)
                     buffer1(i + imax*(2 - 1)) = buffer1(i + imax*(2 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 2 + s_offset_c1)
                  END DO
                  imax = 1*3*3
                  kmax = 15
                  i = 0
                  DO i1 = 1, 3
                  DO i2 = 1, 3
                  DO i3 = 1, 1
                     i = i + 1
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 9 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) + &
                        buffer1(2 + (i - 1)*kmax)*sphi_d(2, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(2 + (i - 1)*kmax)*sphi_d(2, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) + &
                        buffer1(3 + (i - 1)*kmax)*sphi_d(3, 6 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 8) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 8) + &
                        buffer1(3 + (i - 1)*kmax)*sphi_d(3, 8 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 9 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) + &
                        buffer1(5 + (i - 1)*kmax)*sphi_d(5, 2 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(5 + (i - 1)*kmax)*sphi_d(5, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(6 + (i - 1)*kmax)*sphi_d(6, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(6 + (i - 1)*kmax)*sphi_d(6, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) + &
                        buffer1(7 + (i - 1)*kmax)*sphi_d(7, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(7 + (i - 1)*kmax)*sphi_d(7, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) + &
                        buffer1(8 + (i - 1)*kmax)*sphi_d(8, 6 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 8) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 8) + &
                        buffer1(8 + (i - 1)*kmax)*sphi_d(8, 8 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(9 + (i - 1)*kmax)*sphi_d(9, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) + &
                        buffer1(10 + (i - 1)*kmax)*sphi_d(10, 6 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(11 + (i - 1)*kmax)*sphi_d(11, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(11 + (i - 1)*kmax)*sphi_d(11, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) + &
                        buffer1(11 + (i - 1)*kmax)*sphi_d(11, 9 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) + &
                        buffer1(12 + (i - 1)*kmax)*sphi_d(12, 2 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(12 + (i - 1)*kmax)*sphi_d(12, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(13 + (i - 1)*kmax)*sphi_d(13, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(13 + (i - 1)*kmax)*sphi_d(13, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(14 + (i - 1)*kmax)*sphi_d(14, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(15 + (i - 1)*kmax)*sphi_d(15, 5 + s_offset_d1)
                  END DO
                  END DO
                  END DO
                  s_offset_d1 = s_offset_d1 + 9
               END DO
               s_offset_c1 = s_offset_c1 + 3
            END DO
            s_offset_b1 = s_offset_b1 + 3
         END DO
         s_offset_a1 = s_offset_a1 + 1
      END DO
   END SUBROUTINE contract_sppg
#endif
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
! **************************************************************************************************
!> \brief ...
!> \param work ...
!> \param nl_a ...
!> \param nl_b ...
!> \param nl_c ...
!> \param nl_d ...
!> \param sphi_a ...
!> \param sphi_b ...
!> \param sphi_c ...
!> \param sphi_d ...
!> \param primitives ...
!> \param buffer1 ...
!> \param buffer2 ...
! **************************************************************************************************
   SUBROUTINE contract_spds(work, &
                            nl_a, nl_b, nl_c, nl_d, &
                            sphi_a, sphi_b, sphi_c, sphi_d, &
                            primitives, &
                            buffer1, buffer2)
      REAL(dp), DIMENSION(1*3*6*1), INTENT(IN)           :: work
      INTEGER                                            :: nl_a, nl_b, nl_c, nl_d
      REAL(dp), DIMENSION(1, 1*nl_a), INTENT(IN)         :: sphi_a
      REAL(dp), DIMENSION(3, 3*nl_b), INTENT(IN)         :: sphi_b
      REAL(dp), DIMENSION(6, 5*nl_c), INTENT(IN)         :: sphi_c
      REAL(dp), DIMENSION(1, 1*nl_d), INTENT(IN)         :: sphi_d
      REAL(dp), &
         DIMENSION(1*nl_a, 3*nl_b, 5*nl_c, 1*nl_d)       :: primitives
      REAL(dp), DIMENSION(1*3*6*1)                       :: buffer1, buffer2

      INTEGER                                            :: i, i1, i2, i3, ia, ib, ic, id, imax, &
                                                            kmax, s_offset_a1, s_offset_b1, &
                                                            s_offset_c1, s_offset_d1

      s_offset_a1 = 0
      DO ia = 1, nl_a
         s_offset_b1 = 0
         DO ib = 1, nl_b
            s_offset_c1 = 0
            DO ic = 1, nl_c
               s_offset_d1 = 0
               DO id = 1, nl_d
                  buffer1 = 0.0_dp
                  imax = 3*6*1
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + work(1 + (i - 1)*kmax)*sphi_a(1, 1 + s_offset_a1)
                  END DO
                  buffer2 = 0.0_dp
                  imax = 1*6*1
                  kmax = 3
                  DO i = 1, imax
                     buffer2(i + imax*(3 - 1)) = buffer2(i + imax*(3 - 1)) + buffer1(1 + (i - 1)*kmax)*sphi_b(1, 3 + s_offset_b1)
                     buffer2(i + imax*(1 - 1)) = buffer2(i + imax*(1 - 1)) + buffer1(2 + (i - 1)*kmax)*sphi_b(2, 1 + s_offset_b1)
                     buffer2(i + imax*(2 - 1)) = buffer2(i + imax*(2 - 1)) + buffer1(3 + (i - 1)*kmax)*sphi_b(3, 2 + s_offset_b1)
                  END DO
                  buffer1 = 0.0_dp
                  imax = 1*3*1
                  kmax = 6
                  DO i = 1, imax
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 3 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 5 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 1 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 4 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 3 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 5 + s_offset_c1)
                     buffer1(i + imax*(2 - 1)) = buffer1(i + imax*(2 - 1)) + buffer2(5 + (i - 1)*kmax)*sphi_c(5, 2 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(6 + (i - 1)*kmax)*sphi_c(6, 3 + s_offset_c1)
                  END DO
                  imax = 1*3*5
                  kmax = 1
                  i = 0
                  DO i1 = 1, 5
                  DO i2 = 1, 3
                  DO i3 = 1, 1
                     i = i + 1
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) &
                        + buffer1(1 + (i - 1)*kmax)*sphi_d(1, 1 + s_offset_d1)
                  END DO
                  END DO
                  END DO
                  s_offset_d1 = s_offset_d1 + 1
               END DO
               s_offset_c1 = s_offset_c1 + 5
            END DO
            s_offset_b1 = s_offset_b1 + 3
         END DO
         s_offset_a1 = s_offset_a1 + 1
      END DO
   END SUBROUTINE contract_spds
#endif
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
! **************************************************************************************************
!> \brief ...
!> \param work ...
!> \param nl_a ...
!> \param nl_b ...
!> \param nl_c ...
!> \param nl_d ...
!> \param sphi_a ...
!> \param sphi_b ...
!> \param sphi_c ...
!> \param sphi_d ...
!> \param primitives ...
!> \param buffer1 ...
!> \param buffer2 ...
! **************************************************************************************************
   SUBROUTINE contract_spdp(work, &
                            nl_a, nl_b, nl_c, nl_d, &
                            sphi_a, sphi_b, sphi_c, sphi_d, &
                            primitives, &
                            buffer1, buffer2)
      REAL(dp), DIMENSION(1*3*6*3), INTENT(IN)           :: work
      INTEGER                                            :: nl_a, nl_b, nl_c, nl_d
      REAL(dp), DIMENSION(1, 1*nl_a), INTENT(IN)         :: sphi_a
      REAL(dp), DIMENSION(3, 3*nl_b), INTENT(IN)         :: sphi_b
      REAL(dp), DIMENSION(6, 5*nl_c), INTENT(IN)         :: sphi_c
      REAL(dp), DIMENSION(3, 3*nl_d), INTENT(IN)         :: sphi_d
      REAL(dp), &
         DIMENSION(1*nl_a, 3*nl_b, 5*nl_c, 3*nl_d)       :: primitives
      REAL(dp), DIMENSION(1*3*6*3)                       :: buffer1, buffer2

      INTEGER                                            :: i, i1, i2, i3, ia, ib, ic, id, imax, &
                                                            kmax, s_offset_a1, s_offset_b1, &
                                                            s_offset_c1, s_offset_d1

      s_offset_a1 = 0
      DO ia = 1, nl_a
         s_offset_b1 = 0
         DO ib = 1, nl_b
            s_offset_c1 = 0
            DO ic = 1, nl_c
               s_offset_d1 = 0
               DO id = 1, nl_d
                  buffer1 = 0.0_dp
                  imax = 3*6*3
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + work(1 + (i - 1)*kmax)*sphi_a(1, 1 + s_offset_a1)
                  END DO
                  buffer2 = 0.0_dp
                  imax = 1*6*3
                  kmax = 3
                  DO i = 1, imax
                     buffer2(i + imax*(3 - 1)) = buffer2(i + imax*(3 - 1)) + buffer1(1 + (i - 1)*kmax)*sphi_b(1, 3 + s_offset_b1)
                     buffer2(i + imax*(1 - 1)) = buffer2(i + imax*(1 - 1)) + buffer1(2 + (i - 1)*kmax)*sphi_b(2, 1 + s_offset_b1)
                     buffer2(i + imax*(2 - 1)) = buffer2(i + imax*(2 - 1)) + buffer1(3 + (i - 1)*kmax)*sphi_b(3, 2 + s_offset_b1)
                  END DO
                  buffer1 = 0.0_dp
                  imax = 1*3*3
                  kmax = 6
                  DO i = 1, imax
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 3 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 5 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 1 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 4 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 3 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 5 + s_offset_c1)
                     buffer1(i + imax*(2 - 1)) = buffer1(i + imax*(2 - 1)) + buffer2(5 + (i - 1)*kmax)*sphi_c(5, 2 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(6 + (i - 1)*kmax)*sphi_c(6, 3 + s_offset_c1)
                  END DO
                  imax = 1*3*5
                  kmax = 3
                  i = 0
                  DO i1 = 1, 5
                  DO i2 = 1, 3
                  DO i3 = 1, 1
                     i = i + 1
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) &
                        + buffer1(1 + (i - 1)*kmax)*sphi_d(1, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) &
                        + buffer1(2 + (i - 1)*kmax)*sphi_d(2, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) &
                        + buffer1(3 + (i - 1)*kmax)*sphi_d(3, 2 + s_offset_d1)
                  END DO
                  END DO
                  END DO
                  s_offset_d1 = s_offset_d1 + 3
               END DO
               s_offset_c1 = s_offset_c1 + 5
            END DO
            s_offset_b1 = s_offset_b1 + 3
         END DO
         s_offset_a1 = s_offset_a1 + 1
      END DO
   END SUBROUTINE contract_spdp
#endif
#if __MAX_CONTR > 2 || __MAX_CONTR == 2
! **************************************************************************************************
!> \brief ...
!> \param work ...
!> \param nl_a ...
!> \param nl_b ...
!> \param nl_c ...
!> \param nl_d ...
!> \param sphi_a ...
!> \param sphi_b ...
!> \param sphi_c ...
!> \param sphi_d ...
!> \param primitives ...
!> \param buffer1 ...
!> \param buffer2 ...
! **************************************************************************************************
   SUBROUTINE contract_spdd(work, &
                            nl_a, nl_b, nl_c, nl_d, &
                            sphi_a, sphi_b, sphi_c, sphi_d, &
                            primitives, &
                            buffer1, buffer2)
      REAL(dp), DIMENSION(1*3*6*6), INTENT(IN)           :: work
      INTEGER                                            :: nl_a, nl_b, nl_c, nl_d
      REAL(dp), DIMENSION(1, 1*nl_a), INTENT(IN)         :: sphi_a
      REAL(dp), DIMENSION(3, 3*nl_b), INTENT(IN)         :: sphi_b
      REAL(dp), DIMENSION(6, 5*nl_c), INTENT(IN)         :: sphi_c
      REAL(dp), DIMENSION(6, 5*nl_d), INTENT(IN)         :: sphi_d
      REAL(dp), &
         DIMENSION(1*nl_a, 3*nl_b, 5*nl_c, 5*nl_d)       :: primitives
      REAL(dp), DIMENSION(1*3*6*6)                       :: buffer1, buffer2

      INTEGER                                            :: i, i1, i2, i3, ia, ib, ic, id, imax, &
                                                            kmax, s_offset_a1, s_offset_b1, &
                                                            s_offset_c1, s_offset_d1

      s_offset_a1 = 0
      DO ia = 1, nl_a
         s_offset_b1 = 0
         DO ib = 1, nl_b
            s_offset_c1 = 0
            DO ic = 1, nl_c
               s_offset_d1 = 0
               DO id = 1, nl_d
                  buffer1 = 0.0_dp
                  imax = 3*6*6
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + work(1 + (i - 1)*kmax)*sphi_a(1, 1 + s_offset_a1)
                  END DO
                  buffer2 = 0.0_dp
                  imax = 1*6*6
                  kmax = 3
                  DO i = 1, imax
                     buffer2(i + imax*(3 - 1)) = buffer2(i + imax*(3 - 1)) + buffer1(1 + (i - 1)*kmax)*sphi_b(1, 3 + s_offset_b1)
                     buffer2(i + imax*(1 - 1)) = buffer2(i + imax*(1 - 1)) + buffer1(2 + (i - 1)*kmax)*sphi_b(2, 1 + s_offset_b1)
                     buffer2(i + imax*(2 - 1)) = buffer2(i + imax*(2 - 1)) + buffer1(3 + (i - 1)*kmax)*sphi_b(3, 2 + s_offset_b1)
                  END DO
                  buffer1 = 0.0_dp
                  imax = 1*3*6
                  kmax = 6
                  DO i = 1, imax
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 3 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 5 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 1 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 4 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 3 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 5 + s_offset_c1)
                     buffer1(i + imax*(2 - 1)) = buffer1(i + imax*(2 - 1)) + buffer2(5 + (i - 1)*kmax)*sphi_c(5, 2 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(6 + (i - 1)*kmax)*sphi_c(6, 3 + s_offset_c1)
                  END DO
                  imax = 1*3*5
                  kmax = 6
                  i = 0
                  DO i1 = 1, 5
                  DO i2 = 1, 3
                  DO i3 = 1, 1
                     i = i + 1
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) + &
                        buffer1(2 + (i - 1)*kmax)*sphi_d(2, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(3 + (i - 1)*kmax)*sphi_d(3, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) + &
                        buffer1(5 + (i - 1)*kmax)*sphi_d(5, 2 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(6 + (i - 1)*kmax)*sphi_d(6, 3 + s_offset_d1)
                  END DO
                  END DO
                  END DO
                  s_offset_d1 = s_offset_d1 + 5
               END DO
               s_offset_c1 = s_offset_c1 + 5
            END DO
            s_offset_b1 = s_offset_b1 + 3
         END DO
         s_offset_a1 = s_offset_a1 + 1
      END DO
   END SUBROUTINE contract_spdd
#endif
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
! **************************************************************************************************
!> \brief ...
!> \param work ...
!> \param nl_a ...
!> \param nl_b ...
!> \param nl_c ...
!> \param nl_d ...
!> \param sphi_a ...
!> \param sphi_b ...
!> \param sphi_c ...
!> \param sphi_d ...
!> \param primitives ...
!> \param buffer1 ...
!> \param buffer2 ...
! **************************************************************************************************
   SUBROUTINE contract_spdf(work, &
                            nl_a, nl_b, nl_c, nl_d, &
                            sphi_a, sphi_b, sphi_c, sphi_d, &
                            primitives, &
                            buffer1, buffer2)
      REAL(dp), DIMENSION(1*3*6*10), INTENT(IN)          :: work
      INTEGER                                            :: nl_a, nl_b, nl_c, nl_d
      REAL(dp), DIMENSION(1, 1*nl_a), INTENT(IN)         :: sphi_a
      REAL(dp), DIMENSION(3, 3*nl_b), INTENT(IN)         :: sphi_b
      REAL(dp), DIMENSION(6, 5*nl_c), INTENT(IN)         :: sphi_c
      REAL(dp), DIMENSION(10, 7*nl_d), INTENT(IN)        :: sphi_d
      REAL(dp), &
         DIMENSION(1*nl_a, 3*nl_b, 5*nl_c, 7*nl_d)       :: primitives
      REAL(dp), DIMENSION(1*3*6*10)                      :: buffer1, buffer2

      INTEGER                                            :: i, i1, i2, i3, ia, ib, ic, id, imax, &
                                                            kmax, s_offset_a1, s_offset_b1, &
                                                            s_offset_c1, s_offset_d1

      s_offset_a1 = 0
      DO ia = 1, nl_a
         s_offset_b1 = 0
         DO ib = 1, nl_b
            s_offset_c1 = 0
            DO ic = 1, nl_c
               s_offset_d1 = 0
               DO id = 1, nl_d
                  buffer1 = 0.0_dp
                  imax = 3*6*10
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + work(1 + (i - 1)*kmax)*sphi_a(1, 1 + s_offset_a1)
                  END DO
                  buffer2 = 0.0_dp
                  imax = 1*6*10
                  kmax = 3
                  DO i = 1, imax
                     buffer2(i + imax*(3 - 1)) = buffer2(i + imax*(3 - 1)) + buffer1(1 + (i - 1)*kmax)*sphi_b(1, 3 + s_offset_b1)
                     buffer2(i + imax*(1 - 1)) = buffer2(i + imax*(1 - 1)) + buffer1(2 + (i - 1)*kmax)*sphi_b(2, 1 + s_offset_b1)
                     buffer2(i + imax*(2 - 1)) = buffer2(i + imax*(2 - 1)) + buffer1(3 + (i - 1)*kmax)*sphi_b(3, 2 + s_offset_b1)
                  END DO
                  buffer1 = 0.0_dp
                  imax = 1*3*10
                  kmax = 6
                  DO i = 1, imax
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 3 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 5 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 1 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 4 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 3 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 5 + s_offset_c1)
                     buffer1(i + imax*(2 - 1)) = buffer1(i + imax*(2 - 1)) + buffer2(5 + (i - 1)*kmax)*sphi_c(5, 2 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(6 + (i - 1)*kmax)*sphi_c(6, 3 + s_offset_c1)
                  END DO
                  imax = 1*3*5
                  kmax = 10
                  i = 0
                  DO i1 = 1, 5
                  DO i2 = 1, 3
                  DO i3 = 1, 1
                     i = i + 1
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) + &
                        buffer1(2 + (i - 1)*kmax)*sphi_d(2, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(2 + (i - 1)*kmax)*sphi_d(2, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(3 + (i - 1)*kmax)*sphi_d(3, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) + &
                        buffer1(3 + (i - 1)*kmax)*sphi_d(3, 6 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) + &
                        buffer1(5 + (i - 1)*kmax)*sphi_d(5, 2 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(6 + (i - 1)*kmax)*sphi_d(6, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) + &
                        buffer1(7 + (i - 1)*kmax)*sphi_d(7, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(7 + (i - 1)*kmax)*sphi_d(7, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(8 + (i - 1)*kmax)*sphi_d(8, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) + &
                        buffer1(8 + (i - 1)*kmax)*sphi_d(8, 6 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(9 + (i - 1)*kmax)*sphi_d(9, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(10 + (i - 1)*kmax)*sphi_d(10, 4 + s_offset_d1)
                  END DO
                  END DO
                  END DO
                  s_offset_d1 = s_offset_d1 + 7
               END DO
               s_offset_c1 = s_offset_c1 + 5
            END DO
            s_offset_b1 = s_offset_b1 + 3
         END DO
         s_offset_a1 = s_offset_a1 + 1
      END DO
   END SUBROUTINE contract_spdf
#endif
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
! **************************************************************************************************
!> \brief ...
!> \param work ...
!> \param nl_a ...
!> \param nl_b ...
!> \param nl_c ...
!> \param nl_d ...
!> \param sphi_a ...
!> \param sphi_b ...
!> \param sphi_c ...
!> \param sphi_d ...
!> \param primitives ...
!> \param buffer1 ...
!> \param buffer2 ...
! **************************************************************************************************
   SUBROUTINE contract_spdg(work, &
                            nl_a, nl_b, nl_c, nl_d, &
                            sphi_a, sphi_b, sphi_c, sphi_d, &
                            primitives, &
                            buffer1, buffer2)
      REAL(dp), DIMENSION(1*3*6*15), INTENT(IN)          :: work
      INTEGER                                            :: nl_a, nl_b, nl_c, nl_d
      REAL(dp), DIMENSION(1, 1*nl_a), INTENT(IN)         :: sphi_a
      REAL(dp), DIMENSION(3, 3*nl_b), INTENT(IN)         :: sphi_b
      REAL(dp), DIMENSION(6, 5*nl_c), INTENT(IN)         :: sphi_c
      REAL(dp), DIMENSION(15, 9*nl_d), INTENT(IN)        :: sphi_d
      REAL(dp), &
         DIMENSION(1*nl_a, 3*nl_b, 5*nl_c, 9*nl_d)       :: primitives
      REAL(dp), DIMENSION(1*3*6*15)                      :: buffer1, buffer2

      INTEGER                                            :: i, i1, i2, i3, ia, ib, ic, id, imax, &
                                                            kmax, s_offset_a1, s_offset_b1, &
                                                            s_offset_c1, s_offset_d1

      s_offset_a1 = 0
      DO ia = 1, nl_a
         s_offset_b1 = 0
         DO ib = 1, nl_b
            s_offset_c1 = 0
            DO ic = 1, nl_c
               s_offset_d1 = 0
               DO id = 1, nl_d
                  buffer1 = 0.0_dp
                  imax = 3*6*15
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + work(1 + (i - 1)*kmax)*sphi_a(1, 1 + s_offset_a1)
                  END DO
                  buffer2 = 0.0_dp
                  imax = 1*6*15
                  kmax = 3
                  DO i = 1, imax
                     buffer2(i + imax*(3 - 1)) = buffer2(i + imax*(3 - 1)) + buffer1(1 + (i - 1)*kmax)*sphi_b(1, 3 + s_offset_b1)
                     buffer2(i + imax*(1 - 1)) = buffer2(i + imax*(1 - 1)) + buffer1(2 + (i - 1)*kmax)*sphi_b(2, 1 + s_offset_b1)
                     buffer2(i + imax*(2 - 1)) = buffer2(i + imax*(2 - 1)) + buffer1(3 + (i - 1)*kmax)*sphi_b(3, 2 + s_offset_b1)
                  END DO
                  buffer1 = 0.0_dp
                  imax = 1*3*15
                  kmax = 6
                  DO i = 1, imax
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 3 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 5 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 1 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 4 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 3 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 5 + s_offset_c1)
                     buffer1(i + imax*(2 - 1)) = buffer1(i + imax*(2 - 1)) + buffer2(5 + (i - 1)*kmax)*sphi_c(5, 2 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(6 + (i - 1)*kmax)*sphi_c(6, 3 + s_offset_c1)
                  END DO
                  imax = 1*3*5
                  kmax = 15
                  i = 0
                  DO i1 = 1, 5
                  DO i2 = 1, 3
                  DO i3 = 1, 1
                     i = i + 1
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 9 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) + &
                        buffer1(2 + (i - 1)*kmax)*sphi_d(2, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(2 + (i - 1)*kmax)*sphi_d(2, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) + &
                        buffer1(3 + (i - 1)*kmax)*sphi_d(3, 6 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 8) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 8) + &
                        buffer1(3 + (i - 1)*kmax)*sphi_d(3, 8 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 9 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) + &
                        buffer1(5 + (i - 1)*kmax)*sphi_d(5, 2 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(5 + (i - 1)*kmax)*sphi_d(5, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(6 + (i - 1)*kmax)*sphi_d(6, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(6 + (i - 1)*kmax)*sphi_d(6, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) + &
                        buffer1(7 + (i - 1)*kmax)*sphi_d(7, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(7 + (i - 1)*kmax)*sphi_d(7, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) + &
                        buffer1(8 + (i - 1)*kmax)*sphi_d(8, 6 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 8) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 8) + &
                        buffer1(8 + (i - 1)*kmax)*sphi_d(8, 8 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(9 + (i - 1)*kmax)*sphi_d(9, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) + &
                        buffer1(10 + (i - 1)*kmax)*sphi_d(10, 6 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(11 + (i - 1)*kmax)*sphi_d(11, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(11 + (i - 1)*kmax)*sphi_d(11, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) + &
                        buffer1(11 + (i - 1)*kmax)*sphi_d(11, 9 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) + &
                        buffer1(12 + (i - 1)*kmax)*sphi_d(12, 2 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(12 + (i - 1)*kmax)*sphi_d(12, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(13 + (i - 1)*kmax)*sphi_d(13, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(13 + (i - 1)*kmax)*sphi_d(13, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(14 + (i - 1)*kmax)*sphi_d(14, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(15 + (i - 1)*kmax)*sphi_d(15, 5 + s_offset_d1)
                  END DO
                  END DO
                  END DO
                  s_offset_d1 = s_offset_d1 + 9
               END DO
               s_offset_c1 = s_offset_c1 + 5
            END DO
            s_offset_b1 = s_offset_b1 + 3
         END DO
         s_offset_a1 = s_offset_a1 + 1
      END DO
   END SUBROUTINE contract_spdg
#endif
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
! **************************************************************************************************
!> \brief ...
!> \param work ...
!> \param nl_a ...
!> \param nl_b ...
!> \param nl_c ...
!> \param nl_d ...
!> \param sphi_a ...
!> \param sphi_b ...
!> \param sphi_c ...
!> \param sphi_d ...
!> \param primitives ...
!> \param buffer1 ...
!> \param buffer2 ...
! **************************************************************************************************
   SUBROUTINE contract_spfs(work, &
                            nl_a, nl_b, nl_c, nl_d, &
                            sphi_a, sphi_b, sphi_c, sphi_d, &
                            primitives, &
                            buffer1, buffer2)
      REAL(dp), DIMENSION(1*3*10*1), INTENT(IN)          :: work
      INTEGER                                            :: nl_a, nl_b, nl_c, nl_d
      REAL(dp), DIMENSION(1, 1*nl_a), INTENT(IN)         :: sphi_a
      REAL(dp), DIMENSION(3, 3*nl_b), INTENT(IN)         :: sphi_b
      REAL(dp), DIMENSION(10, 7*nl_c), INTENT(IN)        :: sphi_c
      REAL(dp), DIMENSION(1, 1*nl_d), INTENT(IN)         :: sphi_d
      REAL(dp), &
         DIMENSION(1*nl_a, 3*nl_b, 7*nl_c, 1*nl_d)       :: primitives
      REAL(dp), DIMENSION(1*3*10*1)                      :: buffer1, buffer2

      INTEGER                                            :: i, i1, i2, i3, ia, ib, ic, id, imax, &
                                                            kmax, s_offset_a1, s_offset_b1, &
                                                            s_offset_c1, s_offset_d1

      s_offset_a1 = 0
      DO ia = 1, nl_a
         s_offset_b1 = 0
         DO ib = 1, nl_b
            s_offset_c1 = 0
            DO ic = 1, nl_c
               s_offset_d1 = 0
               DO id = 1, nl_d
                  buffer1 = 0.0_dp
                  imax = 3*10*1
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + work(1 + (i - 1)*kmax)*sphi_a(1, 1 + s_offset_a1)
                  END DO
                  buffer2 = 0.0_dp
                  imax = 1*10*1
                  kmax = 3
                  DO i = 1, imax
                     buffer2(i + imax*(3 - 1)) = buffer2(i + imax*(3 - 1)) + buffer1(1 + (i - 1)*kmax)*sphi_b(1, 3 + s_offset_b1)
                     buffer2(i + imax*(1 - 1)) = buffer2(i + imax*(1 - 1)) + buffer1(2 + (i - 1)*kmax)*sphi_b(2, 1 + s_offset_b1)
                     buffer2(i + imax*(2 - 1)) = buffer2(i + imax*(2 - 1)) + buffer1(3 + (i - 1)*kmax)*sphi_b(3, 2 + s_offset_b1)
                  END DO
                  buffer1 = 0.0_dp
                  imax = 1*3*1
                  kmax = 10
                  DO i = 1, imax
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 7 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 1 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 3 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 4 + s_offset_c1)
                     buffer1(i + imax*(6 - 1)) = buffer1(i + imax*(6 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 6 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 7 + s_offset_c1)
                     buffer1(i + imax*(2 - 1)) = buffer1(i + imax*(2 - 1)) + buffer2(5 + (i - 1)*kmax)*sphi_c(5, 2 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(6 + (i - 1)*kmax)*sphi_c(6, 5 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(7 + (i - 1)*kmax)*sphi_c(7, 1 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(7 + (i - 1)*kmax)*sphi_c(7, 3 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(8 + (i - 1)*kmax)*sphi_c(8, 4 + s_offset_c1)
                     buffer1(i + imax*(6 - 1)) = buffer1(i + imax*(6 - 1)) + buffer2(8 + (i - 1)*kmax)*sphi_c(8, 6 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(9 + (i - 1)*kmax)*sphi_c(9, 3 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(10 + (i - 1)*kmax)*sphi_c(10, 4 + s_offset_c1)
                  END DO
                  imax = 1*3*7
                  kmax = 1
                  i = 0
                  DO i1 = 1, 7
                  DO i2 = 1, 3
                  DO i3 = 1, 1
                     i = i + 1
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) &
                        + buffer1(1 + (i - 1)*kmax)*sphi_d(1, 1 + s_offset_d1)
                  END DO
                  END DO
                  END DO
                  s_offset_d1 = s_offset_d1 + 1
               END DO
               s_offset_c1 = s_offset_c1 + 7
            END DO
            s_offset_b1 = s_offset_b1 + 3
         END DO
         s_offset_a1 = s_offset_a1 + 1
      END DO
   END SUBROUTINE contract_spfs
#endif
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
! **************************************************************************************************
!> \brief ...
!> \param work ...
!> \param nl_a ...
!> \param nl_b ...
!> \param nl_c ...
!> \param nl_d ...
!> \param sphi_a ...
!> \param sphi_b ...
!> \param sphi_c ...
!> \param sphi_d ...
!> \param primitives ...
!> \param buffer1 ...
!> \param buffer2 ...
! **************************************************************************************************
   SUBROUTINE contract_spfp(work, &
                            nl_a, nl_b, nl_c, nl_d, &
                            sphi_a, sphi_b, sphi_c, sphi_d, &
                            primitives, &
                            buffer1, buffer2)
      REAL(dp), DIMENSION(1*3*10*3), INTENT(IN)          :: work
      INTEGER                                            :: nl_a, nl_b, nl_c, nl_d
      REAL(dp), DIMENSION(1, 1*nl_a), INTENT(IN)         :: sphi_a
      REAL(dp), DIMENSION(3, 3*nl_b), INTENT(IN)         :: sphi_b
      REAL(dp), DIMENSION(10, 7*nl_c), INTENT(IN)        :: sphi_c
      REAL(dp), DIMENSION(3, 3*nl_d), INTENT(IN)         :: sphi_d
      REAL(dp), &
         DIMENSION(1*nl_a, 3*nl_b, 7*nl_c, 3*nl_d)       :: primitives
      REAL(dp), DIMENSION(1*3*10*3)                      :: buffer1, buffer2

      INTEGER                                            :: i, i1, i2, i3, ia, ib, ic, id, imax, &
                                                            kmax, s_offset_a1, s_offset_b1, &
                                                            s_offset_c1, s_offset_d1

      s_offset_a1 = 0
      DO ia = 1, nl_a
         s_offset_b1 = 0
         DO ib = 1, nl_b
            s_offset_c1 = 0
            DO ic = 1, nl_c
               s_offset_d1 = 0
               DO id = 1, nl_d
                  buffer1 = 0.0_dp
                  imax = 3*10*3
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + work(1 + (i - 1)*kmax)*sphi_a(1, 1 + s_offset_a1)
                  END DO
                  buffer2 = 0.0_dp
                  imax = 1*10*3
                  kmax = 3
                  DO i = 1, imax
                     buffer2(i + imax*(3 - 1)) = buffer2(i + imax*(3 - 1)) + buffer1(1 + (i - 1)*kmax)*sphi_b(1, 3 + s_offset_b1)
                     buffer2(i + imax*(1 - 1)) = buffer2(i + imax*(1 - 1)) + buffer1(2 + (i - 1)*kmax)*sphi_b(2, 1 + s_offset_b1)
                     buffer2(i + imax*(2 - 1)) = buffer2(i + imax*(2 - 1)) + buffer1(3 + (i - 1)*kmax)*sphi_b(3, 2 + s_offset_b1)
                  END DO
                  buffer1 = 0.0_dp
                  imax = 1*3*3
                  kmax = 10
                  DO i = 1, imax
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 7 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 1 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 3 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 4 + s_offset_c1)
                     buffer1(i + imax*(6 - 1)) = buffer1(i + imax*(6 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 6 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 7 + s_offset_c1)
                     buffer1(i + imax*(2 - 1)) = buffer1(i + imax*(2 - 1)) + buffer2(5 + (i - 1)*kmax)*sphi_c(5, 2 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(6 + (i - 1)*kmax)*sphi_c(6, 5 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(7 + (i - 1)*kmax)*sphi_c(7, 1 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(7 + (i - 1)*kmax)*sphi_c(7, 3 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(8 + (i - 1)*kmax)*sphi_c(8, 4 + s_offset_c1)
                     buffer1(i + imax*(6 - 1)) = buffer1(i + imax*(6 - 1)) + buffer2(8 + (i - 1)*kmax)*sphi_c(8, 6 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(9 + (i - 1)*kmax)*sphi_c(9, 3 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(10 + (i - 1)*kmax)*sphi_c(10, 4 + s_offset_c1)
                  END DO
                  imax = 1*3*7
                  kmax = 3
                  i = 0
                  DO i1 = 1, 7
                  DO i2 = 1, 3
                  DO i3 = 1, 1
                     i = i + 1
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) &
                        + buffer1(1 + (i - 1)*kmax)*sphi_d(1, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) &
                        + buffer1(2 + (i - 1)*kmax)*sphi_d(2, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) &
                        + buffer1(3 + (i - 1)*kmax)*sphi_d(3, 2 + s_offset_d1)
                  END DO
                  END DO
                  END DO
                  s_offset_d1 = s_offset_d1 + 3
               END DO
               s_offset_c1 = s_offset_c1 + 7
            END DO
            s_offset_b1 = s_offset_b1 + 3
         END DO
         s_offset_a1 = s_offset_a1 + 1
      END DO
   END SUBROUTINE contract_spfp
#endif
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
! **************************************************************************************************
!> \brief ...
!> \param work ...
!> \param nl_a ...
!> \param nl_b ...
!> \param nl_c ...
!> \param nl_d ...
!> \param sphi_a ...
!> \param sphi_b ...
!> \param sphi_c ...
!> \param sphi_d ...
!> \param primitives ...
!> \param buffer1 ...
!> \param buffer2 ...
! **************************************************************************************************
   SUBROUTINE contract_spfd(work, &
                            nl_a, nl_b, nl_c, nl_d, &
                            sphi_a, sphi_b, sphi_c, sphi_d, &
                            primitives, &
                            buffer1, buffer2)
      REAL(dp), DIMENSION(1*3*10*6), INTENT(IN)          :: work
      INTEGER                                            :: nl_a, nl_b, nl_c, nl_d
      REAL(dp), DIMENSION(1, 1*nl_a), INTENT(IN)         :: sphi_a
      REAL(dp), DIMENSION(3, 3*nl_b), INTENT(IN)         :: sphi_b
      REAL(dp), DIMENSION(10, 7*nl_c), INTENT(IN)        :: sphi_c
      REAL(dp), DIMENSION(6, 5*nl_d), INTENT(IN)         :: sphi_d
      REAL(dp), &
         DIMENSION(1*nl_a, 3*nl_b, 7*nl_c, 5*nl_d)       :: primitives
      REAL(dp), DIMENSION(1*3*10*6)                      :: buffer1, buffer2

      INTEGER                                            :: i, i1, i2, i3, ia, ib, ic, id, imax, &
                                                            kmax, s_offset_a1, s_offset_b1, &
                                                            s_offset_c1, s_offset_d1

      s_offset_a1 = 0
      DO ia = 1, nl_a
         s_offset_b1 = 0
         DO ib = 1, nl_b
            s_offset_c1 = 0
            DO ic = 1, nl_c
               s_offset_d1 = 0
               DO id = 1, nl_d
                  buffer1 = 0.0_dp
                  imax = 3*10*6
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + work(1 + (i - 1)*kmax)*sphi_a(1, 1 + s_offset_a1)
                  END DO
                  buffer2 = 0.0_dp
                  imax = 1*10*6
                  kmax = 3
                  DO i = 1, imax
                     buffer2(i + imax*(3 - 1)) = buffer2(i + imax*(3 - 1)) + buffer1(1 + (i - 1)*kmax)*sphi_b(1, 3 + s_offset_b1)
                     buffer2(i + imax*(1 - 1)) = buffer2(i + imax*(1 - 1)) + buffer1(2 + (i - 1)*kmax)*sphi_b(2, 1 + s_offset_b1)
                     buffer2(i + imax*(2 - 1)) = buffer2(i + imax*(2 - 1)) + buffer1(3 + (i - 1)*kmax)*sphi_b(3, 2 + s_offset_b1)
                  END DO
                  buffer1 = 0.0_dp
                  imax = 1*3*6
                  kmax = 10
                  DO i = 1, imax
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 7 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 1 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 3 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 4 + s_offset_c1)
                     buffer1(i + imax*(6 - 1)) = buffer1(i + imax*(6 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 6 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 7 + s_offset_c1)
                     buffer1(i + imax*(2 - 1)) = buffer1(i + imax*(2 - 1)) + buffer2(5 + (i - 1)*kmax)*sphi_c(5, 2 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(6 + (i - 1)*kmax)*sphi_c(6, 5 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(7 + (i - 1)*kmax)*sphi_c(7, 1 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(7 + (i - 1)*kmax)*sphi_c(7, 3 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(8 + (i - 1)*kmax)*sphi_c(8, 4 + s_offset_c1)
                     buffer1(i + imax*(6 - 1)) = buffer1(i + imax*(6 - 1)) + buffer2(8 + (i - 1)*kmax)*sphi_c(8, 6 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(9 + (i - 1)*kmax)*sphi_c(9, 3 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(10 + (i - 1)*kmax)*sphi_c(10, 4 + s_offset_c1)
                  END DO
                  imax = 1*3*7
                  kmax = 6
                  i = 0
                  DO i1 = 1, 7
                  DO i2 = 1, 3
                  DO i3 = 1, 1
                     i = i + 1
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) + &
                        buffer1(2 + (i - 1)*kmax)*sphi_d(2, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(3 + (i - 1)*kmax)*sphi_d(3, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) + &
                        buffer1(5 + (i - 1)*kmax)*sphi_d(5, 2 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(6 + (i - 1)*kmax)*sphi_d(6, 3 + s_offset_d1)
                  END DO
                  END DO
                  END DO
                  s_offset_d1 = s_offset_d1 + 5
               END DO
               s_offset_c1 = s_offset_c1 + 7
            END DO
            s_offset_b1 = s_offset_b1 + 3
         END DO
         s_offset_a1 = s_offset_a1 + 1
      END DO
   END SUBROUTINE contract_spfd
#endif
#if __MAX_CONTR > 3 || __MAX_CONTR == 3
! **************************************************************************************************
!> \brief ...
!> \param work ...
!> \param nl_a ...
!> \param nl_b ...
!> \param nl_c ...
!> \param nl_d ...
!> \param sphi_a ...
!> \param sphi_b ...
!> \param sphi_c ...
!> \param sphi_d ...
!> \param primitives ...
!> \param buffer1 ...
!> \param buffer2 ...
! **************************************************************************************************
   SUBROUTINE contract_spff(work, &
                            nl_a, nl_b, nl_c, nl_d, &
                            sphi_a, sphi_b, sphi_c, sphi_d, &
                            primitives, &
                            buffer1, buffer2)
      REAL(dp), DIMENSION(1*3*10*10), INTENT(IN)         :: work
      INTEGER                                            :: nl_a, nl_b, nl_c, nl_d
      REAL(dp), DIMENSION(1, 1*nl_a), INTENT(IN)         :: sphi_a
      REAL(dp), DIMENSION(3, 3*nl_b), INTENT(IN)         :: sphi_b
      REAL(dp), DIMENSION(10, 7*nl_c), INTENT(IN)        :: sphi_c
      REAL(dp), DIMENSION(10, 7*nl_d), INTENT(IN)        :: sphi_d
      REAL(dp), &
         DIMENSION(1*nl_a, 3*nl_b, 7*nl_c, 7*nl_d)       :: primitives
      REAL(dp), DIMENSION(1*3*10*10)                     :: buffer1, buffer2

      INTEGER                                            :: i, i1, i2, i3, ia, ib, ic, id, imax, &
                                                            kmax, s_offset_a1, s_offset_b1, &
                                                            s_offset_c1, s_offset_d1

      s_offset_a1 = 0
      DO ia = 1, nl_a
         s_offset_b1 = 0
         DO ib = 1, nl_b
            s_offset_c1 = 0
            DO ic = 1, nl_c
               s_offset_d1 = 0
               DO id = 1, nl_d
                  buffer1 = 0.0_dp
                  imax = 3*10*10
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + work(1 + (i - 1)*kmax)*sphi_a(1, 1 + s_offset_a1)
                  END DO
                  buffer2 = 0.0_dp
                  imax = 1*10*10
                  kmax = 3
                  DO i = 1, imax
                     buffer2(i + imax*(3 - 1)) = buffer2(i + imax*(3 - 1)) + buffer1(1 + (i - 1)*kmax)*sphi_b(1, 3 + s_offset_b1)
                     buffer2(i + imax*(1 - 1)) = buffer2(i + imax*(1 - 1)) + buffer1(2 + (i - 1)*kmax)*sphi_b(2, 1 + s_offset_b1)
                     buffer2(i + imax*(2 - 1)) = buffer2(i + imax*(2 - 1)) + buffer1(3 + (i - 1)*kmax)*sphi_b(3, 2 + s_offset_b1)
                  END DO
                  buffer1 = 0.0_dp
                  imax = 1*3*10
                  kmax = 10
                  DO i = 1, imax
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 7 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 1 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 3 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 4 + s_offset_c1)
                     buffer1(i + imax*(6 - 1)) = buffer1(i + imax*(6 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 6 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 7 + s_offset_c1)
                     buffer1(i + imax*(2 - 1)) = buffer1(i + imax*(2 - 1)) + buffer2(5 + (i - 1)*kmax)*sphi_c(5, 2 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(6 + (i - 1)*kmax)*sphi_c(6, 5 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(7 + (i - 1)*kmax)*sphi_c(7, 1 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(7 + (i - 1)*kmax)*sphi_c(7, 3 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(8 + (i - 1)*kmax)*sphi_c(8, 4 + s_offset_c1)
                     buffer1(i + imax*(6 - 1)) = buffer1(i + imax*(6 - 1)) + buffer2(8 + (i - 1)*kmax)*sphi_c(8, 6 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(9 + (i - 1)*kmax)*sphi_c(9, 3 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(10 + (i - 1)*kmax)*sphi_c(10, 4 + s_offset_c1)
                  END DO
                  imax = 1*3*7
                  kmax = 10
                  i = 0
                  DO i1 = 1, 7
                  DO i2 = 1, 3
                  DO i3 = 1, 1
                     i = i + 1
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) + &
                        buffer1(2 + (i - 1)*kmax)*sphi_d(2, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(2 + (i - 1)*kmax)*sphi_d(2, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(3 + (i - 1)*kmax)*sphi_d(3, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) + &
                        buffer1(3 + (i - 1)*kmax)*sphi_d(3, 6 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) + &
                        buffer1(5 + (i - 1)*kmax)*sphi_d(5, 2 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(6 + (i - 1)*kmax)*sphi_d(6, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) + &
                        buffer1(7 + (i - 1)*kmax)*sphi_d(7, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(7 + (i - 1)*kmax)*sphi_d(7, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(8 + (i - 1)*kmax)*sphi_d(8, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) + &
                        buffer1(8 + (i - 1)*kmax)*sphi_d(8, 6 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(9 + (i - 1)*kmax)*sphi_d(9, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(10 + (i - 1)*kmax)*sphi_d(10, 4 + s_offset_d1)
                  END DO
                  END DO
                  END DO
                  s_offset_d1 = s_offset_d1 + 7
               END DO
               s_offset_c1 = s_offset_c1 + 7
            END DO
            s_offset_b1 = s_offset_b1 + 3
         END DO
         s_offset_a1 = s_offset_a1 + 1
      END DO
   END SUBROUTINE contract_spff
#endif
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
! **************************************************************************************************
!> \brief ...
!> \param work ...
!> \param nl_a ...
!> \param nl_b ...
!> \param nl_c ...
!> \param nl_d ...
!> \param sphi_a ...
!> \param sphi_b ...
!> \param sphi_c ...
!> \param sphi_d ...
!> \param primitives ...
!> \param buffer1 ...
!> \param buffer2 ...
! **************************************************************************************************
   SUBROUTINE contract_spfg(work, &
                            nl_a, nl_b, nl_c, nl_d, &
                            sphi_a, sphi_b, sphi_c, sphi_d, &
                            primitives, &
                            buffer1, buffer2)
      REAL(dp), DIMENSION(1*3*10*15), INTENT(IN)         :: work
      INTEGER                                            :: nl_a, nl_b, nl_c, nl_d
      REAL(dp), DIMENSION(1, 1*nl_a), INTENT(IN)         :: sphi_a
      REAL(dp), DIMENSION(3, 3*nl_b), INTENT(IN)         :: sphi_b
      REAL(dp), DIMENSION(10, 7*nl_c), INTENT(IN)        :: sphi_c
      REAL(dp), DIMENSION(15, 9*nl_d), INTENT(IN)        :: sphi_d
      REAL(dp), &
         DIMENSION(1*nl_a, 3*nl_b, 7*nl_c, 9*nl_d)       :: primitives
      REAL(dp), DIMENSION(1*3*10*15)                     :: buffer1, buffer2

      INTEGER                                            :: i, i1, i2, i3, ia, ib, ic, id, imax, &
                                                            kmax, s_offset_a1, s_offset_b1, &
                                                            s_offset_c1, s_offset_d1

      s_offset_a1 = 0
      DO ia = 1, nl_a
         s_offset_b1 = 0
         DO ib = 1, nl_b
            s_offset_c1 = 0
            DO ic = 1, nl_c
               s_offset_d1 = 0
               DO id = 1, nl_d
                  buffer1 = 0.0_dp
                  imax = 3*10*15
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + work(1 + (i - 1)*kmax)*sphi_a(1, 1 + s_offset_a1)
                  END DO
                  buffer2 = 0.0_dp
                  imax = 1*10*15
                  kmax = 3
                  DO i = 1, imax
                     buffer2(i + imax*(3 - 1)) = buffer2(i + imax*(3 - 1)) + buffer1(1 + (i - 1)*kmax)*sphi_b(1, 3 + s_offset_b1)
                     buffer2(i + imax*(1 - 1)) = buffer2(i + imax*(1 - 1)) + buffer1(2 + (i - 1)*kmax)*sphi_b(2, 1 + s_offset_b1)
                     buffer2(i + imax*(2 - 1)) = buffer2(i + imax*(2 - 1)) + buffer1(3 + (i - 1)*kmax)*sphi_b(3, 2 + s_offset_b1)
                  END DO
                  buffer1 = 0.0_dp
                  imax = 1*3*15
                  kmax = 10
                  DO i = 1, imax
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 7 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 1 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 3 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 4 + s_offset_c1)
                     buffer1(i + imax*(6 - 1)) = buffer1(i + imax*(6 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 6 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 7 + s_offset_c1)
                     buffer1(i + imax*(2 - 1)) = buffer1(i + imax*(2 - 1)) + buffer2(5 + (i - 1)*kmax)*sphi_c(5, 2 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(6 + (i - 1)*kmax)*sphi_c(6, 5 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(7 + (i - 1)*kmax)*sphi_c(7, 1 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(7 + (i - 1)*kmax)*sphi_c(7, 3 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(8 + (i - 1)*kmax)*sphi_c(8, 4 + s_offset_c1)
                     buffer1(i + imax*(6 - 1)) = buffer1(i + imax*(6 - 1)) + buffer2(8 + (i - 1)*kmax)*sphi_c(8, 6 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(9 + (i - 1)*kmax)*sphi_c(9, 3 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(10 + (i - 1)*kmax)*sphi_c(10, 4 + s_offset_c1)
                  END DO
                  imax = 1*3*7
                  kmax = 15
                  i = 0
                  DO i1 = 1, 7
                  DO i2 = 1, 3
                  DO i3 = 1, 1
                     i = i + 1
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 9 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) + &
                        buffer1(2 + (i - 1)*kmax)*sphi_d(2, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(2 + (i - 1)*kmax)*sphi_d(2, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) + &
                        buffer1(3 + (i - 1)*kmax)*sphi_d(3, 6 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 8) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 8) + &
                        buffer1(3 + (i - 1)*kmax)*sphi_d(3, 8 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 9 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) + &
                        buffer1(5 + (i - 1)*kmax)*sphi_d(5, 2 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(5 + (i - 1)*kmax)*sphi_d(5, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(6 + (i - 1)*kmax)*sphi_d(6, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(6 + (i - 1)*kmax)*sphi_d(6, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) + &
                        buffer1(7 + (i - 1)*kmax)*sphi_d(7, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(7 + (i - 1)*kmax)*sphi_d(7, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) + &
                        buffer1(8 + (i - 1)*kmax)*sphi_d(8, 6 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 8) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 8) + &
                        buffer1(8 + (i - 1)*kmax)*sphi_d(8, 8 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(9 + (i - 1)*kmax)*sphi_d(9, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) + &
                        buffer1(10 + (i - 1)*kmax)*sphi_d(10, 6 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(11 + (i - 1)*kmax)*sphi_d(11, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(11 + (i - 1)*kmax)*sphi_d(11, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 9) + &
                        buffer1(11 + (i - 1)*kmax)*sphi_d(11, 9 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) + &
                        buffer1(12 + (i - 1)*kmax)*sphi_d(12, 2 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(12 + (i - 1)*kmax)*sphi_d(12, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(13 + (i - 1)*kmax)*sphi_d(13, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(13 + (i - 1)*kmax)*sphi_d(13, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(14 + (i - 1)*kmax)*sphi_d(14, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(15 + (i - 1)*kmax)*sphi_d(15, 5 + s_offset_d1)
                  END DO
                  END DO
                  END DO
                  s_offset_d1 = s_offset_d1 + 9
               END DO
               s_offset_c1 = s_offset_c1 + 7
            END DO
            s_offset_b1 = s_offset_b1 + 3
         END DO
         s_offset_a1 = s_offset_a1 + 1
      END DO
   END SUBROUTINE contract_spfg
#endif
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
! **************************************************************************************************
!> \brief ...
!> \param work ...
!> \param nl_a ...
!> \param nl_b ...
!> \param nl_c ...
!> \param nl_d ...
!> \param sphi_a ...
!> \param sphi_b ...
!> \param sphi_c ...
!> \param sphi_d ...
!> \param primitives ...
!> \param buffer1 ...
!> \param buffer2 ...
! **************************************************************************************************
   SUBROUTINE contract_spgs(work, &
                            nl_a, nl_b, nl_c, nl_d, &
                            sphi_a, sphi_b, sphi_c, sphi_d, &
                            primitives, &
                            buffer1, buffer2)
      REAL(dp), DIMENSION(1*3*15*1), INTENT(IN)          :: work
      INTEGER                                            :: nl_a, nl_b, nl_c, nl_d
      REAL(dp), DIMENSION(1, 1*nl_a), INTENT(IN)         :: sphi_a
      REAL(dp), DIMENSION(3, 3*nl_b), INTENT(IN)         :: sphi_b
      REAL(dp), DIMENSION(15, 9*nl_c), INTENT(IN)        :: sphi_c
      REAL(dp), DIMENSION(1, 1*nl_d), INTENT(IN)         :: sphi_d
      REAL(dp), &
         DIMENSION(1*nl_a, 3*nl_b, 9*nl_c, 1*nl_d)       :: primitives
      REAL(dp), DIMENSION(1*3*15*1)                      :: buffer1, buffer2

      INTEGER                                            :: i, i1, i2, i3, ia, ib, ic, id, imax, &
                                                            kmax, s_offset_a1, s_offset_b1, &
                                                            s_offset_c1, s_offset_d1

      s_offset_a1 = 0
      DO ia = 1, nl_a
         s_offset_b1 = 0
         DO ib = 1, nl_b
            s_offset_c1 = 0
            DO ic = 1, nl_c
               s_offset_d1 = 0
               DO id = 1, nl_d
                  buffer1 = 0.0_dp
                  imax = 3*15*1
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + work(1 + (i - 1)*kmax)*sphi_a(1, 1 + s_offset_a1)
                  END DO
                  buffer2 = 0.0_dp
                  imax = 1*15*1
                  kmax = 3
                  DO i = 1, imax
                     buffer2(i + imax*(3 - 1)) = buffer2(i + imax*(3 - 1)) + buffer1(1 + (i - 1)*kmax)*sphi_b(1, 3 + s_offset_b1)
                     buffer2(i + imax*(1 - 1)) = buffer2(i + imax*(1 - 1)) + buffer1(2 + (i - 1)*kmax)*sphi_b(2, 1 + s_offset_b1)
                     buffer2(i + imax*(2 - 1)) = buffer2(i + imax*(2 - 1)) + buffer1(3 + (i - 1)*kmax)*sphi_b(3, 2 + s_offset_b1)
                  END DO
                  buffer1 = 0.0_dp
                  imax = 1*3*1
                  kmax = 15
                  DO i = 1, imax
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 7 + s_offset_c1)
                     buffer1(i + imax*(9 - 1)) = buffer1(i + imax*(9 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 9 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 1 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 3 + s_offset_c1)
                     buffer1(i + imax*(6 - 1)) = buffer1(i + imax*(6 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 6 + s_offset_c1)
                     buffer1(i + imax*(8 - 1)) = buffer1(i + imax*(8 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 8 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 5 + s_offset_c1)
                     buffer1(i + imax*(9 - 1)) = buffer1(i + imax*(9 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 9 + s_offset_c1)
                     buffer1(i + imax*(2 - 1)) = buffer1(i + imax*(2 - 1)) + buffer2(5 + (i - 1)*kmax)*sphi_c(5, 2 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(5 + (i - 1)*kmax)*sphi_c(5, 4 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(6 + (i - 1)*kmax)*sphi_c(6, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(6 + (i - 1)*kmax)*sphi_c(6, 7 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(7 + (i - 1)*kmax)*sphi_c(7, 1 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(7 + (i - 1)*kmax)*sphi_c(7, 3 + s_offset_c1)
                     buffer1(i + imax*(6 - 1)) = buffer1(i + imax*(6 - 1)) + buffer2(8 + (i - 1)*kmax)*sphi_c(8, 6 + s_offset_c1)
                     buffer1(i + imax*(8 - 1)) = buffer1(i + imax*(8 - 1)) + buffer2(8 + (i - 1)*kmax)*sphi_c(8, 8 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(9 + (i - 1)*kmax)*sphi_c(9, 3 + s_offset_c1)
                     buffer1(i + imax*(6 - 1)) = buffer1(i + imax*(6 - 1)) + buffer2(10 + (i - 1)*kmax)*sphi_c(10, 6 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(11 + (i - 1)*kmax)*sphi_c(11, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(11 + (i - 1)*kmax)*sphi_c(11, 7 + s_offset_c1)
                     buffer1(i + imax*(9 - 1)) = buffer1(i + imax*(9 - 1)) + buffer2(11 + (i - 1)*kmax)*sphi_c(11, 9 + s_offset_c1)
                     buffer1(i + imax*(2 - 1)) = buffer1(i + imax*(2 - 1)) + buffer2(12 + (i - 1)*kmax)*sphi_c(12, 2 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(12 + (i - 1)*kmax)*sphi_c(12, 4 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(13 + (i - 1)*kmax)*sphi_c(13, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(13 + (i - 1)*kmax)*sphi_c(13, 7 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(14 + (i - 1)*kmax)*sphi_c(14, 4 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(15 + (i - 1)*kmax)*sphi_c(15, 5 + s_offset_c1)
                  END DO
                  imax = 1*3*9
                  kmax = 1
                  i = 0
                  DO i1 = 1, 9
                  DO i2 = 1, 3
                  DO i3 = 1, 1
                     i = i + 1
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) &
                        + buffer1(1 + (i - 1)*kmax)*sphi_d(1, 1 + s_offset_d1)
                  END DO
                  END DO
                  END DO
                  s_offset_d1 = s_offset_d1 + 1
               END DO
               s_offset_c1 = s_offset_c1 + 9
            END DO
            s_offset_b1 = s_offset_b1 + 3
         END DO
         s_offset_a1 = s_offset_a1 + 1
      END DO
   END SUBROUTINE contract_spgs
#endif
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
! **************************************************************************************************
!> \brief ...
!> \param work ...
!> \param nl_a ...
!> \param nl_b ...
!> \param nl_c ...
!> \param nl_d ...
!> \param sphi_a ...
!> \param sphi_b ...
!> \param sphi_c ...
!> \param sphi_d ...
!> \param primitives ...
!> \param buffer1 ...
!> \param buffer2 ...
! **************************************************************************************************
   SUBROUTINE contract_spgp(work, &
                            nl_a, nl_b, nl_c, nl_d, &
                            sphi_a, sphi_b, sphi_c, sphi_d, &
                            primitives, &
                            buffer1, buffer2)
      REAL(dp), DIMENSION(1*3*15*3), INTENT(IN)          :: work
      INTEGER                                            :: nl_a, nl_b, nl_c, nl_d
      REAL(dp), DIMENSION(1, 1*nl_a), INTENT(IN)         :: sphi_a
      REAL(dp), DIMENSION(3, 3*nl_b), INTENT(IN)         :: sphi_b
      REAL(dp), DIMENSION(15, 9*nl_c), INTENT(IN)        :: sphi_c
      REAL(dp), DIMENSION(3, 3*nl_d), INTENT(IN)         :: sphi_d
      REAL(dp), &
         DIMENSION(1*nl_a, 3*nl_b, 9*nl_c, 3*nl_d)       :: primitives
      REAL(dp), DIMENSION(1*3*15*3)                      :: buffer1, buffer2

      INTEGER                                            :: i, i1, i2, i3, ia, ib, ic, id, imax, &
                                                            kmax, s_offset_a1, s_offset_b1, &
                                                            s_offset_c1, s_offset_d1

      s_offset_a1 = 0
      DO ia = 1, nl_a
         s_offset_b1 = 0
         DO ib = 1, nl_b
            s_offset_c1 = 0
            DO ic = 1, nl_c
               s_offset_d1 = 0
               DO id = 1, nl_d
                  buffer1 = 0.0_dp
                  imax = 3*15*3
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + work(1 + (i - 1)*kmax)*sphi_a(1, 1 + s_offset_a1)
                  END DO
                  buffer2 = 0.0_dp
                  imax = 1*15*3
                  kmax = 3
                  DO i = 1, imax
                     buffer2(i + imax*(3 - 1)) = buffer2(i + imax*(3 - 1)) + buffer1(1 + (i - 1)*kmax)*sphi_b(1, 3 + s_offset_b1)
                     buffer2(i + imax*(1 - 1)) = buffer2(i + imax*(1 - 1)) + buffer1(2 + (i - 1)*kmax)*sphi_b(2, 1 + s_offset_b1)
                     buffer2(i + imax*(2 - 1)) = buffer2(i + imax*(2 - 1)) + buffer1(3 + (i - 1)*kmax)*sphi_b(3, 2 + s_offset_b1)
                  END DO
                  buffer1 = 0.0_dp
                  imax = 1*3*3
                  kmax = 15
                  DO i = 1, imax
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 7 + s_offset_c1)
                     buffer1(i + imax*(9 - 1)) = buffer1(i + imax*(9 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 9 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 1 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 3 + s_offset_c1)
                     buffer1(i + imax*(6 - 1)) = buffer1(i + imax*(6 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 6 + s_offset_c1)
                     buffer1(i + imax*(8 - 1)) = buffer1(i + imax*(8 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 8 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 5 + s_offset_c1)
                     buffer1(i + imax*(9 - 1)) = buffer1(i + imax*(9 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 9 + s_offset_c1)
                     buffer1(i + imax*(2 - 1)) = buffer1(i + imax*(2 - 1)) + buffer2(5 + (i - 1)*kmax)*sphi_c(5, 2 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(5 + (i - 1)*kmax)*sphi_c(5, 4 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(6 + (i - 1)*kmax)*sphi_c(6, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(6 + (i - 1)*kmax)*sphi_c(6, 7 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(7 + (i - 1)*kmax)*sphi_c(7, 1 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(7 + (i - 1)*kmax)*sphi_c(7, 3 + s_offset_c1)
                     buffer1(i + imax*(6 - 1)) = buffer1(i + imax*(6 - 1)) + buffer2(8 + (i - 1)*kmax)*sphi_c(8, 6 + s_offset_c1)
                     buffer1(i + imax*(8 - 1)) = buffer1(i + imax*(8 - 1)) + buffer2(8 + (i - 1)*kmax)*sphi_c(8, 8 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(9 + (i - 1)*kmax)*sphi_c(9, 3 + s_offset_c1)
                     buffer1(i + imax*(6 - 1)) = buffer1(i + imax*(6 - 1)) + buffer2(10 + (i - 1)*kmax)*sphi_c(10, 6 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(11 + (i - 1)*kmax)*sphi_c(11, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(11 + (i - 1)*kmax)*sphi_c(11, 7 + s_offset_c1)
                     buffer1(i + imax*(9 - 1)) = buffer1(i + imax*(9 - 1)) + buffer2(11 + (i - 1)*kmax)*sphi_c(11, 9 + s_offset_c1)
                     buffer1(i + imax*(2 - 1)) = buffer1(i + imax*(2 - 1)) + buffer2(12 + (i - 1)*kmax)*sphi_c(12, 2 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(12 + (i - 1)*kmax)*sphi_c(12, 4 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(13 + (i - 1)*kmax)*sphi_c(13, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(13 + (i - 1)*kmax)*sphi_c(13, 7 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(14 + (i - 1)*kmax)*sphi_c(14, 4 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(15 + (i - 1)*kmax)*sphi_c(15, 5 + s_offset_c1)
                  END DO
                  imax = 1*3*9
                  kmax = 3
                  i = 0
                  DO i1 = 1, 9
                  DO i2 = 1, 3
                  DO i3 = 1, 1
                     i = i + 1
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) &
                        + buffer1(1 + (i - 1)*kmax)*sphi_d(1, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) &
                        + buffer1(2 + (i - 1)*kmax)*sphi_d(2, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) &
                        + buffer1(3 + (i - 1)*kmax)*sphi_d(3, 2 + s_offset_d1)
                  END DO
                  END DO
                  END DO
                  s_offset_d1 = s_offset_d1 + 3
               END DO
               s_offset_c1 = s_offset_c1 + 9
            END DO
            s_offset_b1 = s_offset_b1 + 3
         END DO
         s_offset_a1 = s_offset_a1 + 1
      END DO
   END SUBROUTINE contract_spgp
#endif
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
! **************************************************************************************************
!> \brief ...
!> \param work ...
!> \param nl_a ...
!> \param nl_b ...
!> \param nl_c ...
!> \param nl_d ...
!> \param sphi_a ...
!> \param sphi_b ...
!> \param sphi_c ...
!> \param sphi_d ...
!> \param primitives ...
!> \param buffer1 ...
!> \param buffer2 ...
! **************************************************************************************************
   SUBROUTINE contract_spgd(work, &
                            nl_a, nl_b, nl_c, nl_d, &
                            sphi_a, sphi_b, sphi_c, sphi_d, &
                            primitives, &
                            buffer1, buffer2)
      REAL(dp), DIMENSION(1*3*15*6), INTENT(IN)          :: work
      INTEGER                                            :: nl_a, nl_b, nl_c, nl_d
      REAL(dp), DIMENSION(1, 1*nl_a), INTENT(IN)         :: sphi_a
      REAL(dp), DIMENSION(3, 3*nl_b), INTENT(IN)         :: sphi_b
      REAL(dp), DIMENSION(15, 9*nl_c), INTENT(IN)        :: sphi_c
      REAL(dp), DIMENSION(6, 5*nl_d), INTENT(IN)         :: sphi_d
      REAL(dp), &
         DIMENSION(1*nl_a, 3*nl_b, 9*nl_c, 5*nl_d)       :: primitives
      REAL(dp), DIMENSION(1*3*15*6)                      :: buffer1, buffer2

      INTEGER                                            :: i, i1, i2, i3, ia, ib, ic, id, imax, &
                                                            kmax, s_offset_a1, s_offset_b1, &
                                                            s_offset_c1, s_offset_d1

      s_offset_a1 = 0
      DO ia = 1, nl_a
         s_offset_b1 = 0
         DO ib = 1, nl_b
            s_offset_c1 = 0
            DO ic = 1, nl_c
               s_offset_d1 = 0
               DO id = 1, nl_d
                  buffer1 = 0.0_dp
                  imax = 3*15*6
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + work(1 + (i - 1)*kmax)*sphi_a(1, 1 + s_offset_a1)
                  END DO
                  buffer2 = 0.0_dp
                  imax = 1*15*6
                  kmax = 3
                  DO i = 1, imax
                     buffer2(i + imax*(3 - 1)) = buffer2(i + imax*(3 - 1)) + buffer1(1 + (i - 1)*kmax)*sphi_b(1, 3 + s_offset_b1)
                     buffer2(i + imax*(1 - 1)) = buffer2(i + imax*(1 - 1)) + buffer1(2 + (i - 1)*kmax)*sphi_b(2, 1 + s_offset_b1)
                     buffer2(i + imax*(2 - 1)) = buffer2(i + imax*(2 - 1)) + buffer1(3 + (i - 1)*kmax)*sphi_b(3, 2 + s_offset_b1)
                  END DO
                  buffer1 = 0.0_dp
                  imax = 1*3*6
                  kmax = 15
                  DO i = 1, imax
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 7 + s_offset_c1)
                     buffer1(i + imax*(9 - 1)) = buffer1(i + imax*(9 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 9 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 1 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 3 + s_offset_c1)
                     buffer1(i + imax*(6 - 1)) = buffer1(i + imax*(6 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 6 + s_offset_c1)
                     buffer1(i + imax*(8 - 1)) = buffer1(i + imax*(8 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 8 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 5 + s_offset_c1)
                     buffer1(i + imax*(9 - 1)) = buffer1(i + imax*(9 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 9 + s_offset_c1)
                     buffer1(i + imax*(2 - 1)) = buffer1(i + imax*(2 - 1)) + buffer2(5 + (i - 1)*kmax)*sphi_c(5, 2 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(5 + (i - 1)*kmax)*sphi_c(5, 4 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(6 + (i - 1)*kmax)*sphi_c(6, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(6 + (i - 1)*kmax)*sphi_c(6, 7 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(7 + (i - 1)*kmax)*sphi_c(7, 1 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(7 + (i - 1)*kmax)*sphi_c(7, 3 + s_offset_c1)
                     buffer1(i + imax*(6 - 1)) = buffer1(i + imax*(6 - 1)) + buffer2(8 + (i - 1)*kmax)*sphi_c(8, 6 + s_offset_c1)
                     buffer1(i + imax*(8 - 1)) = buffer1(i + imax*(8 - 1)) + buffer2(8 + (i - 1)*kmax)*sphi_c(8, 8 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(9 + (i - 1)*kmax)*sphi_c(9, 3 + s_offset_c1)
                     buffer1(i + imax*(6 - 1)) = buffer1(i + imax*(6 - 1)) + buffer2(10 + (i - 1)*kmax)*sphi_c(10, 6 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(11 + (i - 1)*kmax)*sphi_c(11, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(11 + (i - 1)*kmax)*sphi_c(11, 7 + s_offset_c1)
                     buffer1(i + imax*(9 - 1)) = buffer1(i + imax*(9 - 1)) + buffer2(11 + (i - 1)*kmax)*sphi_c(11, 9 + s_offset_c1)
                     buffer1(i + imax*(2 - 1)) = buffer1(i + imax*(2 - 1)) + buffer2(12 + (i - 1)*kmax)*sphi_c(12, 2 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(12 + (i - 1)*kmax)*sphi_c(12, 4 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(13 + (i - 1)*kmax)*sphi_c(13, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(13 + (i - 1)*kmax)*sphi_c(13, 7 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(14 + (i - 1)*kmax)*sphi_c(14, 4 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(15 + (i - 1)*kmax)*sphi_c(15, 5 + s_offset_c1)
                  END DO
                  imax = 1*3*9
                  kmax = 6
                  i = 0
                  DO i1 = 1, 9
                  DO i2 = 1, 3
                  DO i3 = 1, 1
                     i = i + 1
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) + &
                        buffer1(2 + (i - 1)*kmax)*sphi_d(2, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(3 + (i - 1)*kmax)*sphi_d(3, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) + &
                        buffer1(5 + (i - 1)*kmax)*sphi_d(5, 2 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(6 + (i - 1)*kmax)*sphi_d(6, 3 + s_offset_d1)
                  END DO
                  END DO
                  END DO
                  s_offset_d1 = s_offset_d1 + 5
               END DO
               s_offset_c1 = s_offset_c1 + 9
            END DO
            s_offset_b1 = s_offset_b1 + 3
         END DO
         s_offset_a1 = s_offset_a1 + 1
      END DO
   END SUBROUTINE contract_spgd
#endif
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
! **************************************************************************************************
!> \brief ...
!> \param work ...
!> \param nl_a ...
!> \param nl_b ...
!> \param nl_c ...
!> \param nl_d ...
!> \param sphi_a ...
!> \param sphi_b ...
!> \param sphi_c ...
!> \param sphi_d ...
!> \param primitives ...
!> \param buffer1 ...
!> \param buffer2 ...
! **************************************************************************************************
   SUBROUTINE contract_spgf(work, &
                            nl_a, nl_b, nl_c, nl_d, &
                            sphi_a, sphi_b, sphi_c, sphi_d, &
                            primitives, &
                            buffer1, buffer2)
      REAL(dp), DIMENSION(1*3*15*10), INTENT(IN)         :: work
      INTEGER                                            :: nl_a, nl_b, nl_c, nl_d
      REAL(dp), DIMENSION(1, 1*nl_a), INTENT(IN)         :: sphi_a
      REAL(dp), DIMENSION(3, 3*nl_b), INTENT(IN)         :: sphi_b
      REAL(dp), DIMENSION(15, 9*nl_c), INTENT(IN)        :: sphi_c
      REAL(dp), DIMENSION(10, 7*nl_d), INTENT(IN)        :: sphi_d
      REAL(dp), &
         DIMENSION(1*nl_a, 3*nl_b, 9*nl_c, 7*nl_d)       :: primitives
      REAL(dp), DIMENSION(1*3*15*10)                     :: buffer1, buffer2

      INTEGER                                            :: i, i1, i2, i3, ia, ib, ic, id, imax, &
                                                            kmax, s_offset_a1, s_offset_b1, &
                                                            s_offset_c1, s_offset_d1

      s_offset_a1 = 0
      DO ia = 1, nl_a
         s_offset_b1 = 0
         DO ib = 1, nl_b
            s_offset_c1 = 0
            DO ic = 1, nl_c
               s_offset_d1 = 0
               DO id = 1, nl_d
                  buffer1 = 0.0_dp
                  imax = 3*15*10
                  kmax = 1
                  DO i = 1, imax
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + work(1 + (i - 1)*kmax)*sphi_a(1, 1 + s_offset_a1)
                  END DO
                  buffer2 = 0.0_dp
                  imax = 1*15*10
                  kmax = 3
                  DO i = 1, imax
                     buffer2(i + imax*(3 - 1)) = buffer2(i + imax*(3 - 1)) + buffer1(1 + (i - 1)*kmax)*sphi_b(1, 3 + s_offset_b1)
                     buffer2(i + imax*(1 - 1)) = buffer2(i + imax*(1 - 1)) + buffer1(2 + (i - 1)*kmax)*sphi_b(2, 1 + s_offset_b1)
                     buffer2(i + imax*(2 - 1)) = buffer2(i + imax*(2 - 1)) + buffer1(3 + (i - 1)*kmax)*sphi_b(3, 2 + s_offset_b1)
                  END DO
                  buffer1 = 0.0_dp
                  imax = 1*3*10
                  kmax = 15
                  DO i = 1, imax
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 7 + s_offset_c1)
                     buffer1(i + imax*(9 - 1)) = buffer1(i + imax*(9 - 1)) + buffer2(1 + (i - 1)*kmax)*sphi_c(1, 9 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 1 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(2 + (i - 1)*kmax)*sphi_c(2, 3 + s_offset_c1)
                     buffer1(i + imax*(6 - 1)) = buffer1(i + imax*(6 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 6 + s_offset_c1)
                     buffer1(i + imax*(8 - 1)) = buffer1(i + imax*(8 - 1)) + buffer2(3 + (i - 1)*kmax)*sphi_c(3, 8 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 5 + s_offset_c1)
                     buffer1(i + imax*(9 - 1)) = buffer1(i + imax*(9 - 1)) + buffer2(4 + (i - 1)*kmax)*sphi_c(4, 9 + s_offset_c1)
                     buffer1(i + imax*(2 - 1)) = buffer1(i + imax*(2 - 1)) + buffer2(5 + (i - 1)*kmax)*sphi_c(5, 2 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(5 + (i - 1)*kmax)*sphi_c(5, 4 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(6 + (i - 1)*kmax)*sphi_c(6, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(6 + (i - 1)*kmax)*sphi_c(6, 7 + s_offset_c1)
                     buffer1(i + imax*(1 - 1)) = buffer1(i + imax*(1 - 1)) + buffer2(7 + (i - 1)*kmax)*sphi_c(7, 1 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(7 + (i - 1)*kmax)*sphi_c(7, 3 + s_offset_c1)
                     buffer1(i + imax*(6 - 1)) = buffer1(i + imax*(6 - 1)) + buffer2(8 + (i - 1)*kmax)*sphi_c(8, 6 + s_offset_c1)
                     buffer1(i + imax*(8 - 1)) = buffer1(i + imax*(8 - 1)) + buffer2(8 + (i - 1)*kmax)*sphi_c(8, 8 + s_offset_c1)
                     buffer1(i + imax*(3 - 1)) = buffer1(i + imax*(3 - 1)) + buffer2(9 + (i - 1)*kmax)*sphi_c(9, 3 + s_offset_c1)
                     buffer1(i + imax*(6 - 1)) = buffer1(i + imax*(6 - 1)) + buffer2(10 + (i - 1)*kmax)*sphi_c(10, 6 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(11 + (i - 1)*kmax)*sphi_c(11, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(11 + (i - 1)*kmax)*sphi_c(11, 7 + s_offset_c1)
                     buffer1(i + imax*(9 - 1)) = buffer1(i + imax*(9 - 1)) + buffer2(11 + (i - 1)*kmax)*sphi_c(11, 9 + s_offset_c1)
                     buffer1(i + imax*(2 - 1)) = buffer1(i + imax*(2 - 1)) + buffer2(12 + (i - 1)*kmax)*sphi_c(12, 2 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(12 + (i - 1)*kmax)*sphi_c(12, 4 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(13 + (i - 1)*kmax)*sphi_c(13, 5 + s_offset_c1)
                     buffer1(i + imax*(7 - 1)) = buffer1(i + imax*(7 - 1)) + buffer2(13 + (i - 1)*kmax)*sphi_c(13, 7 + s_offset_c1)
                     buffer1(i + imax*(4 - 1)) = buffer1(i + imax*(4 - 1)) + buffer2(14 + (i - 1)*kmax)*sphi_c(14, 4 + s_offset_c1)
                     buffer1(i + imax*(5 - 1)) = buffer1(i + imax*(5 - 1)) + buffer2(15 + (i - 1)*kmax)*sphi_c(15, 5 + s_offset_c1)
                  END DO
                  imax = 1*3*9
                  kmax = 10
                  i = 0
                  DO i1 = 1, 9
                  DO i2 = 1, 3
                  DO i3 = 1, 1
                     i = i + 1
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(1 + (i - 1)*kmax)*sphi_d(1, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) + &
                        buffer1(2 + (i - 1)*kmax)*sphi_d(2, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(2 + (i - 1)*kmax)*sphi_d(2, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(3 + (i - 1)*kmax)*sphi_d(3, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) + &
                        buffer1(3 + (i - 1)*kmax)*sphi_d(3, 6 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 7) + &
                        buffer1(4 + (i - 1)*kmax)*sphi_d(4, 7 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 2) + &
                        buffer1(5 + (i - 1)*kmax)*sphi_d(5, 2 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 5) + &
                        buffer1(6 + (i - 1)*kmax)*sphi_d(6, 5 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 1) + &
                        buffer1(7 + (i - 1)*kmax)*sphi_d(7, 1 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(7 + (i - 1)*kmax)*sphi_d(7, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(8 + (i - 1)*kmax)*sphi_d(8, 4 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 6) + &
                        buffer1(8 + (i - 1)*kmax)*sphi_d(8, 6 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 3) + &
                        buffer1(9 + (i - 1)*kmax)*sphi_d(9, 3 + s_offset_d1)
                     primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) = &
                        primitives(s_offset_a1 + i3, s_offset_b1 + i2, s_offset_c1 + i1, s_offset_d1 + 4) + &
                        buffer1(10 + (i - 1)*kmax)*sphi_d(10, 4 + s_offset_d1)
                  END DO
                  END DO
                  END DO
                  s_offset_d1 = s_offset_d1 + 7
               END DO
               s_offset_c1 = s_offset_c1 + 9
            END DO
            s_offset_b1 = s_offset_b1 + 3
         END DO
         s_offset_a1 = s_offset_a1 + 1
      END DO
   END SUBROUTINE contract_spgf
#endif
#if __MAX_CONTR > 4 || __MAX_CONTR == 4
! **************************************************************************************************
!> \brief ...
!> \param work ...
!> \param nl_a ...
!> \param nl_b ...
!> \param nl_c 