// ---------------------------------------------------------------------------
// - Lstack.hpp                                                              -
// - afnix:sps module - literal stack class definition                      -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2015 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  AFNIX_LSTACK_HPP
#define  AFNIX_LSTACK_HPP

#ifndef  AFNIX_STRING_HPP
#include "String.hpp"
#endif

#ifndef  AFNIX_ITERABLE_HPP
#include "Iterable.hpp"
#endif

namespace afnix {

  /// The Lstack class implements a literal stack facility which is optimized
  /// for a set of standard literals, namely the boolean, integer and real
  /// types. This implementation is therefore faster and designed to store
  /// larger dataset as compared with a vector implementation. As a literal,
  /// the implementation is designed to operate with a resolved value pointed
  /// by a stack index.
  /// @author amaury darsch

  class Lstack : public Iterable, public Literal {
  public:
    enum t_lstk {
      LSTK_BOOL, // boolean type
      LSTK_INTG, // integer type
      LSTK_REAL, // real type
      LSTK_OTHR  // other type
    };
  protected:
    /// the stack size
    long   d_size;
    /// the stack length
    long   d_slen;
    /// the literal index
    long   d_lidx;
    /// the stack type
    t_lstk d_type;
    /// the allocated stack
    union {
      bool*     p_bstk;
      long*     p_istk;
      t_real*   p_rstk;
      Literal** p_lstk;
    };

  public:
    /// create an empty stack.
    Lstack (void);

    /// create a literal stack by size
    /// @param size the requested size  
    Lstack (const long size);
  
    /// copy constructor for this stack
    /// @param that the stack to copy
    Lstack (const Lstack& that);

    /// destroy this literal stack
    ~Lstack (void);

    /// @return the class name
    String repr (void) const;

    /// @return a clone of this object
    Object* clone (void) const;

    /// clear this literal stack
    void clear (void);

    /// @return a literal representation of this literal
    String toliteral (void) const;

    /// @return a string representation of this literal
    String tostring (void) const;

    /// @return a styled string of this literal
    String format (const Style& lstl) const;

    /// @return the stack serial code
    t_byte serialid (void) const;

    /// serialize this stack to an output stream
    /// @param os the output stream to write
    void wrstream (class OutputStream& os) const;

    /// deserialize a stack from an input stream
    /// @param is the input steam to read in
    void rdstream (class InputStream& is);

    /// assign a stack to this one
    /// @param that the stack to assign
    Lstack& operator = (const Lstack& that);

    /// @return true if the stack is empty
    virtual bool empty (void) const;

    /// @return the number of elements
    virtual long length (void) const;

    /// resize this literal stack
    virtual void resize (const long size);

    /// set the literal index
    /// @param lidx the index to set
    virtual void setlidx (const long lidx);

    /// @return the literal index
    virtual long getlidx (void) const;

    /// add a boolean to this stack
    /// @param bval the boolean to add
    virtual void add (const bool bval);

    /// add an integer to this stack
    /// @param ival the integer to add
    virtual void add (const long ival);

    /// add a real to this stack
    /// @param ival the integer to add
    virtual void add (const t_real rval);

    /// add a literal to this stack
    /// @param lobj the literal object to add
    virtual void add (Literal* lobj);

    /// set a literal at a given position
    /// @param index the stack index
    /// @param lobj  the literal object
    virtual void set (const long index, Literal* lobj);

    /// get a literal at a certain index
    /// @param index the stack index
    virtual Literal* get (const long index) const;

    /// @return the first literal in this stack
    virtual Literal* first (void) const;

    /// @return the last literal in this stack
    virtual Literal* last (void) const;
    
    /// @return a new iterator for this stack
    Iterator* makeit (void);

  private:
    // make the literal stack iterator a friend
    friend class Lstackit;

  public:
    /// create a new object in a generic way
    /// @param argv the argument vector
    static Object* mknew (Vector* argv);
    
    /// @return true if the given quark is defined
    bool isquark (const long quark, const bool hflg) const;

    /// apply this object with a set of arguments and a quark
    /// @param robj  the current runnable
    /// @param nset  the current nameset    
    /// @param quark the quark to apply these arguments
    /// @param argv  the arguments to apply
    Object* apply (Runnable* robj, Nameset* nset, const long quark,
		   Vector* argv);
  };

  /// The Lstackit class is the iterator for the stack class. Such 
  /// iterator is constructed with the "makeit" stack method. The iterator
  /// is reset to the beginning of the stack.
  /// @author amaury darsch

  class Lstackit : public Iterator {
  private:
    /// the stack to iterate
    Lstack* p_sobj;
    /// the iterable stack index
    long    d_sidx;

  public:
    /// create a new iterator from a literal stack
    /// @param vobj the stack to iterate
    Lstackit (Lstack* vobj);

    /// destroy this stack iterator
    ~Lstackit (void);

    /// @return the class name
    String repr (void) const;

    /// reset the iterator to the begining
    void begin (void);

    /// reset the iterator to the end
    void end (void);

    /// move the stack iterator to the next position
    void next (void);

    /// move the stack iterator to the previous position
    void prev (void);

    /// @return the object at the current position
    Object* getobj (void) const;

    /// @return true if the iterator is at the end
    bool isend (void) const;

  private:
    // make the copy constructor private
    Lstackit (const Lstackit&);
    // make the assignment operator private
    Lstackit& operator = (const Lstackit&);
  };
}

#endif
