# Print native libraries required by given package.
# USAGE: dependencies_libraries_list $package
dependencies_libraries_list() {
	local package
	package="$1"

	local dependencies_libraries
	dependencies_libraries=$(context_value "${package}_DEPENDENCIES_LIBRARIES")

	# Try to find a fallback value based on the package identifier:
	# current package → fallback package
	# PKG_BIN64       → PKG_BIN
	# PKG_BIN32       → PKG_BIN
	# PKG_LIBS64      → PKG_LIBS
	# PKG_LIBS32      → PKG_LIBS
	if [ -z "$dependencies_libraries" ]; then
		case "$package" in
			('PKG_BIN64'|'PKG_BIN32')
				dependencies_libraries=$(dependencies_libraries_list 'PKG_BIN')
			;;
			('PKG_LIBS64'|'PKG_LIBS32')
				dependencies_libraries=$(dependencies_libraries_list 'PKG_LIBS')
			;;
		esac
	fi

	# Fall back on default dependencies list for current game engine.
	if [ -z "$dependencies_libraries" ]; then
		local game_engine
		game_engine=$(game_engine)
		case "$game_engine" in
			('gamemaker')
				dependencies_libraries=$(dependencies_libraries_list_gamemaker "$package")
			;;
			('visionaire')
				dependencies_libraries=$(dependencies_libraries_list_visionaire "$package")
			;;
		esac
	fi

	printf '%s' "$dependencies_libraries" | list_clean
}

# Print native packages providing native libraries required by given package.
# USAGE: dependencies_libraries_list_packages $package
dependencies_libraries_list_packages() {
	local package
	package="$1"

	local dependencies_libraries
	dependencies_libraries=$(dependencies_libraries_list "$package")
	# Return early if current package does not require any native library.
	if [ -z "$dependencies_libraries" ]; then
		return 0
	fi

	local option_package dependencies_packages
	option_package=$(option_value 'package')
	case "$option_package" in
		('arch')
			dependencies_packages=$(
				set_current_package "$package"
				dependencies_libraries_list_archlinux "$dependencies_libraries"
			)
		;;
		('deb')
			dependencies_packages=$(dependencies_libraries_list_debian "$dependencies_libraries")
		;;
		('gentoo')
			dependencies_packages=$(
				set_current_package "$package"
				dependencies_libraries_list_gentoo "$dependencies_libraries"
			)
		;;
	esac

	printf '%s' "$dependencies_packages" | list_clean
}

# Print the list of native libraries required by all packages
# USAGE: dependencies_list_native_libraries_all
# RETURNS: a list of native library names,
#          one per line
dependencies_list_native_libraries_all() {
	local packages_list package dependencies_libraries dependencies_libraries_all
	packages_list=$(packages_list)
	for package in $packages_list; do
		dependencies_libraries=$(dependencies_libraries_list "$package")
		dependencies_libraries_all="${dependencies_libraries_all:-}
		$dependencies_libraries"
	done

	printf '%s' "$dependencies_libraries_all" | list_clean
}

# Print the path to a temporary files used for unknown libraries listing
# USAGE: dependencies_unknown_libraries_file
dependencies_unknown_libraries_file() {
	printf '%s/unknown_libraries_list' "$PLAYIT_WORKDIR"
}

# Print a list of unknown libraries
# USAGE: dependencies_unknown_libraries_list
dependencies_unknown_libraries_list() {
	local unknown_libraries_list
	unknown_libraries_list=$(dependencies_unknown_libraries_file)

	# Return early if there is no unknown library
	if [ ! -e "$unknown_libraries_list" ]; then
		return 0
	fi

	list_clean < "$unknown_libraries_list"
}

# Clear the list of unknown libraries
# USAGE: dependencies_unknown_libraries_clear
dependencies_unknown_libraries_clear() {
	local unknown_libraries_list
	unknown_libraries_list=$(dependencies_unknown_libraries_file)

	rm --force "$unknown_libraries_list"
}

# Add a library to the list of unknown ones
# USAGE: dependencies_unknown_libraries_add $unknown_library
dependencies_unknown_libraries_add() {
	local unknown_library unknown_libraries_list
	unknown_library="$1"
	unknown_libraries_list=$(dependencies_unknown_libraries_file)

	# Do nothing if this library is already included in the list
	if
		[ -e "$unknown_libraries_list" ] &&
		grep --quiet --fixed-strings --word-regexp "$unknown_library" "$unknown_libraries_list"
	then
		return 0
	fi

	printf '%s\n' "$unknown_library" >> "$unknown_libraries_list"
}

# Add a depdendency to the list of the given package.
# This function is used to update the native libraries dependencies list.
# USAGE: dependencies_add_native_libraries $package $dependency
dependencies_add_native_libraries() {
	local package dependency
	package="$1"
	dependency="$2"

	local current_dependencies
	current_dependencies=$(dependencies_libraries_list "$package")

	local dependencies_variable_name
	dependencies_variable_name=$(context_name "${package}_DEPENDENCIES_LIBRARIES")
	if [ -z "$dependencies_variable_name" ]; then
		dependencies_variable_name="${package}_DEPENDENCIES_LIBRARIES"
	fi
	export $dependencies_variable_name="$current_dependencies
	$dependency"
}

