// Boost.Geometry (aka GGL, Generic Geometry Library)
// Unit Test

// Copyright (c) 2014-2019 Barend Gehrels, Amsterdam, the Netherlands.

// Use, modification and distribution is subject to the Boost Software License,
// Version 1.0. (See accompanying file LICENSE_1_0.txt or copy at
// http://www.boost.org/LICENSE_1_0.txt)

#include "test_buffer.hpp"


template <typename MultiPolygon>
std::string read_from_file(std::string const& filename)
{
    MultiPolygon mp;
    std::ifstream in(filename.c_str());
    while (in.good())
    {
        std::string line;
        std::getline(in, line);
        if (! line.empty())
        {
            typename boost::range_value<MultiPolygon>::type pol;
            bg::read_wkt(line, pol);
            mp.push_back(pol);
        }
    }
    std::ostringstream out;
    if (! mp.empty())
    {
        out << std::fixed << std::setprecision(19) << bg::wkt(mp);
    }

    BOOST_CHECK(! out.str().empty());

    return out.str();
}


/*

Area's from PostGIS:

query:
    with viewy as
    (
      select ST_GeomFromText
      (
      'MULTIPOLYGON(((-702580.6096782692475244403 6430990.7687819330021739006,-703230.0475875560659915209 6431039.6897066971287131310,-703260.8830865059280768037 6431616.0447590677067637444,-702209.5818154554581269622 6433969.2453343933448195457,-701931.2830884709255769849 6434497.0344218844547867775,-701467.5260898249689489603 6434786.0182951148599386215,-700756.1945436559617519379 6434833.5779418582096695900,-699921.2983627079520374537 6434065.5660789832472801208,-699643.1109552145935595036 6433585.3568368786945939064,-699086.5135012482060119510 6432576.2640118366107344627,-698993.6730459277750924230 6432239.8116209311410784721,-699797.6224084376590326428 6431375.9207902560010552406,-700292.3262255217414349318 6431087.0551158646121621132,-702580.6096782692475244403 6430990.7687819330021739006)),((-735358.0771448947489261627 7178577.3841606164351105690,-735605.4290534354513511062 7178156.5045146262273192406,-736935.1403709617443382740 7177207.7119220057502388954,-737491.6265054368413984776 7177576.5995239885523915291,-738326.6340058789355680346 7178735.1250268649309873581,-740893.1048661170061677694 7182528.6822584075853228569,-749705.9349937491351738572 7205006.6147208353504538536,-762724.0815250772284343839 7224794.0963690327480435371,-762600.4055708040250465274 7225641.0741515457630157471,-764765.0130692812381312251 7234599.5305097363889217377,-765445.2864775179186835885 7235820.5329339085146784782,-770021.7420635188464075327 7241232.3524428531527519226,-771103.9901530132628977299 7242295.3629226731136441231,-776391.6659656920237466693 7246596.8462748657912015915,-782050.3696411874843761325 7249518.3589066835120320320,-782668.8607320359442383051 7249677.3384689968079328537,-783287.2405033904360607266 7249572.3082885984331369400,-785730.0354093580972403288 7248190.2115479269996285439,-797140.2832156693330034614 7234175.6422693207859992981,-798779.2400786185171455145 7225111.4401018721982836723,-795377.7617179383523762226 7219339.8478290000930428505,-795099.5743104477878659964 7218756.8999792803078889847,-794975.7870366862043738365 7218121.5951453419402241707,-795470.6021732615772634745 7212936.2726683067157864571,-810498.7334303526440635324 7193863.9115921957418322563,-811179.0068385921185836196 7193284.3684667171910405159,-814116.6168811351526528597 7191490.3536884263157844543,-816002.8143331365427002311 7191596.1295820372179150581,-841513.5680797586683183908 7194233.5802007140591740608,-847141.4362563013564795256 7197189.2738139629364013672,-848254.6311642369255423546 7197876.6753672696650028229,-848656.4945260005770251155 7198404.4778307117521762848,-848718.3881628799717873335 7199670.4357275264337658882,-848594.7122086095623672009 7201097.6388064818456768990,-848749.3349813210079446435 7201730.2385471686720848083,-849120.3628441347973421216 7202364.2203078540042042732,-856819.9980638342676684260 7208600.8437178684398531914,-857438.3778351887594908476 7208600.8437178684398531914,-862478.7017393268179148436 7208177.0236210795119404793,-870023.7141863148426637053 7207966.3590856380760669708,-870580.3116402812302112579 7207966.3590856380760669708,-874012.6254999083466827869 7210450.9231998035684227943,-875187.7140447233105078340 7211560.6087152827531099319,-875806.0938160805962979794 7212724.1476780120283365250,-875991.6634072330780327320 7214047.8326450148597359657,-876022.6102256742306053638 7218227.7297820551320910454,-876610.0431785903638228774 7223839.8538713958114385605,-876702.8836339107947424054 7224475.6212571412324905396,-876981.1823608925333246589 7225111.4401018721982836723,-879454.9240853014634922147 7226066.0278311185538768768,-884031.3796713051851838827 7227071.4743356276303529739,-893926.5692079173168167472 7238101.1117624733597040176,-895596.3615698162466287613 7238896.9872776530683040619,-896771.3387951399199664593 7239853.0616000862792134285,-897853.5868846315424889326 7241073.3528286991640925407,-907996.1283297898480668664 7253664.5514006679877638817,-908305.2625557214487344027 7254302.7396087851375341415,-909665.6980527064297348261 7258503.5592825226485729218,-895781.9311609688447788358 7266808.4238847019150853157,-885082.7922618469456210732 7271603.5667797476053237915,-872033.5875925865257158875 7275761.4648210369050502777,-862973.5168759049847722054 7283550.4871917199343442917,-865787.3396446865517646074 7289796.4451190056279301643,-873796.2204098060028627515 7290971.6515730358660221100,-874445.5469996044412255287 7291293.0323402481153607368,-880135.1974935409380123019 7297012.0099565647542476654,-880970.0936744889477267861 7298293.9856944410130381584,-882485.3745831678388640285 7303643.4018597202375531197,-882238.0226746243424713612 7304285.6780861504375934601,-873085.0001831282861530781 7310334.7953147590160369873,-871600.6660928905475884676 7310709.9959785044193267822,-871013.2331399742979556322 7310496.7502252506092190742,-870425.6888675697846338153 7309853.9632561607286334038,-861520.1296041056048125029 7305890.8288697442039847374,-840833.2946715191937983036 7315960.4538994673639535904,-832985.8271680481266230345 7328929.2893263995647430420,-830226.9962277177255600691 7334311.5376756498590111732,-828650.0443211420206353068 7339095.0725443745031952858,-824475.4520969019504263997 7352813.8748141247779130936,-815569.8928334405645728111 7370657.6415467225015163422,-807345.3862146517494693398 7375579.2351953061297535896,-807622.9057052005082368851 7371088.9971275608986616135,-807190.0955249957041814923 7370819.1056331051513552666,-789811.7871782749425619841 7369039.1054780697450041771,-787430.7745896981796249747 7369200.5359869319945573807,-786379.4733186477096751332 7369740.0217997981235384941,-785389.9543649854604154825 7370495.9865155210718512535,-784524.2226850874722003937 7371413.6954120760783553123,-782483.2911408834625035524 7373897.2790033649653196335,-781431.9898698300821706653 7375516.8260253472253680229,-778587.1089631190989166498 7383297.7232969645410776138,-775556.7697847438976168633 7392222.0447648391127586365,-766960.4560667055193334818 7394495.1678112838417291641,-759724.5778456518892198801 7394550.3234224319458007812,-749211.1198571709683164954 7396932.7178549375385046005,-736192.9733258428750559688 7402187.9909061845391988754,-725308.3761550596682354808 7407229.3494948502629995346,-709816.4878993228776380420 7408422.4015268804505467415,-707249.7944001023424789310 7408151.4058539085090160370,-696303.4149119253270328045 7403107.8879097951576113701,-686655.7999228363623842597 7404409.0014582388103008270,-684553.0860612413380295038 7404897.2679815273731946945,-683749.0253792431904003024 7404735.0945201823487877846,-679481.8153386650374159217 7402783.4239942766726016998,-678894.3823857487877830863 7402458.9736336832866072655,-675338.3925718484679237008 7399695.3543038265779614449,-674688.9546625588554888964 7398557.0712014045566320419,-672091.4256643884582445025 7393304.2529149157926440239,-671658.5041646952740848064 7392006.5564168058335781097,-671627.6686657454119995236 7391194.9242857862263917923,-672648.0231183547293767333 7381838.4430372426286339760,-673421.1369819147512316704 7378811.9759865151718258858,-671936.9142111683031544089 7373087.5347176529467105865,-650600.6413692950736731291 7341729.4264621166512370110,-644694.4744657649425789714 7333076.6175461346283555031,-643643.1731947144726291299 7333774.4993700264021754265,-642931.9529680367559194565 7333882.2134874602779746056,-639746.9910169501090422273 7333505.8973950138315558434,-638571.9024721352616325021 7332861.2130459556356072426,-638077.1986550511792302132 7332485.1395546719431877136,-635696.1860664744162932038 7329532.4808910498395562172,-635417.8873394897673279047 7328995.5782044418156147003,-633717.2594786409754306078 7325506.4177591772750020027,-633191.4975236244499683380 7324326.0056811599060893059,-633021.4013416939415037632 7317593.0931803137063980103,-633005.9279324719682335854 7316390.1439637970179319382,-633129.6038867422612383962 7315639.4076945250853896141,-633469.7962506090989336371 7314674.0333597715944051743,-633779.0417960318736732006 7313870.4174501933157444000,-634119.1228404046269133687 7313281.7431463403627276421,-635758.0797033538110554218 7311407.2306827269494533539,-639258.4097718583652749658 7308761.1154594300314784050,-639592.3682442387798801064 7308568.5479629402980208397,-640019.0558524483349174261 7308503.7815833184868097305,-641985.7372962941881269217 7308503.7815833184868097305,-644075.9833749192766845226 7307658.2055283375084400177,-647477.4617355966474860907 7305890.8288697442039847374,-648065.0060080040711909533 7305570.3889232203364372253,-652610.5147755667567253113 7302306.0416051279753446579,-653569.1982302792603150010 7301449.3883516052737832069,-654342.2007743450812995434 7300487.1135643599554896355,-655486.3425007188925519586 7298990.3059588475152850151,-655826.4235450944397598505 7298348.4646951872855424881,-659444.3069958741543814540 7291133.0134554086253046989,-659567.9829501473577693105 7290330.4681210117414593697,-657712.7323165842099115252 7285525.3807995188981294632,-657310.7576353321783244610 7285044.5346792098134756088,-656599.5374086515512317419 7284830.4504301641136407852,-655362.5552269573090597987 7285258.6247859327122569084,-654898.7982283142628148198 7285685.2850175667554140091,-654558.6058644475415349007 7286271.6925147818401455879,-653971.0615920429117977619 7288408.5788867557421326637,-651868.4590499391779303551 7291827.1575358547270298004,-647477.4617355966474860907 7295299.9569906648248434067,-645993.2389648500829935074 7296423.2570062335580587387,-639042.0046817560214549303 7298690.5552950426936149597,-638763.5946352829923853278 7298819.1559704272076487541,-638318.4279915994266048074 7298883.0720631638541817665,-626636.1153497900813817978 7299899.4466651547700166702,-623296.4193065008148550987 7299845.1493975305929780006,-620822.6775820920011028647 7299632.1999242166057229042,-620235.1333096873713657260 7299310.4762344490736722946,-616586.3030404637102037668 7295087.1312795402482151985,-616184.3283592087682336569 7294605.5046063205227255821,-615967.9232691064244136214 7293964.0231761783361434937,-615349.5434977519325911999 7290384.6993595836684107780,-609969.0272297499468550086 7272402.8508998267352581024,-605052.3792798821814358234 7263400.9426858676597476006,-604526.7286443569464609027 7262975.4949916983023285866,-604433.9995085247792303562 7262282.3713537119328975677,-605052.3792798821814358234 7257865.6035474222153425217,-605176.0552341524744406343 7257067.9925211919471621513,-607928.2070050371112301946 7243515.8429014263674616814,-612356.2737098102224990726 7239513.5310624139383435249,-612690.2321821906371042132 7239353.8020243635401129723,-613153.9891808364773169160 7239322.4680012036114931107,-613821.9061255943961441517 7239736.5075368294492363930,-614230.1146983328508213162 7240372.7991492254659533501,-617291.4006951490882784128 7246553.8178517967462539673,-617421.3105409058043733239 7246936.1097532864660024643,-617476.9702863016864284873 7247351.1250555245205760002,-617514.0396767350612208247 7248306.8912714971229434013,-617347.0604405448539182544 7249136.7063253857195377350,-617198.5602398286573588848 7249550.4989193044602870941,-617031.6923231268301606178 7249900.7953358525410294533,-615844.2473148361314088106 7251400.6847681496292352676,-617359.4169040236156433821 7254250.2890685461461544037,-616895.5485858893953263760 7255631.9227854702621698380,-616771.8726316161919385195 7256376.9229258280247449875,-617112.0649954801192507148 7261006.2797204386442899704,-617823.1739026695722714067 7265263.9857233623042702675,-618039.6903122602961957455 7265903.1192052718251943588,-618905.5333116523688659072 7268460.1747035766020417213,-619121.9384017547126859426 7269045.4850603407248854637,-619833.1586284324293956161 7270110.7394062280654907227,-623358.3129433830035850406 7273468.5620510326698422432,-627780.1457566726021468639 7276775.0316838556900620461,-629975.5887540967669337988 7278160.6439954033121466637,-632356.6013426764402538538 7279441.0702628456056118011,-633717.2594786409754306078 7279867.6054819477722048759,-634366.4747489481233060360 7279655.0071296412497758865,-634582.9911585417576134205 7278960.5978231728076934814,-635046.7481571848038583994 7271869.8578517930582165718,-631830.9507071482948958874 7269844.3163336357101798058,-631336.2468900642124935985 7269578.0941229732707142830,-628400.8632373372092843056 7264020.9494932899251580238,-620421.1481788019882515073 7241887.6865395959466695786,-619846.0716893646167591214 7239944.9685657471418380737,-619066.8352538122562691569 7235710.5318127926439046860,-619252.2935254735639318824 7235360.1044908678159117699,-618843.6396747701801359653 7225430.1317164767533540726,-619183.7207191428169608116 7223946.0657785069197416306,-621904.9256715864175930619 7220504.5390589004382491112,-624316.8850786042166873813 7217592.4677235791459679604,-628738.7178918938152492046 7213306.8386734155938029289,-629326.2621643012389540672 7212989.9712846651673316956,-629913.6951172174885869026 7213148.4033825341612100601,-630315.7811179606942459941 7213623.5283982306718826294,-630841.4317534888396039605 7214787.3727535428479313850,-632789.5228423695079982281 7216322.0976330684497952461,-633408.0139332180842757225 7216640.2424348145723342896,-634181.0164772868156433105 7216798.7481244029477238655,-637922.5758823397336527705 7217169.4962707208469510078,-639654.2618811180582270026 7217275.6180500760674476624,-648157.7351438361220061779 7215793.0913441013544797897,-648714.3325978023931384087 7215528.5063093286007642746,-651837.5122314980253577232 7213465.0867246836423873901,-652765.2488677693763747811 7212671.9741857927292585373,-653785.6033203788101673126 7210662.9868646310642361641,-653878.3324562108609825373 7209075.6939942780882120132,-653661.9273661085171625018 7207648.5650895787402987480,-652981.6539578718366101384 7205799.1447007078677415848,-652888.9248220397857949138 7205058.9276083298027515411,-652765.2488677693763747811 7202681.9910357510671019554,-652919.7603209896478801966 7200991.7351339543238282204,-655115.2033184138126671314 7192229.2954357927665114403,-656383.0209990608273074031 7189011.9584304438903927803,-657186.9703615678008645773 7188009.6574359694495797157,-659042.4436341105028986931 7186428.4453326268121600151,-664391.9017641830723732710 7182265.4030445581302046776,-672802.6458910661749541759 7176102.8584930133074522018,-675462.0685261187609285116 7175417.5443301461637020111,-676266.0178886285284534097 7175365.4280199278146028519,-677038.9091132060857489705 7175523.1046035401523113251,-679976.6304752404103055596 7177050.1911828350275754929,-686934.0986498210113495588 7183318.9521107096225023270,-687645.2075570075539872050 7184372.6421100758016109467,-688603.8910117199411615729 7186744.0550359664484858513,-689315.1112383977742865682 7187851.7307325564324855804,-689809.8150554818566888571 7188221.1171403760090470314,-690490.0884637213312089443 7188484.5955257071182131767,-697636.4658141745021566749 7189085.8072432382032275200,-697695.0198663338087499142 7189486.9569816533476114273,-698251.5060008089058101177 7190066.4104806575924158096,-698931.7794090455863624811 7190487.9270409662276506424,-700044.9743169782450422645 7190751.4812934864312410355,-705641.8956750826910138130 7192071.4739870699122548103,-706847.9310383386909961700 7192071.4739870699122548103,-707435.3639912548242136836 7191807.6856330027803778648,-707992.0727647095918655396 7191332.5470526618883013725,-708393.9361264732433483005 7190751.4812934864312410355,-708579.5057176257250830531 7190172.1672446671873331070,-708672.3461729490663856268 7189591.1871162438765168190,-708641.3993545080302283168 7189011.9584304438903927803,-717732.4168896306073293090 7181210.4776329882442951202,-731090.7557848254218697548 7179788.2020574221387505531,-735358.0771448947489261627 7178577.3841606164351105690)),((-688882.1897387016797438264 7412436.5099641056731343269,-694015.2427786719053983688 7418624.0284140892326831818,-697169.2579113173997029662 7418407.8061552522704005241,-698839.1615927076200023293 7418516.6985288234427571297,-699364.7009087415644899011 7418788.0563712492585182190,-699241.0249544712714850903 7419602.5779014062136411667,-697169.2579113173997029662 7421068.9713027896359562874,-696025.2275044348789379001 7421829.9468861259520053864,-694510.0579152472782880068 7421992.6735327132046222687,-689191.4352841245708987117 7420797.5336704570800065994,-687768.9948307690210640430 7420254.4913531979545950890,-687367.0201495141955092549 7419818.8334604389965534210,-687150.5037399205612018704 7418950.7192241176962852478,-688078.1290567035321146250 7413467.8112517688423395157,-688480.2150574468541890383 7412761.3775515491142868996,-688882.1897387016797438264 7412436.5099641056731343269)),((-698468.0224104025401175022 7473833.5146531024947762489,-701374.6856345045380294323 7474653.6489164056256413460,-703230.0475875560659915209 7475583.7647755797952413559,-704157.6729043362429365516 7476567.9621847020462155342,-704374.0779944385867565870 7477114.9677619626745581627,-703755.6982230813009664416 7482205.5315547427162528038,-703632.0222688110079616308 7482918.3282846277579665184,-699735.7287715553538873792 7500351.2504732506349682808,-696087.0098218229832127690 7511500.4964982029050588608,-695870.6047317205229774117 7512214.6175983380526304245,-695901.5515501616755500436 7513038.8378656907007098198,-696241.6325945372227579355 7513698.6733295693993568420,-696767.2832300624577328563 7514083.8343340521678328514,-697447.5566382991382852197 7514302.7473613265901803970,-704126.7260858950903639197 7515018.5136068165302276611,-704837.9463125757174566388 7515018.5136068165302276611,-705518.2197208123980090022 7514797.9966831970959901810,-706384.0627202015602961183 7513753.1013900088146328926,-706693.1969461359549313784 7513038.8378656907007098198,-707157.1765837614657357335 7511719.3362279795110225677,-707589.9867639662697911263 7510400.0602311035618185997,-708053.8550821004901081324 7508862.2448687991127371788,-708826.8576261692214757204 7506994.0948427924886345863,-710187.5157621366670355201 7504467.4917841125279664993,-711671.7385328831151127815 7502326.0925307748839259148,-713650.6651207137620076537 7499692.5552066601812839508,-717948.9332992242416366935 7495028.9645483279600739479,-719402.3205710208276286721 7493766.8251204313710331917,-720608.1332952943630516529 7493000.1262787943705916405,-721907.0091138707939535379 7492726.1545226909220218658,-722618.1180210572201758623 7492890.1811141148209571838,-723731.4242484811693429947 7493658.2508851466700434685,-724133.2876102448208257556 7494316.4315773332491517067,-724597.1559283791575580835 7495412.9975283788517117500,-724689.8850642112083733082 7497058.3359622266143560410,-723576.6901562785496935248 7503149.4548785416409373283,-723051.0395207533147186041 7505565.5006046025082468987,-717979.7687981741037219763 7528337.0292042447254061699,-717083.2016193262534216046 7529327.9761461354792118073,-709074.2095347127178683877 7532468.2375808423385024071,-702395.1514066079398617148 7531311.8393505858257412910,-689315.1112383977742865682 7541787.6536649884656071663,-686068.2556504289386793971 7543996.0762523645535111427,-683718.1898802933283150196 7545210.4222629889845848083,-683130.6456078859046101570 7545375.7677722154185175896,-682326.6962453760206699371 7545210.4222629889845848083,-681491.8000644280109554529 7544271.8755555301904678345,-681120.6608821229310706258 7542836.3568872893229126930,-681584.5292002571513876319 7537540.8627359746024012566,-681460.8532459868583828211 7534342.3719757227227091789,-679389.0862028329866006970 7520903.3390643103048205376,-679265.4102485626935958862 7520133.9395713303238153458,-678770.7064314757008105516 7518042.6464647594839334488,-678306.8381133413640782237 7516722.2886397214606404305,-677750.2406593750929459929 7515512.4231265708804130554,-676327.8002060195431113243 7514248.3154224921017885208,-674534.3318898474099114537 7513203.4945802800357341766,-673730.2712078491458669305 7512214.6175983380526304245,-673482.9192993056494742632 7511609.1241234233602881432,-671287.4763018814846873283 7502985.2102015949785709381,-670669.0965305241988971829 7495138.9401649823412299156,-670978.2307564557995647192 7494534.9820824339985847473,-672184.2661197116831317544 7493000.1262787943705916405,-676451.4761602898361161351 7489052.5557834561914205551,-693396.7516878233291208744 7477662.0120453117415308952,-698468.0224104025401175022 7473833.5146531024947762489)),((-142767.2469423743896186352 6547139.2345692599192261696,-152229.4036598020466044545 6554104.9963345807045698166,-163979.8438299772387836128 6561516.0021916879341006279,-166268.0159632335416972637 6562638.7391632050275802612,-166886.5070540792366955429 6562882.6951917400583624840,-168339.7830063846195116639 6562930.8198394523933529854,-169576.7651880816847551614 6562687.0380198908969759941,-171215.6107315395784098655 6562003.6794201470911502838,-174678.8714096079056616873 6560930.5466297725215554237,-174771.6005454399564769119 6561467.7101931860670447350,-172637.9398654038668610156 6564053.9251385293900966644,-169576.7651880816847551614 6567567.2077301768586039543,-145704.8569849174818955362 6580905.3427298367023468018,-144375.2569868824211880565 6581051.6178714158013463020,-143756.7658960339031182230 6580953.5727541074156761169,-124461.4245983647560933605 6574990.3297461569309234619,-123997.5562802276544971392 6574599.8518867725506424904,-121771.1664643623080337420 6572010.5021970951929688454,-117967.7134224300389178097 6566199.7380468165501952171,-130058.3459564691875129938 6551181.9892134414985775948,-132624.8168167072290088981 6549719.5585852097719907761,-135284.1281322685827035457 6548599.8054481893777847290,-138221.7381748116458766162 6547771.9240455087274312973,-142767.2469423743896186352 6547139.2345692599192261696)),((-106928.6047984238393837586 6581490.8109566625207662582,-108289.1516148999944562092 6581784.9690047213807702065,-113824.1793361222225939855 6583496.2583401361480355263,-113731.4502002901717787609 6584132.9769384199753403664,-113422.2046548673097277060 6584572.3344440879300236702,-108629.2326592727040406317 6591326.2465774957090616226,-108072.6352053063747007400 6591423.1816138168796896935,-107361.5262981198466150090 6591228.2552289562299847603,-105660.7871177797496784478 6589955.0025413725525140762,-105351.5415723540645558387 6589417.2116007748991250992,-104207.3998459831345826387 6582566.6249391864985227585,-104454.7517545237933518365 6582029.3165351273491978645,-106928.6047984238393837586 6581490.8109566625207662582)),((-519150.8017779086367227137 6649348.9302046010270714760,-519769.4041882483870722353 6649545.2418169863522052765,-520233.1611868914333172143 6649890.6622821614146232605,-520356.8371411645784974098 6650581.3694077720865607262,-520016.7560967890312895179 6654131.8595868386328220367,-519800.2396871982491575181 6655514.0480420421808958054,-519367.3181875022710300982 6655859.7204466462135314941,-518625.2624618746340274811 6655859.7204466462135314941,-518346.8524154016631655395 6655365.0190682355314493179,-517233.6575074690044857562 6650433.6728300666436553001,-517295.5511443483410403132 6649791.7917294288054108620,-517821.3130993648082949221 6649545.2418169863522052765,-519150.8017779086367227137 6649348.9302046010270714760)),((100094.8125781164126237854 6684726.0087605575099587440,98548.6961704877467127517 6684924.5976608581840991974,87664.0989997017313726246 6686905.0463614761829376221,86952.8787730239710072055 6687103.6882660342380404472,85283.0864111249684356153 6688045.2524453820660710335,84540.9193660061137052253 6688936.4502402693033218384,82963.9674594304087804630 6691364.4189215200021862984,82469.1523228522128192708 6692405.6823790417984127998,81819.8257330566557357088 6693843.4366568028926849365,82438.3168239023507339880 6697660.7728029768913984299,83365.9421406824840232730 6700140.4544257819652557373,84633.7598213294113520533 6700339.2401692178100347519,88066.0736809566442389041 6699495.9075616495683789253,99321.6987145564635284245 6696173.4326590122655034065,100651.4100320827565155923 6695726.6230171313509345055,101177.0606676079914905131 6695379.4386507561430335045,103588.9087551346019608900 6692158.0223465478047728539,104269.2934828625002410263 6691215.9836819367483258247,105073.2428453723114216700 6689679.5164131233468651772,105475.2175266272242879495 6688540.5088516287505626678,105506.1643450683186529204 6687846.5876310169696807861,105413.3238897450355580077 6687203.0110306832939386368,104918.6200726608949480578 6686856.1890392992645502090,100713.1923494708753423765 6684875.5739196399226784706,100094.8125781164126237854 6684726.0087605575099587440)),((-484394.4077234524884261191 7006049.7621033526957035065,-490053.2227184362709522247 7006307.4369544526562094688,-498587.6427995953708887100 7015743.4408356416970491409,-500319.3287983737536706030 7018115.9862014595419168472,-508451.7741982779698446393 7033717.6659270338714122772,-508637.3437894305097870529 7055102.4795240126550197601,-508389.9918808870133943856 7055724.8110826518386602402,-507616.8780173270497471094 7056708.9933504080399870872,-498123.8858009495306760073 7060807.9525244953110814095,-493052.6150783703196793795 7061896.6329905483871698380,-492372.3416701336973346770 7062000.6893656672909855843,-491599.3391260649659670889 7062000.6893656672909855843,-480220.0381381948245689273 7059562.5416110474616289139,-477838.9142301268293522298 7058731.8781612580642104149,-476849.3952764644636772573 7057953.9569422556087374687,-476107.3395508368266746402 7057020.9627588037401437759,-475334.2256872769212350249 7055206.4469419745728373528,-470015.6030561570660211146 7042719.9294385062530636787,-459409.4159318468300625682 7040650.0641330182552337646,-458883.7652963215950876474 7040908.6746784290298819542,-458172.5450696410262025893 7041010.9660405069589614868,-457523.2184798454400151968 7040959.9133565612137317657,-451957.1326206909143365920 7039717.6081727687269449234,-450472.9098499444080516696 7039303.8600313169881701469,-450442.0743509945459663868 7038631.7747488524764776230,-455296.7173444888903759420 7030305.4587014671415090561,-456564.5350251329946331680 7028961.1894702669233083725,-462347.0259743899223394692 7024828.9599360972642898560,-469551.8460575111676007509 7020129.0798942381516098976,-481518.6913177887327037752 7009863.6067247306928038597,-484394.4077234524884261191 7006049.7621033526957035065)),((-509564.9691062106285244226 7027153.0635162591934204102,-512842.7715126176481135190 7030511.4449932724237442017,-519243.6422332319198176265 7034647.9294924885034561157,-520635.1358681463170796633 7035010.0486788097769021988,-521315.4092763858498074114 7035993.1596986297518014908,-522521.4446396389394067228 7038941.5453599998727440834,-522304.9282300453633069992 7039458.8500605020672082901,-521129.9510047245421446860 7041994.8189928056672215462,-520820.7054592988570220768 7042564.8758499575778841972,-520140.4320510621764697134 7042668.6794203175231814384,-515038.2145100419293157756 7042978.6072979765012860298,-514481.6170560755999758840 7042668.6794203175231814384,-511049.3031964483670890331 7038165.5582551229745149612,-507678.7716542092384770513 7030925.8527166405692696571,-507555.0956999360932968557 7030201.8163049044087529182,-507555.0956999360932968557 7029477.8457472492009401321,-507709.7184726503328420222 7028702.9669241644442081451,-508018.9640180732239969075 7028135.2044416479766368866,-508884.6956979711540043354 7027256.6662269476801156998,-509564.9691062106285244226 7027153.0635162591934204102)),((-296666.4429640754242427647 6730809.2905607484281063080,-297748.8023730582208372653 6729365.0000257585197687149,-300438.9491875665844418108 6726228.4902110705152153969,-302201.5820047889137640595 6724685.4097353378310799599,-317013.1968522862880490720 6718269.7042935220524668694,-322022.5739379832521080971 6717125.4435921907424926758,-324929.2371620881021954119 6716976.5386439291760325432,-328918.1484756816644221544 6717324.8229897394776344299,-330216.9129747668630443513 6717672.9432423794642090797,-336308.5381499582435935736 6714740.2493202565237879753,-346822.1074579276028089225 6708430.6989329699426889420,-347533.3276846053777262568 6707535.9388622017577290535,-348182.6542744037578813732 6706443.5887363040819764137,-350841.9655899651115760207 6701827.5188080035150051117,-351800.5377251863828860223 6698999.9858346609398722649,-351614.9681340338429436088 6698355.5290799643844366074,-351553.1858166428864933550 6697709.8733980739489197731,-351614.9681340338429436088 6696173.4326590122655034065,-351955.1604978977702558041 6694735.2678222069516777992,-352202.5124064384144730866 6694091.1478441227227449417,-352480.8111334230052307248 6693545.2294893041253089905,-353624.9528597967582754791 6692206.9114117827266454697,-354614.4718134562717750669 6691563.1693708989769220352,-363829.1653028520522639155 6688887.4024045290425419807,-364416.8208947507082484663 6688737.7637376058846712112,-372487.3726577727356925607 6688540.5088516287505626678,-373971.7067480076220817864 6688540.5088516287505626678,-378857.4078794342931360006 6688737.7637376058846712112,-380310.6838317424990236759 6688887.4024045290425419807,-384175.9191910657682456076 6689630.4641125248745083809,-388350.4000958118122071028 6690572.1413981299847364426,-394349.2961351711419411004 6691959.2574109639972448349,-396328.3340424958732910454 6693346.4307649247348308563,-408759.0476209077169187367 6705550.4764933045953512192,-417386.1968378960737027228 6714740.2493202565237879753,-417866.3178016852471046150 6716183.2003521975129842758,-418684.9613369812141172588 6718170.0024194139987230301,-420973.2447897259262390435 6722298.0005760667845606804,-426972.1408290852559730411 6731556.1845118729397654533,-427497.7914646104909479618 6731854.3516772650182247162,-439681.0418149904580786824 6730411.3097816780209541321,-441443.6746322099352255464 6729862.7735847430303692818,-467572.8081501893466338515 6717821.8609223226085305214,-467789.3245597829227335751 6716430.3385093230754137039,-468160.3524225967703387141 6715982.5962533196434378624,-468809.7903318835305981338 6715882.9223695350810885429,-469459.1169216819689609110 6716082.2713566971942782402,-472303.8865089045721106231 6717524.2072677453979849815,-476200.1800061601679772139 6720358.1752467108890414238,-477900.8078670090180821717 6728567.7653740793466567993,-477900.8078670090180821717 6729315.8875402389094233513,-477684.4027769066160544753 6729962.6187606593593955040,-473169.7295082937343977392 6733249.7686292603611946106,-471097.9624651398626156151 6734046.0274155708029866219,-468809.7903318835305981338 6732900.8022920805960893631,-465377.3651527651236392558 6732153.9642069246619939804,-464697.0917445255909115076 6732103.2211532425135374069,-463367.3804270021500997245 6732302.9677236676216125488,-458234.3273870319826528430 6734245.8216936066746711731,-453967.1173464538296684623 6736388.9867214597761631012,-453627.0363020811346359551 6736836.4176607690751552582,-452575.6237115394324064255 6739229.5325915897265076637,-452390.1654398753307759762 6739927.8440811540931463242,-452513.8413941484759561718 6740624.7796155549585819244,-453085.4669793718494474888 6741509.3617311874404549599,-453595.9781641488079912961 6741971.0771495644003152847,-467263.5626047664554789662 6742320.2533843247219920158,-474468.3826878877007402480 6741921.7089691357687115669,-493825.6176224390510469675 6752750.3917194996029138565,-509224.8880618350813165307 6752200.3463254561647772789,-515223.7841011944110505283 6751251.5156823564320802689,-517171.8751900779898278415 6750852.5330523196607828140,-519552.8877786547527648509 6750103.8644746709614992142,-520511.4599138760240748525 6749455.3496915753930807114,-521253.6269589948933571577 6748506.8395475624129176140,-521655.4903207585448399186 6747159.4591391524299979210,-521717.3839576378813944757 6744165.4814152186736464500,-522119.3586388928233645856 6740076.9900818299502134323,-524160.1788636056007817388 6737136.2137496760115027428,-524685.9408186221262440085 6736887.1902486104518175125,-532478.3051741536473855376 6733847.6727800481021404266,-533189.4140813400736078620 6733996.7073211427778005600,-533746.1228547976352274418 6734295.1432984098792076111,-535137.5051702208584174514 6735391.2361996844410896301,-537425.7886229683645069599 6736189.1397955883294343948,-538167.9556680872337892652 6736338.2172435978427529335,-538817.3935773740522563457 6736238.2937905825674533844,-541229.2416649034712463617 6735441.8203898128122091293,-541723.9454819876700639725 6735192.6693880939856171608,-544383.2567975489655509591 6732950.1157429544255137444,-550072.9072914826683700085 6727074.7735988246276974678,-557185.1095582658890634775 6729265.1621820526197552681,-561328.5323250824585556984 6731107.4303566515445709229,-562318.1625982332043349743 6731755.7383567309007048607,-558966.1100914680864661932 6740426.2644657352939248085,-553325.9967709350166842341 6741712.2116987816989421844,-546022.2136604981496930122 6745512.3655102550983428955,-544816.1782972421497106552 6745961.7356973951682448387,-541136.5125290714204311371 6748457.4316534223034977913,-541816.7859373081009835005 6750153.2823813902214169502,-543733.9302077505271881819 6754448.3974975654855370522,-544352.3099791079293936491 6751950.8608885398134589195,-544414.2036159901181235909 6750502.8103392431512475014,-544630.7200255808420479298 6749854.2637134613469243050,-545403.7225696495734155178 6748905.7070041354745626450,-545929.3732051748083904386 6748556.0680754212662577629,-546949.8389772782102227211 6748057.1489544212818145752,-557048.9658210255438461900 6746779.8974972786381840706,-557791.1328661444131284952 6746600.2720435233786702156,-563839.4547594159375876188 6746959.3472899133339524269,-577871.9444908523000776768 6747658.3231466887518763542,-582046.4253955984022468328 6750203.9585106121376156807,-584087.2456203140318393707 6751500.9796426724642515182,-574532.3597670543240383267 6757244.5433394722640514374,-573419.1648591215489432216 6757745.2861772757023572922,-572769.7269498319365084171 6757844.2153189647942781448,-570419.6611796963261440396 6757794.7505974583327770233,-569090.0611816612072288990 6758044.5954917315393686295,-568595.3573645771248266101 6758343.7359562143683433533,-568224.2181822720449417830 6758793.8165175551548600197,-567945.9194552875123918056 6760243.3462205436080694199,-567945.9194552875123918056 6760992.9501268640160560608,-569368.2485891545657068491 6769694.5714717023074626923,-569492.0358629161491990089 6770395.5073453346267342567,-569955.9041810504859313369 6772247.1268248278647661209,-570172.3092711528297513723 6772847.9084392869845032692,-570759.8535435602534562349 6773848.8294274490326642990,-571749.2611777313286438584 6774550.1244038157165050507,-576820.5319003105396404862 6776503.1116253528743982315,-577562.6989454294089227915 6776652.9309746790677309036,-580345.6862152611138299108 6776753.3529545711353421211,-585416.8456183490343391895 6776753.3529545711353421211,-583221.4026209219591692090 6785020.3456817232072353363,-567945.9194552875123918056 6792845.0460455110296607018,-567389.3220013211248442531 6793146.9183306964114308357,-565688.5828209810424596071 6795655.7299079764634370804,-565472.0664113903185352683 6796258.2495336793363094330,-565131.9853670147713273764 6799421.9422668581828474998,-539281.1505760198924690485 6802636.4299470074474811554,-537920.6037595437373965979 6802887.6607939898967742920,-537487.6822598505532369018 6803239.6503190854564309120,-530746.6191753724124282598 6811785.0676737958565354347,-529788.0470401511993259192 6813344.8049502596259117126,-528334.7710878457874059677 6816464.0953564802184700966,-527314.3053157423855736852 6818880.6518225604668259621,-525737.2420896753901615739 6820592.5748667381703853607,-519800.2396871982491575181 6823714.6641129478812217712,-517295.5511443483410403132 6824420.2928618714213371277,-516646.1132350615225732327 6824521.3092674687504768372,-515038.2145100419293157756 6824420.2928618714213371277,-511018.3563780072727240622 6823966.3693792633712291718,-503968.1590675974148325622 6823815.6716964272782206535,-500937.7085697310394607484 6824822.9151354199275374413,-500412.0579342058044858277 6825074.8363033309578895569,-477777.1319127358729019761 6840455.9986039269715547562,-471067.1269661900005303323 6845252.9463985934853553772,-466830.6411050704773515463 6850253.9696522578597068787,-461202.8842480161692947149 6858243.5609792144969105721,-459842.3374315428663976490 6860874.3301865793764591217,-456224.3426612690673209727 6869482.4721138169988989830,-455853.3147984552197158337 6870900.3640763349831104279,-455513.2337540825246833265 6872370.1356682535260915756,-451926.2971217410522513092 6888451.3487260565161705017,-452080.8085749612073414028 6892157.9926208704710006714,-454245.4160734384204261005 6900799.5060561988502740860,-459069.2235679829027503729 6909094.0329884337261319160,-459347.6336144558736123145 6909653.4035110054537653923,-459471.3095687262248247862 6910314.7279802327975630760,-459316.6867960147792473435 6911079.8647817652672529221,-457121.1324790993239730597 6918670.8410148052498698235,-451215.0768950632773339748 6930758.7002626033499836922,-452328.2718029959360137582 6932136.9039406375959515572,-459811.3906131017720326781 6941996.8956455085426568985,-461542.9652923917165026069 6945218.3859691601246595383,-461790.4285204235929995775 6945781.5711939493194222450,-461914.1044746939442120492 6946446.0665490739047527313,-461914.1044746939442120492 6947213.1862867726013064384,-460460.8285223885322920978 6960216.4198623634874820709,-460306.0944301859126426280 6961035.5885519329458475113,-460089.6893400835106149316 6961599.8879740536212921143,-459378.4691134057356975973 6962727.3147383863106369972,-458450.8437966256169602275 6963495.9969000769779086113,-460151.5829769656993448734 6967188.0293207084760069847,-471252.5852378512499853969 6967289.3793843016028404236,-479725.2230016194516792893 6965545.9311952916905283928,-480498.2255456881248392165 6965341.8000272838398814201,-491382.8227164713316597044 6961702.6430851500481367111,-494320.4327590144239366055 6959448.2372251432389020920,-498402.0732084428891539574 6956221.0161899179220199585,-500968.6553881721338257194 6951510.4893854716792702675,-505328.7058840707759372890 6944809.6918431045487523079,-526293.8395436417777091265 6942662.3653290001675486565,-527747.2268154383637011051 6942559.8543665027245879173,-528489.2825410659424960613 6942712.9771497584879398346,-529447.8546762871555984020 6943429.1226814882829785347,-529976.8448965380666777492 6944058.2645517345517873764,-529015.0444960824679583311 6947366.2139230202883481979,-527778.1736338793998584151 6949669.3476609392091631889,-525644.5129538433393463492 6952739.1353130275383591652,-517759.4194624854717403650 6962112.2008093828335404396,-514172.4828301439411006868 6966213.3620561584830284119,-510523.6525609202799387276 6968930.5431283572688698769,-506812.9286548172240145504 6969854.5729465251788496971,-505421.4350199028267525136 6970161.8679190166294574738,-503906.3767502064583823085 6971136.8320272667333483696,-501587.1464790178579278290 6973035.2043069805949926376,-496763.3389844733756035566 6977501.6763582713901996613,-485847.7949952490744180977 6987879.3680515941232442856,-485074.6811316891689784825 6988805.7774775763973593712,-484765.5469057575101032853 6989370.7456465400755405426,-484301.6785876204376108944 6990708.3542606756091117859,-483683.1874967747135087848 6994618.0482657440006732941,-483590.4583609426626935601 6995492.8107249001041054726,-482044.3419533139676786959 7003782.8049474963918328285,-467139.8866504933102987707 7021213.6930979220196604729,-462347.0259743899223394692 7024828.9599360972642898560,-460646.2867940498399548233 7025603.6477430770173668861,-459873.1729304898763075471 7025603.6477430770173668861,-456997.4565248289727605879 7025189.3310140827670693398,-452668.3528473686310462654 7024983.6683287052437663078,-447349.8415357400663197041 7027049.4621537392958998680,-445618.1555369588313624263 7027876.8226692667230963707,-429074.8546906802803277969 7035785.7280582245439291000,-415530.9462043329258449376 7036872.6628930959850549698,-414943.4019319283543154597 7036561.4830648237839341164,-411047.2197541639907285571 7035993.1596986297518014908,-402203.5541275819996371865 7034802.8288447102531790733,-401492.3339009042829275131 7034906.3449965324252843857,-399575.1896304617985151708 7035372.3705378724262118340,-385351.0077358777634799480 7042512.3223147960379719734,-377156.6686990942107513547 7046810.1867410531267523766,-374342.7346108242636546493 7047276.7238292396068572998,-371405.1245682783192023635 7047484.4553862772881984711,-369889.9549790907767601311 7047484.4553862772881984711,-369024.1119797016144730151 7047380.5889292778447270393,-357521.1350375611218623817 7038216.7792665986344218254,-348491.8998198266490362585 7030771.2148909578099846840,-347471.4340477260411716998 7030047.1925449492409825325,-345337.7733676899224519730 7028961.1894702669233083725,-343821.2679446131805889308 7031238.3010051175951957703,-343266.0063245359924621880 7031959.4539151135832071304,-343049.6012344364426098764 7032631.1619424512609839439,-343142.3303702656994573772 7033355.4194014230743050575,-347997.1960027425084263086 7043341.2922983570024371147,-351120.2643169468501582742 7047690.8870053039863705635,-352264.2947238293709233403 7049142.2134381914511322975,-352821.0034972869325429201 7049504.9931898834183812141,-353594.0060413556639105082 7050385.9920698264613747597,-354799.9300851204316131771 7052562.3267318932339549065,-355171.0692674226011149585 7053805.3343668170273303986,-355356.6388585751410573721 7054635.2962300619110465050,-354799.9300851204316131771 7055672.1704734433442354202,-354026.9275410517002455890 7056606.4997908826917409897,-352975.6262699983781203628 7057384.1023639896884560585,-342389.4766540309065021574 7058931.3191051157191395760,-341664.2301715115318074822 7059230.4899098463356494904,-340829.4453100546961650252 7059355.0543599035590887070,-340031.6185195372090674937 7059106.4876597877591848373,-339345.1112198169576004148 7058732.4383816169574856758,-338899.8332566422177478671 7058203.2344924295321106911,-338510.2150388660957105458 7057549.3410466406494379044,-337285.7006401418475434184 7054780.6931206267327070236,-337063.0616585530224256217 7053131.8355975281447172165,-332783.4951544989598914981 7051214.2909560613334178925,-327557.6016592083615250885 7040650.0641330182552337646,-325975.9743340178392827511 7039086.4692494664341211319,-322795.6878015432157553732 7036665.2083219364285469055,-320878.4322116123512387276 7035940.6495621176436543465,-318621.2068967971717938781 7035476.0802941126748919487,-309066.2097240461735054851 7037286.0983680160716176033,-306685.1971354694105684757 7039252.8183029191568493843,-306283.2224542145268060267 7039717.6081727687269449234,-300840.9238688215264119208 7047328.0035564331337809563,-300655.4655971602187491953 7048053.6005296036601066589,-301057.4402784151607193053 7048467.6176822083070874214,-301892.3364593632286414504 7048624.2784440144896507263,-302510.7162307205144315958 7048467.6176822083070874214,-307829.3388618432218208909 7046344.9821726074442267418,-308757.0754981145728379488 7045670.8101515313610434532,-309097.1565424872678704560 7045101.8355482658371329308,-309344.5084510279120877385 7044480.3351130951195955276,-317662.5234420846682041883 7042512.3223147960379719734,-318497.4196230327361263335 7042616.1251992043107748032,-319888.9132579499855637550 7043186.2265946781262755394,-320940.3258484916877932847 7043755.0641425708308815956,-328825.4193398524075746536 7049919.0859430814161896706,-329938.6142477851244620979 7051318.2075183065608143806,-334712.9958884173538535833 7060579.1393866641446948051,-335436.5725785738904960454 7062011.5247577866539359093,-345925.3176400973461568356 7085700.3273308007046580315,-345832.5885042680893093348 7087313.0469446722418069839,-345554.2897772835567593575 7087990.9518432654440402985,-344379.2012324686511419713 7090333.4216981232166290283,-342091.0290992152295075357 7094396.8940942101180553436,-331979.5457919891341589391 7111767.0194333102554082870,-331577.4597912430181168020 7112340.7957691308110952377,-331082.7559741588775068521 7112706.4776853527873754501,-327526.7661602584994398057 7115057.9622176717966794968,-323073.9865285278065130115 7117408.6407044371590018272,-322702.8473462227848358452 7117879.3493779106065630913,-322548.2245735113392584026 7118558.3705776063725352287,-322888.4169373752665705979 7119133.9567664228379726410,-323290.3916186302085407078 7119500.1409429926425218582,-323970.6650268668890930712 7119761.3350805733352899551,-325547.7282529338262975216 7120022.7260320717468857765,-327217.5206148328143171966 7120074.2901339065283536911,-330185.9661563285626471043 7119864.6483984775841236115,-333154.5230173127492889762 7119708.2675620289519429207,-334669.6926065003499388695 7120126.0427638562396168709,-336370.4317868376383557916 7121014.9235066156834363937,-337700.0317848726408556104 7122374.8803893411532044411,-338473.1456484326045028865 7123315.5991421435028314590,-339679.0696921973722055554 7125721.8406816367059946060,-339988.3152376202051527798 7126716.2602372914552688599,-340421.2367373161832801998 7128756.9069768153131008148,-340606.6950089774909429252 7130274.9713758537545800209,-340606.6950089774909429252 7131896.7979978239163756371,-339864.6392833498539403081 7152341.7575662042945623398,-339771.7988280265708453953 7153023.7466275608167052269,-339215.3126935514737851918 7154335.9240338336676359177,-338380.3051931121735833585 7155228.4729645876213908195,-336741.4596496514277532697 7156174.6245612036436796188,-326444.4067512757028453052 7160797.3695420371368527412,-322486.4422561203828081489 7169945.3185568284243345261,-315436.1336262193508446217 7186270.5503361336886882782,-313642.6653100471594370902 7197242.8650309965014457703,-313209.8551298452657647431 7212459.8563267467543482780,-325795.0801614772644825280 7199618.1586596202105283737,-326753.6522966985357925296 7198878.5228944523259997368,-350903.8592268444481305778 7185690.0477500427514314651,-357861.3274014250491745770 7188273.3185697188600897789,-359129.0337625779211521149 7188854.0116235781461000443,-359994.8767619699356146157 7189750.6675885217264294624,-376321.7725181461428292096 7216586.5188759565353393555,-379506.7344692326732911170 7222728.4333535833284258842,-379815.9800146555644460022 7224052.0884321387857198715,-379846.8155136054265312850 7225482.3899835590273141861,-382258.7749206232838332653 7236350.8898326409980654716,-385845.7115529647562652826 7242188.9024894414469599724,-388845.2152323900372721255 7246914.8814502703025937080,-392370.3695473405532538891 7251272.8605816522613167763,-395957.3061796791735105217 7255525.2811666997149586678,-402203.5541275819996371865 7263241.4929156471043825150,-404058.9160806335275992751 7266542.3048335723578929901,-404553.6198977176682092249 7267714.0249515157192945480,-404460.8907618884695693851 7268406.0951544297859072685,-397317.8529961553285829723 7292788.3670299844816327095,-381733.1242850980488583446 7326150.4946560785174369812,-377558.6433803490945138037 7337912.6026934264227747917,-376383.5548355370992794633 7339470.1102552358061075211,-366457.6411194635438732803 7349637.6401906181126832962,-365870.0968470561201684177 7350014.7357774116098880768,-364014.7348940045922063291 7350498.0494048753753304482,-357180.9426736971363425255 7352059.6247507967054843903,-355634.9375855597318150103 7352167.5918185878545045853,-355047.3933131523081101477 7351898.3553642285987734795,-354552.5781765769352205098 7351413.4043160993605852127,-354367.0085854243952780962 7350659.2898099292069673538,-353872.3047683374024927616 7350337.0061289258301258087,-349141.2264096250291913748 7348561.4872619006782770157,-348368.2238655562978237867 7348561.4872619006782770157,-345461.4493219630676321685 7349044.9045187048614025116,-336896.0824223656672984362 7351305.4476988706737756729,-336741.4596496514277532697 7352059.6247507967054843903,-336525.0545595490257255733 7356152.7590438164770603180,-337421.7330578909022733569 7357068.5847051348537206650,-338565.8747842646553181112 7357768.6180988922715187073,-339638.6607170382048934698 7357959.4943135073408484459,-341070.5633271117694675922 7357068.5847051348537206650,-341627.1607810780988074839 7356852.5161710949614644051,-349945.1757721348549239337 7355075.5119649851694703102,-350687.3428172536659985781 7355075.5119649851694703102,-374095.3827022807672619820 7356313.9225882794708013535,-375857.9042000090121291578 7356529.9761815508827567101,-377434.9674260788015089929 7356852.5161710949614644051,-383526.5926012673298828304 7359223.5853684665635228157,-397534.2580862577306106687 7360086.4123517451807856560,-398554.7238583611906506121 7356368.6142240101471543312,-398647.4529941903892904520 7355506.0050766775384545326,-402512.7996730048907920718 7337697.0645915912464261055,-403966.0756253130966797471 7338126.5988389421254396439,-410645.2450729091069661081 7338824.9327172618359327316,-411449.1944354160805232823 7338824.9327172618359327316,-424374.6118309149169363081 7332592.8358689714223146439,-430775.4825515291304327548 7324165.3088273415341973305,-440113.9633146864944137633 7316871.5715599860996007919,-441381.7809953305986709893 7316443.2056702664121985435,-442989.7910398386302404106 7316443.2056702664121985435,-455915.2084353374084457755 7316925.9872876051813364029,-460398.9348855091957375407 7318265.8529948890209197998,-484054.2153595885611139238 7334311.5376756498590111732,-484549.0304961639340035617 7334795.4259652942419052124,-487146.5594943342730402946 7338449.9260353958234190941,-487672.2101298595080152154 7339524.6838094489648938179,-487795.8860841326531954110 7340277.7217413242906332016,-487517.5873571481206454337 7342051.3544057011604309082,-487548.5341755892150104046 7342804.6364329317584633827,-487734.1037667416967451572 7343450.1433990737423300743,-488135.9671285054064355791 7343933.2437204523012042046,-489032.7569463356630876660 7344041.0982611542567610741,-489589.3544003019924275577 7343718.8945121262222528458,-492403.2884885747916996479 7339201.3120461301878094673,-492650.6403971154359169304 7338610.7239055326208472252,-493052.6150783703196793795 7337105.0426507415249943733,-493021.6682599292253144085 7335440.2702085031196475029,-492928.9391241000266745687 7334795.4259652942419052124,-491042.7416720986366271973 7328136.1856125276535749435,-485322.1443597238394431770 7317569.5495664961636066437,-483806.8634510450647212565 7305998.1589573910459876060,-483714.1343152130139060318 7305249.9621166158467531204,-484023.3798606386990286410 7303697.7255198266357183456,-484889.2228600278613157570 7301556.6574587998911738396,-485909.6886321313213557005 7300059.6450568856671452522,-487115.6126758931786753237 7299364.7697884002700448036,-488383.3190370460506528616 7299150.2931946692988276482,-490053.2227184362709522247 7299417.7159931827336549759,-500535.7338884761556982994 7303643.4018597202375531197,-501617.9819779677200131118 7304392.9861292643472552299,-507276.7969729543547146022 7311085.0218444764614105225,-507493.2020630567567422986 7311675.0732801137492060661,-507616.8780173270497471094 7313174.3129446320235729218,-507802.4476084795896895230 7313817.3732218593358993530,-511915.1461958375293761492 7319337.8339232783764600754,-521068.0573678423534147441 7323092.6648230087012052536,-527716.2799969972111284733 7327331.1827524537220597267,-534024.4215817794902250171 7333398.1884740386158227921,-534921.1000801213085651398 7334366.0751424813643097878,-535910.6190337808802723885 7335064.0723519250750541687,-537178.4367144248681142926 7335601.1994602633640170097,-540146.8822559206746518612 7336407.0562121253460645676,-544785.2314788039075210690 7333184.3220268506556749344,-547722.9528408382320776582 7330766.6467525400221347809,-548619.6313391772564500570 7329853.9057584553956985474,-549423.5807016870239749551 7328888.1247963691130280495,-552268.4616083981236442924 7323628.9683988010510802269,-552361.1907442301744595170 7322716.8687183316797018051,-552144.6743346365401521325 7321912.6177066862583160400,-550444.0464737876318395138 7316871.5715599860996007919,-546795.2162045668810606003 7307068.6501135462895035744,-545465.6162065317621454597 7304072.6075961450114846230,-544445.1504344283603131771 7303268.7329980144277215004,-543795.7125251415418460965 7302948.2079297648742794991,-542868.0872083613649010658 7302145.9897665642201900482,-542156.8669816836481913924 7301021.8672180408611893654,-541940.4618915813043713570 7300487.1135643599554896355,-540548.9682566639967262745 7290651.8225432001054286957,-540858.2138020896818488836 7289957.7827585916966199875,-541476.5935734441736713052 7289850.6726551959291100502,-547970.3047493788180872798 7292414.2180133098736405373,-549083.4996573114767670631 7293162.7263257186859846115,-550042.0717925328062847257 7294018.4720156816765666008,-550846.0211550425738096237 7294872.7719973642379045486,-551247.9958362975157797337 7295514.3279841132462024689,-552206.5679715187288820744 7298722.8977807499468326569,-552515.8135169416200369596 7300059.6450568856671452522,-552546.7603353826561942697 7300754.5821008244529366493,-552330.2439257890218868852 7301449.3883516052737832069,-551526.2945632792543619871 7302520.6048074951395392418,-551124.3198820272227749228 7303910.9784320062026381493,-551804.4819707726128399372 7308246.4561829632148146629,-552144.6743346365401521325 7308783.0904304319992661476,-557154.0514203335624188185 7316067.9224346978589892387,-560524.5829625726910308003 7319713.4673932855948805809,-564915.4689574239309877157 7324272.8903694152832031250,-568007.8130921697011217475 7327331.1827524537220597267,-569213.7371359316166490316 7328780.4796105446293950081,-569955.9041810504859313369 7329800.7528070313856005669,-571285.5041790854884311557 7331895.0597502989694476128,-572429.5345859680091962218 7334203.8176293242722749710,-575521.7674012253992259502 7340760.6255087079480290413,-576233.0989473944064229727 7342588.9630549391731619835,-576573.1799917670432478189 7344041.0982611542567610741,-577129.7774457334307953715 7346676.9306908566504716873,-577268.8154897345229983330 7349194.1014714818447828293,-577253.4534000037238001823 7349798.8627920215949416161,-576449.5040374967502430081 7359709.0229227598756551743,-575985.6357193596195429564 7361972.6634850092232227325,-574656.0357213246170431376 7363591.2837319280952215195,-573542.8408133919583633542 7364130.3809066936373710632,-568286.0004996630595996976 7366503.6677023600786924362,-562627.2968241675989702344 7370927.3331355582922697067,-562503.6208698945119976997 7371845.0929058594629168510,-558112.7348750460660085082 7387460.6773691773414611816,-557865.3829665025696158409 7388165.0084091825410723686,-556504.7248305380344390869 7389462.0636020163074135780,-553690.7907422651769593358 7392006.5564168058335781097,-549361.6870648048352450132 7396606.9521838556975126266,-541940.4618915813043713570 7404517.6976240603253245354,-541631.2163461584132164717 7405114.6755043659359216690,-541445.7580744943115860224 7405872.7220549043267965317,-539652.1784388336818665266 7414335.5076649766415357590,-538693.7176231037592515349 7422426.8887438448145985603,-538600.8771677832119166851 7424111.9521764740347862244,-538291.6316223575267940760 7424709.0696901064366102219,-528798.5280864888336509466 7441684.9049287764355540276,-528396.5534052366856485605 7442229.6465787151828408241,-526417.5154979120707139373 7443863.5129646947607398033,-524716.8876370631624013186 7444789.7393481815233826637,-521253.6269589948933571577 7446533.6286841230466961861,-518377.7992338399053551257 7447078.5152014587074518204,-517728.4726440443191677332 7447297.1084670927375555038,-516677.0600535026169382036 7448005.1261356342583894730,-516275.0853722477331757545 7448496.3416110388934612274,-513615.8853761776117607951 7456840.9701854353770613670,-513553.9917392954230308533 7458533.3387779006734490395,-514512.5638745166943408549 7461918.0069793518632650375,-514728.9689646190381608903 7462573.4907166687771677971,-518687.0447792655904777348 7471700.5883307904005050659,-521624.6548218086827546358 7478374.5869022384285926819,-521933.9003672315157018602 7479029.8906536586582660675,-522645.1205939092906191945 7480070.2945414464920759201,-523046.9839556729421019554 7480563.3530168291181325912,-524067.4497277764021418989 7481384.3986715553328394890,-526479.4091347942594438791 7482698.7262673750519752502,-530901.2419480838580057025 7484834.6896913861855864525,-535848.8367163898656144738 7488176.6438662633299827576,-541105.5657106303842738271 7493438.3433833224698901176,-542775.3580725293140858412 7495357.3164986893534660339,-545898.5377062249463051558 7496894.0233288249000906944,-547228.0263847715687006712 7498319.6308487802743911743,-547351.8136585331521928310 7499856.9274073811247944832,-543362.7910254455637186766 7544934.3906168844550848007,-543208.2795722252922132611 7545706.2707003252580761909,-542806.3048909704666584730 7546259.7905436092987656593,-537011.4574782375711947680 7548368.4992670519277453423,-534043.0119367417646571994 7549030.5716537795960903168,-533504.8935182493878528476 7548932.1518849516287446022,-533152.4560103978728875518 7548733.3279079142957925797,-532354.6292198804439976811 7547871.8829430760815739632,-528953.1508592030731961131 7547694.3573386920616030693,-524685.9408186221262440085 7545375.7677722154185175896,-523881.9914561150944791734 7545210.4222629889845848083,-508761.0197437008027918637 7542725.6989457299932837486,-503504.2907494603423401713 7542230.2410548646003007889,-498896.9996645094943232834 7542725.6989457299932837486,-499731.7845259663299657404 7543720.0881354566663503647,-501061.4958434926229529083 7544326.5199617613106966019,-515100.1081469241180457175 7547583.6295554768294095993,-531727.9004867146722972393 7554389.6878355210646986961,-535463.4486392636317759752 7553722.5784520916640758514,-540412.9358389149419963360 7555725.6721437172964215279,-541720.3832582823233678937 7564745.5551264593377709389,-540973.3181555690243840218 7572438.5017515700310468674,-539479.0766306512523442507 7575617.7549175210297107697,-537517.9611613443121314049 7580976.2699969690293073654,-541813.7803110589738935232 7579468.4106733193621039391,-543401.4188887506024912000 7572606.0290160523727536201,-545269.1373052808921784163 7554723.5626798011362552643,-546950.1729357491713017225 7550718.4179797386750578880,-548631.2085662202443927526 7545214.9930646773427724838,-552361.1907442301744595170 7534563.4486953206360340118,-554247.3881962315645068884 7531806.7938464414328336716,-554927.7729239594191312790 7531531.4404878914356231689,-555638.8818311459617689252 7531476.8867755979299545288,-557958.0007828433299437165 7531973.2405976857990026474,-559473.2816915222210809588 7532524.1871358714997768402,-568842.7092731178272515535 7539140.4077039826661348343,-575892.9065835303626954556 7544326.5199617613106966019,-579542.7387281695846468210 7542881.0575838973745703697,-581597.2512502502650022507 7538882.0674366783350706100,-582157.6335669015534222126 7533387.0918295802548527718,-579758.1419428536901250482 7527510.9738995777443051338,-579170.5976704462664201856 7525363.3668284397572278976,-579356.1672615987481549382 7524481.7385187903419137001,-579758.1419428536901250482 7523931.1683868365362286568,-580500.3089879725594073534 7523820.9798069568350911140,-589158.4050233991583809257 7528116.1310436213389039040,-590364.4403866550419479609 7528722.7238455498591065407,-590797.2505668598460033536 7529382.5146479764953255653,-594291.5693828605581074953 7537981.6177929006516933441,-594477.1389740130398422480 7538699.5863360874354839325,-594507.9744729629019275308 7539526.8421414215117692947,-594538.9212914011441171169 7543720.0881354566663503647,-591075.5492938415845856071 7558912.3560458663851022720,-591038.4799034082097932696 7559476.5824041925370693207,-590945.7507675761589780450 7559908.8388476064428687096,-587884.4647707627154886723 7565285.9010660173371434212,-587049.5685898117953911424 7566713.4577158931642770767,-579170.5976704462664201856 7581785.9207013463601469994,-578706.7293523119296878576 7582340.4458873094990849495,-564637.2815499305725097656 7595721.0870233085006475449,-553721.7375607063295319676 7605005.9760906687006354332,-550726.7979804017813876271 7608345.4135369891300797462,-548032.0870667698327451944 7611351.3489257842302322388,-547475.4896128034451976418 7612576.5869784224778413773,-547753.7883397880941629410 7613243.4136294042691588402,-548526.9022033452056348324 7613133.3411337248980998993,-560184.5019181999377906322 7604839.3427863111719489098,-579510.6787148189032450318 7585779.9151765964925289154,-583190.5671219720970839262 7582119.5895048100501298904,-585416.8456183490343391895 7579734.1062304284423589706,-588595.6849974404321983457 7575166.4594591846689581871,-590655.0955771155422553420 7572207.9582950612530112267,-591861.0196208803681656718 7569449.3521843859925866127,-594328.5274538027588278055 7565297.4575516693294048309,-596239.6604717412265017629 7563371.5499128662049770355,-600104.8958310673478990793 7558415.6560810878872871399,-600878.0096946244593709707 7558304.7739343997091054916,-601589.1186018137959763408 7558359.5180620308965444565,-603722.7792818470625206828 7559077.7971014725044369698,-604341.1590532043483108282 7559410.4817635063081979752,-604650.4045986272394657135 7559963.3952792305499315262,-603227.9641452716896310449 7539857.0941298268735408783,-598898.8604678113479167223 7508862.2448687991127371788,-601589.1186018137959763408 7506114.7613255186006426811,-606072.8450519855832681060 7500569.7742751771584153175,-606412.9260963583365082741 7499966.7742423238232731819,-607062.3640056450385600328 7496345.6157262232154607773,-607464.2273674088064581156 7495851.5490331957116723061,-608793.9386849350994452834 7492726.1545226909220218658,-610742.0297738186782225966 7487682.9021548954769968987,-618401.7012963209999725223 7442711.2787298168987035751,-616184.3283592087682336569 7439561.0136069916188716888,-615906.0296322270296514034 7439016.6555541269481182098,-614019.8321802256396040320 7433083.3396282317116856575,-613896.1562259553465992212 7432376.4882230628281831741,-614823.7815427355235442519 7430526.9147069631144404411,-616184.3283592087682336569 7428243.0212605353444814682,-619709.4826741593424230814 7423568.6777871046215295792,-622925.3914436870254576206 7421449.5475499229505658150,-638726.6365643379976972938 7418788.0563712492585182190,-640365.4821077958913519979 7418732.9238223517313599586,-641849.7048785451333969831 7419059.6192177310585975647,-643148.3580581390997394919 7419656.1524058720096945763,-644075.9833749192766845226 7420579.6864177007228136063,-644292.4997845129109919071 7421285.4633120549842715263,-644539.9630125447874888778 7422752.1748917521908879280,-645127.5072849522111937404 7432104.4586815088987350464,-645189.2896023431094363332 7434661.4039742629975080490,-645096.4491470198845490813 7437165.3010793505236506462,-644941.9376937996130436659 7437981.7554277461022138596,-644292.4997845129109919071 7439179.5473105553537607193,-641107.5378334262641146779 7443318.8525478057563304901,-637582.4948379641864448786 7446044.1072282148525118828,-636871.1632917951792478561 7447078.5152014587074518204,-636623.8113832544768229127 7447788.0827978411689400673,-636345.6239757612347602844 7449477.0995066165924072266,-636531.0822474224260076880 7462465.3537021325901150703,-634706.6671128120506182313 7476951.2756828563287854195,-631088.8949815207161009312 7491354.2706713257357478142,-624255.1027612133184447885 7510510.2550652492791414261,-624718.8597598591586574912 7510785.0565796466544270515,-628676.8242550116265192628 7515018.5136068165302276611,-631428.9760258934693410993 7519417.6984367296099662781,-631645.3811159958131611347 7520078.0798785584047436714,-631929.9137344639748334885 7536007.3507909551262855530,-631985.5734798597404733300 7536470.6476539000868797302,-632059.7122607295168563724 7538423.1922929277643561363,-631837.0732791435439139605 7540872.3651104364544153214,-631391.7953159686876460910 7541567.9585998225957155228,-630557.0104545118520036340 7542361.9494845289736986160,-629499.3639724837848916650 7543488.2180768847465515137,-627130.8191668741637840867 7551672.0716077527031302452,-621348.3282176200300455093 7560738.1781936613842844963,-621008.1358537561027333140 7561345.9525353331118822098,-620111.4573554142843931913 7565827.8790837377309799194,-620729.9484462627442553639 7565995.0630998956039547920,-621472.0041718903230503201 7565884.0716371666640043259,-623110.9610348395071923733 7564832.2231024028733372688,-626234.0293490438489243388 7562508.4646861730143427849,-630018.8920360166812315583 7555141.8220489416271448135,-630204.4616271691629663110 7554776.6895123803988099098,-630965.1077077591326087713 7553317.9241124149411916733,-631206.2257248162059113383 7552986.2984898770228028297,-631651.6150074822362512350 7552921.0492999963462352753,-632022.6428702960256487131 7553119.1857457403093576431,-632226.6914969196077436209 7553450.8170948037877678871,-632282.3512423154897987843 7556269.9622159684076905251,-632263.8722068442730233073 7556701.0433939341455698013,-631837.0732791435439139605 7557961.5868970239534974098,-628089.2799826069967821240 7566991.0689534870907664299,-627007.1432126038707792759 7568762.5962103353813290596,-623543.7712150443112477660 7573636.8009063620120286942,-619307.5079929071944206953 7581009.0137706603854894638,-614607.2651331418892368674 7590887.6317691532894968987,-614359.9132245983928442001 7591720.5208720434457063675,-613555.9638620885089039803 7594665.1462415019050240517,-613586.9106805296614766121 7595498.2452233731746673584,-614112.5613160548964515328 7595830.9100026590749621391,-614792.9460437856614589691 7595553.2549406941980123520,-615844.2473148361314088106 7594831.3587708855047821999,-618812.6928563290275633335 7592220.7775808572769165039,-620482.5965377192478626966 7590332.4891860084608197212,-623141.7965337893692776561 7590277.5168613130226731300,-623018.1205795190762728453 7608901.8606251943856477737,-622987.2850805692141875625 7610571.4441312914714217186,-620451.5383997869212180376 7623330.6740666655823588371,-619956.8345827028388157487 7624781.3151242770254611969,-615163.9739065994508564472 7633152.0074393870308995247,-615102.0802697172621265054 7634828.4906064746901392937,-614978.4043154468527063727 7635609.2590998029336333275,-614607.2651331418892368674 7636336.4125423133373260498,-605268.8956894756993278861 7642952.9402038650587201118,-604267.0202723346883431077 7644261.3506676610559225082,-603413.5337364242877811193 7645100.8640749985352158546,-593506.0990558217745274305 7645737.5061285542324185371,-592634.1334844402736052871 7645569.9848064649850130081,-590945.7507675761589780450 7644428.8433526214212179184,-583499.7013479066081345081 7645782.8100005267187952995,-582850.3747581081697717309 7645615.4890336738899350166,-581180.5823962091235443950 7645670.6582383047789335251,-580036.4406698354287073016 7646286.8084915485233068466,-579046.9217161758570000529 7647125.5371557772159576416,-578119.2963993957964703441 7648187.3202392403036355972,-571687.4788603404304012656 7655576.1483199028298258781,-570017.6864984413841739297 7657705.2339517800137400627,-568595.3573645771248266101 7660170.3822070378810167313,-564235.1955491844564676285 7670151.6414698222652077675,-563987.8436406437540426850 7670825.3199614593759179115,-564328.0360045077977702022 7671273.7985344389453530312,-565070.2030496265506371856 7671161.2723834570497274399,-565595.7423656606115400791 7670655.6322282897308468819,-566338.0207302706548944116 7669590.7257001409307122231,-567265.6460470508318394423 7667626.3301212359219789505,-568162.3245453898562118411 7664542.7202141098678112030,-569461.0890444751130416989 7661178.6958307139575481415,-577871.9444908523000776768 7650929.2628415934741497040,-579201.5444888873025774956 7649363.0664271879941225052,-580469.3621695314068347216 7648915.6693822499364614487,-582077.2608945482643321157 7648691.4770984323695302010,-582788.4811212287750095129 7648748.2799130156636238098,-597600.2072882173815742135 7653426.8951496751978993416,-600976.8614024488488212228 7654300.2713927906006574631,-598584.7168647940270602703 7662862.7428454048931598663,-597063.4247036112938076258 7661618.9416700415313243866,-594855.5139232198707759380 7661719.6247330401092767715,-594336.0971791782649233937 7661786.6129962932318449020,-593593.9301340593956410885 7662122.9755862103775143623,-584767.5190285505959764123 7668749.5301919383928179741,-584334.5975288546178489923 7669253.4285087063908576965,-584025.3519834318431094289 7669871.2794329514726996422,-583932.6228475997922942042 7670768.1509353239089250565,-584767.5190285505959764123 7670768.1509353239089250565,-591372.4383757885079830885 7670610.9888196662068367004,-591724.9872031311970204115 7670342.1238500587642192841,-592504.2236386835575103760 7669466.7117092106491327286,-593747.2170728817582130432 7668557.6738326698541641235,-594396.6549821684602648020 7668120.8651574626564979553,-595157.3010627584299072623 7667784.2386079188436269760,-595713.8985167248174548149 7667784.2386079188436269760,-596530.3156622046371921897 7668120.8651574626564979553,-598478.4067510853055864573 7669938.3391573987901210785,-597995.9480779886944219470 7671183.4946518354117870331,-599239.0528316752752289176 7675371.7275347234681248665,-591632.2580672991462051868 7690264.7426660489290952682,-583561.5949847858864814043 7701524.7897592484951019287,-576233.0989473944064229727 7711221.1246676445007324219,-571501.9092691878322511911 7717824.3887505372986197472,-571161.8282248151954263449 7718446.1837435495108366013,-570141.3624527116771787405 7721947.3952255463227629662,-569955.9041810504859313369 7722738.7454501269385218620,-573790.1927219354547560215 7719179.8410347662866115570,-577624.5925823088036850095 7714549.7747569298371672630,-582664.8051669556880369782 7707837.9005845338106155396,-597043.6098342511104419827 7691334.1557144531980156898,-619369.4016297865891829133 7666785.1488293502479791641,-622585.1990798230981454253 7663477.0502433264628052711,-623605.6648519237060099840 7662524.1280005862936377525,-631954.6266614214982837439 7657369.8722159853205084801,-633377.0671147770481184125 7660450.5904816919937729836,-634242.7987946749199181795 7661347.9715940030291676521,-644601.7453299356857314706 7666281.4131772564724087715,-648621.6034619703423231840 7667122.5390420444309711456,-655022.4741825846722349524 7669365.7246035486459732056,-656290.2918632287764921784 7670094.6794743090867996216,-660186.4740409930236637592 7672676.3729850547388195992,-661547.1321769604692235589 7674079.4069835711270570755,-667453.0764415052253752947 7681099.8598617259413003922,-668721.0054416406201198697 7682729.6276842541992664337,-669184.8737597776344045997 7683966.1445693038403987885,-669153.8156218453077599406 7684753.5647499328479170799,-668844.6813959137070924044 7686384.3149067135527729988,-668009.7852149627869948745 7687508.6565122948959469795,-667112.9953971325885504484 7688409.2208956573158502579,-666525.4511247250484302640 7688801.6313156671822071075,-665103.1219908607890829444 7689195.4793882267549633980,-654849.3723743992159143090 7690264.7426660489290952682,-634428.3683858275180682540 7695555.5464410297572612762,-628800.6115287760039791465 7695048.4914979264140129089,-624780.6420772472629323602 7695385.5110735064372420311,-618503.4473109061364084482 7696118.5749691501259803772,-617266.5764487031847238541 7696681.4423376144841313362,-617483.0928582940250635147 7697412.8062782948836684227,-617977.9079948721919208765 7697862.8488564118742942810,-638912.0948359991889446974 7701411.8169762650504708290,-641447.6188777990173548460 7701467.5932758646085858345,-642282.6263782383175566792 7701243.2736175647005438805,-643364.8744677299400791526 7700622.6767362505197525024,-644849.0972384791821241379 7699327.6982208164408802986,-648572.1776080582058057189 7697333.9501986689865589142,-650334.6991057891864329576 7695340.7228882731869816780,-651076.7548314168816432357 7695002.8946825573220849037,-658918.7676798402098938823 7693753.0400504823774099350,-689222.2707830744329839945 7694935.6145458146929740906,-691201.3086903991643339396 7695837.0555134583264589310,-692932.9946891774889081717 7697920.2210562909021973610,-693613.2680974169634282589 7699046.2631236556917428970,-694077.1364155513001605868 7701130.3050026390701532364,-694107.9719145011622458696 7702088.0560450982302427292,-694077.1364155513001605868 7702933.1348811537027359009,-693829.7845070078037679195 7703609.5088090514764189720,-689500.6808295502560213208 7708739.2431454947218298912,-689098.5948288041399791837 7709191.5786064267158508301,-683501.6734706996940076351 7710770.0903161438181996346,-664237.2789914688328281045 7715340.3573290472850203514,-662752.9449012310942634940 7715622.5974873211234807968,-662196.3474472648231312633 7715284.6826785169541835785,-661268.7221304846461862326 7714380.9408367108553647995,-657341.7044537704205140471 7710543.8724155491217970848,-653167.1122295331442728639 7707724.8342732694000005722,-651590.0490034633548930287 7707555.9483984513208270073,-650755.2641420065192505717 7708570.5378203485161066055,-645498.5351477660005912185 7717146.7529787486419081688,-646271.5376918347319588065 7718219.7379818670451641083,-650044.0439153286861255765 7720082.5254447590559720993,-651188.1856416997034102678 7720931.0242721689864993095,-651930.2413673300761729479 7722003.1185607267543673515,-652177.7045953619526699185 7722567.8995877066627144814,-652208.6514138031052425504 7723528.5505657941102981567,-652115.8109584797639399767 7724376.0091261854395270348,-651713.8362772277323529124 7724998.3388804411515593529,-651219.0211406495654955506 7725336.6846215771511197090,-644076.0946944104507565498 7727395.3631055718287825584,-640458.2112436279421672225 7727994.7370103215798735619,-639530.5859268477652221918 7727881.3713351078331470490,-638200.8746093214722350240 7727316.1963596548885107040,-636531.0822474224260076880 7727371.9589690761640667915,-630593.9685254540527239442 7732517.9178758235648274422,-630161.0470257580745965242 7733367.8027516771107912064,-630408.5102537928614765406 7733933.4269130388274788857,-631459.9228443345054984093 7734837.8610719582065939903,-638850.3125186110846698284 7738404.0893538175150752068,-639716.0441985090728849173 7738460.1370096830651164055,-642035.2744696948211640120 7738177.0496157137677073479,-647663.0313267491292208433 7736762.1780209625139832497,-652919.7603209896478801966 7735517.0721114641055464745,-654404.0944112273864448071 7735347.7729902314022183418,-655331.7197280074469745159 7735517.0721114641055464745,-658702.2512702465755864978 7736875.6759526571258902550,-659227.9019057718105614185 7737272.2188107650727033615,-648961.6845063430955633521 7760288.7283806502819061279,-648095.9528264451073482633 7761366.3798615131527185440,-646333.3200092256302013993 7762445.6152568906545639038,-645003.8313306790078058839 7762901.1521260682493448257,-641583.7626150393625721335 7762207.8391226036474108696,-640210.8593350844457745552 7762514.3120678337290883064,-638745.1155998090980574489 7762616.3361555505543947220,-638411.1571274314774200320 7762377.7368747051805257797,-638207.1085008051013574004 7762072.0872090365737676620,-637743.2401826707646250725 7760845.3187552774325013161,-637001.0731375518953427672 7759448.0945486947894096375,-634211.9632957250578328967 7757904.6196566373109817505,-631367.0823890140745788813 7755974.9503435604274272919,-630192.1051636904012411833 7755236.3097380781546235085,-628831.5583472142461687326 7754555.3368382360786199570,-628212.9559368772897869349 7754329.2444388270378112793,-626697.8976671809796243906 7754612.9332659207284450531,-621070.0294906354974955320 7756031.1275500915944576263,-619616.7535383300855755806 7756484.6460112733766436577,-616493.5739046344533562660 7757960.8111166683956980705,-614916.6219980559544637799 7758869.7401065547019243240,-614514.5359973098384216428 7759379.6298020500689744949,-614731.0524069034727290273 7760061.0347683196887373924,-615565.9485878542764112353 7760004.8277892693877220154,-617544.9864951762137934566 7759153.3939628023654222488,-618534.5054488385794684291 7758300.6250465279445052147,-619956.8345827028388157487 7758017.0029918318614363670,-625182.7280779933789744973 7756995.8059862777590751648,-626883.4672583335777744651 7757051.9907327210530638695,-629388.1558011806337162852 7758472.0703078443184494972,-642715.5478779342956840992 7769435.4249468790367245674,-643890.5251032579690217972 7770230.0911059072241187096,-644230.7174671218963339925 7770742.3806787393987178802,-644447.1225572242401540279 7771481.0963116846978664398,-644385.2289203420514240861 7772277.6141412789002060890,-644075.9833749192766845226 7774041.0589115936309099197,-643272.0340124093927443027 7776087.9705479899421334267,-642529.9782867818139493465 7777169.2983204172924160957,-637273.2492925412952899933 7783260.7100063404068350792,-633587.3496328842593356967 7782827.5264575220644474030,-629282.9588823814410716295 7783853.6354435244575142860,-628874.7503096428699791431 7783920.8857878400012850761,-626110.3533947735559195280 7783169.2740341397002339363,-625219.7974684268701821566 7782349.6796530755236744881,-624681.7903694228734821081 7781290.1600545877590775490,-620884.4598994828993454576 7780072.1718377582728862762,-618874.5864932112162932754 7779048.2127118976786732674,-616802.7081305660540238023 7778250.9848882425576448441,-615967.9232691064244136214 7778080.5270859887823462486,-614267.1840887691359966993 7778136.8678835537284612656,-601094.4147847268031910062 7781779.6237544631585478783,-601218.0907390000065788627 7782577.2220012359321117401,-601558.2831028639338910580 7783146.5177188478410243988,-601991.0932830687379464507 7783545.4806583048775792122,-603444.4805548653239384294 7784229.0558050507679581642,-605980.0045966622419655323 7784343.2643696963787078857,-607495.2855053411331027746 7784000.6438278378918766975,-610216.4904577818233519793 7783090.3447639672085642815,-617854.2320405989885330200 7783944.2595225675031542778,-627798.7361116348765790462 7792058.6766033601015806198,-628522.2014823001809418201 7792469.8080844217911362648,-628763.4308188485447317362 7792811.1688131410628557205,-628911.8197000763611868024 7793222.3410503147169947624,-628856.2712741717696189880 7793804.1375226667150855064,-628522.2014823001809418201 7808301.6504280986264348030,-624799.2324322096537798643 7809284.9224720466881990433,-624298.2947236419422551990 7809867.9519115211442112923,-619010.6189109602710232139 7812371.1263486817479133606,-617841.7642576318467035890 7812783.3599924026057124138,-617470.7363948180573061109 7812645.2602546168491244316,-616802.7081305660540238023 7811067.2737040109932422638,-616542.9997585467062890530 7810690.5234388047829270363,-611669.6550905987387523055 7809102.0406789267435669899,-610154.5968209024285897613 7807559.3439071569591760635,-609196.0246856812154874206 7806817.3156746942549943924,-606660.2780049017164856195 7805502.4915853813290596008,-604279.3767358134500682354 7804816.9980592103675007820,-603568.1565091356169432402 7805160.5595074025914072990,-601805.6350114046363160014 7807159.3224412053823471069,-601589.1186018137959763408 7807788.4659607233479619026,-613667.2833528829505667090 7817038.3112460924312472343,-620179.4735642888117581606 7818410.7478845994919538498,-623260.5744304652325809002 7827969.2937268922105431557,-620922.8651238083839416504 7828243.9923413861542940140,-620440.5177702001528814435 7828381.5516211669892072678,-614545.4828157508745789528 7833885.1464064409956336021,-611422.3031820552423596382 7836293.2464565103873610497,-608113.6652766984188929200 7839620.5575723936781287193,-607680.7437770023243501782 7840252.4017933402210474014,-607000.4703687657602131367 7841515.0088626490905880928,-606536.6020506286295130849 7844958.0876820059493184090,-606505.7665516787674278021 7845934.3455576533451676369,-606722.1716417811112478375 7846508.4183434061706066132,-607557.0678227320313453674 7846795.3676944626495242119,-608051.8829593074042350054 7846451.5672775181010365486,-612628.3385453084483742714 7842547.4375017834827303886,-627013.2657845990033820271 7836236.4720418369397521019,-640210.8593350844457745552 7830618.5294065307825803757,-642251.5682403087848797441 7829930.7611689064651727676,-643210.2516950184945017099 7829987.4883399996906518936,-644385.2289203420514240861 7830733.4385857777670025826,-647539.3553724788362160325 7834286.7872466091066598892,-647786.7072810223326086998 7834859.9775811564177274704,-652332.3273680733982473612 7852827.9567057034000754356,-653105.3299121421296149492 7856334.5611695535480976105,-653600.0337292291224002838 7858692.8719285270199179649,-653507.3045933970715850592 7861281.0070635043084621429,-653352.6818206856260076165 7862202.4039036324247717857,-651342.6970949226524680853 7872913.1762971458956599236,-651188.1856416997034102678 7873720.4263095399364829063,-650074.9907337670447304845 7878101.7436213130131363869,-648281.4110981064150109887 7879254.8272750088945031166,-647601.1376898697344586253 7879543.2295036436989903450,-646859.0819642421556636691 7879484.4684734027832746506,-643086.5757407482014968991 7877525.2677725153043866158,-640118.0188797640148550272 7873893.0491695394739508629,-640056.1252428817097097635 7872164.7161779273301362991,-629573.6140728419413790107 7864964.1596209304407238960,-629017.0166188755538314581 7864676.3131026867777109146,-628151.1736194863915443420 7864791.7405201056972146034,-625615.6495776894735172391 7868131.6093980763107538223,-615689.6245421246858313680 7868879.8776304749771952629,-613555.9638620885089039803 7869571.1948427958413958549,-613215.8828177158720791340 7870205.5454059662297368050,-614236.2372703280998393893 7872854.6742497216910123825,-614669.1587700240779668093 7873258.1816226178780198097,-616184.3283592087682336569 7873834.5395373199135065079,-628244.0140748068224638700 7873834.5395373199135065079,-629357.0976632511010393500 7873028.7295439839363098145,-629666.3432086739921942353 7872509.6873413408175110817,-630346.6166169105563312769 7872278.5984832206740975380,-631088.8949815207161009312 7872509.6873413408175110817,-631645.3811159958131611347 7872970.2266021808609366417,-634026.3937045753700658679 7875333.7323903553187847137,-646797.1883273599669337273 7891606.5970635982230305672,-646920.8642816302599385381 7892358.6498228590935468674,-647601.1376898697344586253 7929017.7337389001622796059,-647508.5198735289741307497 7929947.4355408484116196632,-646055.1326017323881387711 7936453.7502605682238936424,-645560.3174651570152491331 7936918.2089397581294178963,-641973.3808328154264017940 7938952.1220390647649765015,-640643.6695152892498299479 7939533.3110576542094349861,-636407.4062931521330028772 7939940.4002927672117948532,-634143.8357673621503636241 7936511.0737917013466358185,-633494.5091775637120008469 7936057.7277084160596132278,-632919.3213686350500211120 7935500.0260215764865279198,-632659.7243161040823906660 7935117.0222576120868325233,-632566.8838607836514711380 7934663.7600552150979638100,-630563.1330265041906386614 7923679.3942342167720198631,-630253.8874810814158990979 7922519.5037977732717990875,-627965.6040283339098095894 7919967.7048245184123516083,-625027.9939857907593250275 7918112.3364293584600090981,-624193.2091243311297148466 7918169.7293285047635436058,-623667.5584888058947399259 7918575.8748366609215736389,-621410.2218544994248077273 7929076.6727590663358569145,-621255.4877622968051582575 7929889.9528987463563680649,-621224.6522633469430729747 7931689.0539463944733142853,-621472.0041718903230503201 7932386.1485914569348096848,-623110.9610348395071923733 7934535.3372363187372684479,-623574.7180334853474050760 7935058.6634809840470552444,-626234.0293490438489243388 7936337.2216532435268163681,-627248.2612296637380495667 7938883.4789800271391868591,-628287.3173567267367616296 7941184.4087877087295055389,-628361.4561375963967293501 7941673.2349743219092488289,-628324.2754276689374819398 7942860.1574493637308478355,-628231.5462918396806344390 7943383.2248314805328845978,-628336.6318911476992070675 7946281.0648511294275522232,-627347.2242569765076041222 7948200.7963023874908685684,-626512.3280760284978896379 7949307.2264501005411148071,-624626.1306240271078422666 7951286.8429214227944612503,-607526.1210042909951880574 7937732.3360116407275199890,-604959.6501440529245883226 7945990.3089706096798181534,-604526.7286443569464609027 7946514.4317998429760336876,-603908.2375535083701834083 7946862.8197974404320120811,-594291.5693828605581074953 7950704.7460147030651569366,-580809.5545333953341469169 7946514.4317998429760336876,-570790.8003620014060288668 7939940.4002927672117948532,-569059.1143632201710715890 7937848.8862027619034051895,-568038.6485911195632070303 7936046.6402133628726005554,-567543.8334545441903173923 7939649.8891198858618736267,-577686.3748996997019276023 7956470.2657709643244743347,-578552.2178990889806300402 7957461.0562970871105790138,-579201.5444888873025774956 7957811.6297755213454365730,-585849.7671180422184988856 7961250.4654107410460710526,-600908.8451935743214562535 7973854.5646767467260360718,-603011.5590551693458110094 7976307.5482310401275753975,-606474.8197332376148551702 7981392.7043655235320329666,-607155.0931414770893752575 7982621.2739381669089198112,-607278.7690957474987953901 7983380.7848828043788671494,-607124.1463230360532179475 7984257.8544951751828193665,-606598.4956875108182430267 7985779.1793417986482381821,-605918.2222792713437229395 7987066.1763894865289330482,-605423.5184621871449053288 7987591.6951222568750381470,-604093.8071446608519181609 7988061.0070491367951035500,-603259.0222832040162757039 7987884.2743858611211180687,-602052.9869199481327086687 7987124.0974934734404087067,-599362.7287859484786167741 7984902.1427120426669716835,-598002.1819694723235443234 7983323.1031430726870894432,-592157.9087028243811801076 7980866.1458319267258048058,-591199.3365676031680777669 7980750.6109452126547694206,-589715.1137968567200005054 7981335.0378831140697002411,-588787.3771605853689834476 7982094.4188393997028470039,-588323.6201619423227384686 7982736.8375867586582899094,-587921.5341611960902810097 7983323.1031430726870894432,-587736.0758895347826182842 7984082.8955660937353968620,-587086.7492997364606708288 7991981.7917336644604802132,-587148.5316171273589134216 7998599.8071651859208941460,-588972.8354322465602308512 7998659.5048527289181947708,-589715.1137968567200005054 7999069.8074542861431837082,-600723.2756024218397215009 8018431.1903368635103106499,-601001.6856488976627588272 8019841.6262504709884524345,-601032.5211478475248441100 8020722.7544771777465939522,-600692.3287839834811165929 8021546.0100588807836174965,-600259.4072842875029891729 8022015.9684993848204612732,-599641.0275129302171990275 8022369.3560171527788043022,-597909.4528336402727290988 8022133.8330243593081831932,-597352.8553796738851815462 8021781.7261899784207344055,-579263.3268062783172354102 8020195.1228845063596963882,-564575.3879130483837798238 8023133.1108342297375202179,-565008.3094127443619072437 8023662.8540255660191178322,-565750.4764578632311895490 8024014.8375328881666064262,-570667.1244077309966087341 8027955.9372333325445652008,-574934.2231288179755210876 8035959.6835891595110297203,-576047.5293562418082728982 8041203.3859719289466738701,-575398.0914469521958380938 8044386.2614203300327062607,-568780.8156362384324893355 8074868.7607897184789180756,-557123.2159213837003335357 8099415.4077767068520188332,-556411.9956947058672085404 8099770.9718487747013568878,-531241.4343119477853178978 8095199.9599894564598798752,-530777.5659938134485855699 8094784.3578436682000756264,-523387.1763195369276218116 8085590.8264067573472857475,-520604.1890497052227146924 8078301.7823860766366124153,-523418.1231379780219867826 8075223.3722695205360651016,-525180.6446357091190293431 8071908.7092198366299271584,-530004.4521302535431459546 8062743.2399211293086409569,-530282.8621767265722155571 8061797.1660845065489411354,-530375.5913125586230307817 8061029.4598552510142326355,-529509.7483131694607436657 8060852.4880412127822637558,-527809.0091328292619436979 8062092.4423149190843105316,-519645.6169144868035800755 8070607.0664195213466882706,-512502.5791487537208013237 8078065.9861865304410457611,-511915.1461958375293761492 8079487.6984174475073814392,-511729.5766046849894337356 8080315.7336701834574341774,-511853.2525589553406462073 8082983.0015429081395268440,-512224.2804217691300436854 8084522.7927242331206798553,-512038.8221501078805886209 8085471.9587688622996211052,-511513.0601950914133340120 8086953.7778820274397730827,-510801.9512879048706963658 8088199.5612933477386832237,-509657.8095615311176516116 8088970.0614235829561948776,-508173.5867907846113666892 8089326.8369318321347236633,-507183.9565176310716196895 8089207.9098701998591423035,-501679.8756148499087430537 8087546.5450903130695223808,-498278.3972541697439737618 8086123.2997940145432949066,-494351.3795774555183015764 8084049.1083461018279194832,-493392.6961227458668872714 8083041.4553312528878450394,-493021.6682599292253144085 8081560.6246658340096473694,-476138.2863692779210396111 8080257.0879299435764551163,-454245.4160734384204261005 8084879.3560339370742440224,-434517.1532760704285465181 8085530.6454274784773588181,-428394.4699629495735280216 8085887.2568110255524516106,-421375.2194709808682091534 8087310.4571631234139204025,-420787.6751985733862966299 8087665.4457213776186108589,-415252.6474773511872626841 8092885.2754072621464729309,-413459.1791611818480305374 8095022.3914114953950047493,-412902.5817072155186906457 8095438.0067971935495734215,-407460.2831218225182965398 8098643.8299443535506725311,-398090.8555402240599505603 8099118.6540870815515518188,-395555.3314984271419234574 8096386.8503993386402726173,-375548.6586545861791819334 8093538.9718917291611433029,-374590.0865193649078719318 8093419.9777843467891216278,-374126.3295207218616269529 8093953.0092434845864772797,-373848.0307937372708693147 8094725.5984799610450863838,-373105.8637486184597946703 8097099.4171333312988281250,-373013.0232932979706674814 8097870.8354736315086483955,-373445.9447929911548271775 8098405.8947522128000855446,-375517.7118361450848169625 8100306.1672385381534695625,-373353.2156571618979796767 8107378.2808270510286092758,-353594.0060413556639105082 8104999.4823208879679441452,-336803.3532865336164832115 8104584.8323190966621041298,-336153.9153772468562237918 8104168.7077577663585543633,-335720.9938775508198887110 8103633.2355586560443043709,-335473.6419690073234960437 8101554.1788671789690852165,-340142.9380103316507302225 8063392.6046885224059224129,-340668.5886458568857051432 8061857.1564218187704682350,-346172.7808681320748291910 8047923.3855891823768615723,-347533.3276846053777262568 8045328.6519893314689397812,-356500.6692654576618224382 8032309.9628600981086492538,-357180.9426736971363425255 8031545.2727268235757946968,-371528.8005225515225902200 8025720.1701402561739087105,-373291.3220202797092497349 8025720.1701402561739087105,-374002.6535664487164467573 8025486.0238025076687335968,-374497.3573835357092320919 8025074.2742424802854657173,-376476.3952908575301989913 8023133.1108342297375202179,-380743.6053314356831833720 8018666.8085988098755478859,-383186.5115568946348503232 8015143.2477811425924301147,-383650.2685555405332706869 8013498.2314808340743184090,-384701.5698265910032205284 8010739.5907647507265210152,-385443.7368717098142951727 8009448.5301577858626842499,-387762.9671428956207819283 8005811.6842027688398957253,-390855.0886386617203243077 8002057.5707713356241583824,-391318.9569567959988489747 8001531.0388874337077140808,-396977.7719517826335504651 7996607.6954028932377696037,-410150.5412558221141807735 7986188.9895308613777160645,-428023.4421001357259228826 7971519.4539741594344377518,-444937.8821287222090177238 7956937.6391611676663160324,-446576.7276721801026724279 7951985.7540601119399070740,-444412.1201737028895877302 7948491.8471077596768736839,-444288.4442194325383752584 7946746.1300195800140500069,-446947.7555349938920699060 7939592.3324004504829645157,-457152.1906170316506177187 7936279.6902406839653849602,-458636.4133877781569026411 7936046.6402133628726005554,-472334.8333273456664755940 7940116.0101028466597199440,-485198.4684054506942629814 7945349.3450317038223147392,-486064.2000853514764457941 7946397.7474213298410177231,-487362.9645844366750679910 7947096.2047789441421627998,-488939.9164910123799927533 7947444.6196987228468060493,-488816.2405367420287802815 7945816.0284071099013090134,-484703.6532688753213733435 7938487.3286702623590826988,-483745.0811336541082710028 7937615.7876238562166690826,-478395.6230035843909718096 7935116.1855694446712732315,-476818.5597775146015919745 7934651.8379749609157443047,-461945.0512931350385770202 7930701.8574735634028911591,-450089.5255236518569290638 7928426.0700377598404884338,-437083.6241363113513216376 7926986.5002805441617965698,-431332.0800054954597726464 7928612.6971967080608010292,-428641.9331909842439927161 7929599.8290184363722801208,-427652.4142373219365254045 7930585.4177934592589735985,-426879.4116932532051578164 7931805.5106822047382593155,-425982.6218754229485057294 7934884.8448570137843489647,-425271.4016487451735883951 7936163.1643258137628436089,-424281.8826950828079134226 7937208.6147723263129591942,-423261.4169229822000488639 7937963.7641881788149476051,-421715.3005153535632416606 7938196.8735358677804470062,-420231.0777446070569567382 7937673.1199985975399613380,-420014.6726545047131367028 7936104.1698461314663290977,-420540.3232900327420793474 7934535.3372363187372684479,-424776.6978316610329784453 7926986.5002805441617965698,-425580.6471941680065356195 7925768.6496634082868695259,-430311.7255528832902200520 7918865.5747225023806095123,-439062.6620436331722885370 7908146.5803291602060198784,-439928.5050430223345756531 7907162.4612843301147222519,-441783.7556765854824334383 7905079.8117816951125860214,-442804.2214486860902979970 7904210.4574564984068274498,-444041.0923108891001902521 7903574.7026513786986470222,-445401.7504468564875423908 7903112.2934198155999183655,-446613.7970626135356724262 7904153.1706698574125766754,-447615.6724797518108971417 7904430.6510812835767865181,-447819.8324258666834793985 7905264.1952711287885904312,-447560.0127343560452573001 7906271.4045023387297987938,-447077.6653807478141970932 7906966.1530205719172954559,-447102.3783077053376473486 7909014.9291894510388374329,-447442.4593520780326798558 7910404.9956699227914214134,-448741.3351706544635817409 7911100.1249789213761687279,-454771.1780284548876807094 7909999.4984067715704441071,-456162.5603438781108707190 7909536.6952618248760700226,-478302.7825482639018446207 7899235.8357792533934116364,-491630.1746250148280523717 7884100.5738423708826303482,-493794.7821234891889616847 7878562.6237914720550179482,-492836.0986687795375473797 7877639.2308822218328714371,-491444.6050338622881099582 7878101.7436213130131363869,-487393.9114028777694329619 7881620.2998192440718412399,-483528.6760435517062433064 7885370.3156496034935116768,-482044.3419533139676786959 7887680.5423115734010934830,-480065.3040459921467117965 7889585.4105068510398268700,-474159.2484619561000727117 7895247.1371755367144942284,-473046.0535540234413929284 7896114.0029696635901927948,-470634.0941470055840909481 7897269.8401443120092153549,-466583.2891965269809588790 7898600.7080383989959955215,-454270.1290003930917009711 7897339.7704329593107104301,-450472.9098499444080516696 7892647.3448372408747673035,-457677.7299330656533129513 7883407.9262647209689021111,-465253.6891984947724267840 7875966.9064698787406086922,-468747.8966950041940435767 7870493.6025127945467829704,-470757.7701012758770957589 7866230.6901292670518159866,-472056.5346003610757179558 7863008.5110506629571318626,-471438.1548290037899278104 7861627.1391602978110313416,-470355.7954200209933333099 7860764.2205183506011962891,-468995.2486035448382608593 7860130.6614692062139511108,-467170.8334689344046637416 7860073.7079596631228923798,-465222.7423800536780618131 7860821.1792277200147509575,-462965.5170652384986169636 7862317.7935662930831313133,-449885.3655775369843468070 7874238.0992266517132520676,-449545.2845331642893142998 7874872.8415861083194613457,-417908.6192081868648529053 7888958.7861862424761056900,-405141.1641701250919140875 7892877.6015834482386708260,-405419.4628971096826717257 7892242.8005322674289345741,-405481.2452144978451542556 7891317.7338765235617756844,-405048.3237148046609945595 7890856.0748362485319375992,-403471.2604887348716147244 7890510.2649546498432755470,-400533.7617656830116175115 7891260.5448511671274900436,-399946.2174932755879126489 7891490.7593107977882027626,-392092.0708203559624962509 7896923.7363782888278365135,-390329.4380031364853493869 7899062.8409217316657304764,-389989.3569587609381414950 7899583.7110878406092524529,-389278.1367320832214318216 7900972.0410501854494214058,-388906.9975497781415469944 7902475.1732425810769200325,-389030.6735040513449348509 7903226.8516251333057880402,-389308.9722310330835171044 7903921.3198626032099127769,-389463.5950037444708868861 7904731.8913677185773849487,-389339.9190494741778820753 7905542.5499553047120571136,-388288.6177784237079322338 7906526.4577142242342233658,-371776.2637505833990871906 7909420.5819912413135170937,-366179.2310729877208359540 7909942.1676476104184985161,-365468.1221658011781983078 7909593.8144744606688618660,-365127.9298019372508861125 7908957.6059122346341609955,-364695.0083022412727586925 7907683.8908374262973666191,-363025.2159403422265313566 7905542.5499553047120571136,-361355.4235784432385116816 7904210.4574564984068274498,-358819.7882171550882048905 7902823.1978990947827696800,-347687.8391378283849917352 7898657.7445303853601217270,-342956.7607791131595149636 7897385.7666010018438100815,-338627.7684211440500803292 7896691.8994352342560887337,-337854.6545575840864330530 7896807.8170167505741119385,-336463.1609226696891710162 7897328.7397193964570760727,-332783.4951544989598914981 7899178.7949187774211168289,-330340.5889290400082245469 7900451.0750676412135362625,-327557.6016592083615250885 7902359.1685902904719114304,-325454.8877976133371703327 7903921.3198626032099127769,-322610.1182103907340206206 7906005.1081433007493615150,-317569.7943062526173889637 7905772.9920665649697184563,-310643.2729501131107099354 7905079.8117816951125860214,-309715.5363138446118682623 7904963.7670821649953722954,-294532.7822840393055230379 7902069.8936612438410520554,-286091.0913387122564017773 7900335.1015232149511575699,-279411.9218911162461154163 7898657.7445303853601217270,-272114.3726721629500389099 7897963.7604385204613208771,-267197.7247222980367951095 7897558.9311591358855366707,-266300.9349044677801430225 7897558.9311591358855366707,-243356.7633375749283004552 7898022.6654283953830599785,-241872.5405668284511193633 7898311.5771233001723885536,-241254.0494759799330495298 7898716.6549362447112798691,-238254.6571160458843223751 7902591.1796781029552221298,-237265.1381623835477512330 7903574.7026513786986470222,-235997.3204817394434940070 7904037.3485315758734941483,-231049.8370329246390610933 7904442.5254707075655460358,-225112.8346304446749854833 7903285.5894157476723194122,-214815.6704125776886940002 7899930.1450427724048495293,-213547.9640514248167164624 7898427.5195798724889755249,-207054.2528754901140928268 7888199.1726722167804837227,-202817.8783338590001221746 7879311.9254743885248899460,-198303.3163847402029205114 7864159.2944969339296221733,-197499.3670222304062917829 7861108.6719145486131310463,-197406.6378863983263727278 7854322.9903283994644880295,-206590.4958768442447762936 7842088.7466019783169031143,-218062.5260005436139181256 7828384.0264736581593751907,-219515.9132723430811893195 7825863.6018082415685057640,-220165.2398621386382728815 7824547.3261055797338485718,-228668.8244443450530525297 7804931.5168131841346621513,-229503.7206252959731500596 7801962.8116733245551586151,-229967.5889434302516747266 7799165.9901413545012474060,-230987.9433960424503311515 7791122.9890504982322454453,-230833.3206233310338575393 7789584.1477357083931565285,-230122.2117161444912198931 7787646.4188266834244132042,-233028.8749402464891318232 7774495.6529850922524929047,-241223.2139770300709642470 7760230.8845638977363705635,-243820.6316557092068251222 7754952.5976637555286288261,-244500.9050639487104490399 7752400.4817456733435392380,-244624.5810182190325576812 7748997.4949481822550296783,-244531.8518823869817424566 7748317.0801725182682275772,-243758.8493383182503748685 7746333.7345695439726114273,-243264.0342017428774852306 7744237.2962538497522473335,-243387.7101560160517692566 7743387.8288928037509322166,-244253.5531554052140563726 7740215.7584408791735768318,-247098.3227426278463099152 7733140.9131102850660681725,-257766.5148233086511027068 7719122.5119865983724594116,-258725.0869585298933088779 7718106.5176235372200608253,-261384.3982740912178996950 7716186.9059361387044191360,-266053.5829959242837503552 7713590.4577351678162813187,-268867.5170841970248147845 7710939.0473806681111454964,-275855.9320772158680483699 7702143.8372883778065443039,-274155.1928968757856637239 7698314.5398941654711961746,-272519.4642991611035540700 7698975.0949414372444152832,-272516.3473534178920090199 7697357.2623224668204784393,-272609.0764892499428242445 7696455.6417993549257516861,-277030.9093025395413860679 7681663.0372278438881039619,-281329.1774810499628074467 7674472.6987986862659454346,-282225.8559793890453875065 7673462.6303668050095438957,-284019.3242955583846196532 7671498.8558065248653292656,-293976.1848300729761831462 7663141.4353222837671637535,-302881.8554130255361087620 7656360.6491416683420538902,-322300.8726649678428657353 7651321.1642825948074460030,-329351.0699753776425495744 7649754.8876640545204281807,-338009.2773302983259782195 7649418.2626522267237305641,-338937.0139665668830275536 7649530.4693442350253462791,-339740.9633290767087601125 7649530.4693442350253462791,-341781.7835537895443849266 7648915.6693822499364614487,-342956.7607791131595149636 7648244.1193120013922452927,-364911.5247118348488584161 7629915.0750255472958087921,-363427.1906215971685014665 7627626.5736464858055114746,-362066.6438051210134290159 7628017.4850903516635298729,-355449.3679944071918725967 7630194.1779232611879706383,-354861.8237219997681677341 7630473.0900402618572115898,-333463.7685627356404438615 7641309.2897129449993371964,-326413.4599328346084803343 7647068.7463837368413805962,-324960.1839805291965603828 7648411.6991747170686721802,-324372.6397081217728555202 7648691.4770984323695302010,-322919.3637558164191432297 7648971.0638155350461602211,-321094.9486212031333707273 7648803.4717040760442614555,-319486.9385766951018013060 7648411.6991747170686721802,-313395.3134015065152198076 7646566.3073770552873611450,-312684.0931748287985101342 7646286.8084915485233068466,-312158.4425393007113598287 7645056.1682350542396306992,-311601.8450853343820199370 7642204.4286880698055028915,-311478.1691310640308074653 7641421.3775979960337281227,-311663.7387222165707498789 7640638.4065871639177203178,-312034.7665850303601473570 7639912.2529215859249234200,-312745.9868117081350646913 7637788.1106125405058264732,-313395.3134015065152198076 7633152.0074393870308995247,-313178.9083114041713997722 7630976.0818669088184833527,-312158.4425393007113598287 7627793.7029229244217276573,-311756.4678580457693897188 7627292.5268949223682284355,-308509.6122700798441655934 7624893.1614138623699545860,-307891.1211792313260957599 7624614.2515343185514211655,-303995.0503209582529962063 7623330.6740666655823588371,-303160.0428205188945867121 7623163.6420952957123517990,-301026.4934599740663543344 7623330.6740666655823588371,-297810.5846904463833197951 7623051.8210429241880774498,-297099.3644637686084024608 7622829.7898213807493448257,-294285.4303754958673380315 7621714.9129134071990847588,-293759.7797399706323631108 7621379.5180489635095000267,-293172.2354675631504505873 7620933.9533155616372823715,-287853.7241559345857240260 7615137.3490979662165045738,-287451.6381551884114742279 7614636.9998754765838384628,-286956.9343381042708642781 7613355.0917105199769139290,-287513.5317920706002041698 7611963.1416269848123192787,-287946.4532917637843638659 7611461.5962992021813988686,-297594.1796003439812920988 7601724.4468441186472773552,-298119.8302358720684424043 7601334.8734614141285419464,-298676.4276898383977822959 7601001.9700494939461350441,-310364.9742231313721276820 7595830.9100026590749621391,-313920.9640370317501947284 7595165.5948593234643340111,-315528.9740815397817641497 7595054.1812129532918334007,-316920.4677164570312015712 7595609.6653147395700216293,-318126.3917602189467288554 7596388.0507764695212244987,-320043.5360306614311411977 7598054.9133349964395165443,-320631.0803030688548460603 7598444.3202559091150760651,-321465.9764840169227682054 7598722.0799155151471495628,-324681.8852535446058027446 7599332.8663364639505743980,-326351.6776154435938224196 7599389.3044943222776055336,-329165.5003842251608148217 7598777.1127414396032691002,-331299.1610642612795345485 7597276.3588367188349366188,-349048.4972737929783761501 7581065.3174151051789522171,-349821.4998178616515360773 7579956.4910924797877669334,-350780.0719530829228460789 7577017.4980224715545773506,-350965.6415442354627884924 7576297.5412886310368776321,-351120.2643169468501582742 7575464.7264766618609428406,-371961.7220222446485422552 7563227.1312596565112471581,-372549.2662946521304547787 7562895.8749925643205642700,-377651.3725161811453290284 7560627.2625664165243506432,-378393.5395613000146113336 7560516.3485384276136755943,-379537.6812876737676560879 7560903.8576568057760596275,-383062.8356026242836378515 7562231.8118909848853945732,-389339.9190494741778820753 7565274.5438470402732491493,-397565.2049046988249756396 7569094.1052962355315685272,-398802.0757669018348678946 7569371.0050793765112757683,-414077.5589325363398529589 7569704.1353127425536513329,-416025.7613409111509099603 7569482.0468867029994726181,-414881.5082950461073778570 7564555.4867092901840806007,-414696.0500233848579227924 7563891.2803057739511132240,-414263.1285236888215877116 7563503.8198585631325840950,-408140.5565300591406412423 7559852.4908225871622562408,-407645.8527129750582389534 7559576.1326336478814482689,-405883.2198957527289167047 7559410.4817635063081979752,-384577.8938723206520080566 7556811.9024301227182149887,-362314.1070331557421013713 7552776.8266814928501844406,-343760.8214611142175272107 7548081.0225019408389925957,-341472.5380083666532300413 7547363.7690549744293093681,-339864.6392833498539403081 7547196.9893733970820903778,-335968.4571055855485610664 7548357.1669921502470970154,-326722.7054782574414275587 7552335.2238851115107536316,-325547.7282529338262975216 7552942.3347980938851833344,-323970.6650268668890930712 7554047.2722464054822921753,-323537.7435271708527579904 7554434.4556715320795774460,-322331.8194834089372307062 7556148.3642748873680830002,-319023.1815780492033809423 7562673.9831030741333961487,-319301.3689855425618588924 7564057.0277022607624530792,-318961.2879411698668263853 7564777.4328891141340136528,-315467.0804446604452095926 7568596.7473595784977078438,-313951.9108554728445596993 7569758.7612766847014427185,-311045.2476313680526800454 7570423.4747220156714320183,-309715.5363138446118682623 7570645.5903452932834625244,-295800.5999646834097802639 7570312.4193167928606271744,-292893.9367405814118683338 7569316.1825163029134273529,-289492.5696993924793787301 7567045.6756457220762968063,-278546.1902112183161079884 7558968.4980991361662745476,-264414.7374524663900956511 7549848.3991315234452486038,-251612.9960112378757912666 7542230.2410548646003007889,-237141.4622081132256425917 7535720.3350925231352448463,-231513.5940315676853060722 7532027.7978241583332419395,-230740.5914874989539384842 7529327.9761461354792118073,-230245.8876704148133285344 7528116.1310436213389039040,-229441.8269884137844201177 7526629.0999384159222245216,-225050.9409935653093270957 7519913.4744136407971382141,-224154.1511757350235711783 7517713.3279058709740638733,-222731.8220418679120484740 7514797.9966831970959901810,-220412.5917706821055617183 7510125.2724285703152418137,-219701.3715440043597482145 7509136.7898908061906695366,-206095.6807402688718866557 7490478.0977485328912734985,-202570.5264253183559048921 7485929.9104192750528454781,-202106.7694266724865883589 7485492.1190162533894181252,-201302.7087446714285761118 7485272.4438472837209701538,-200560.6530190438206773251 7485382.2806511241942644119,-199787.5391554866801016033 7486477.5790026420727372169,-194469.0278438552340958267 7484121.5190278962254524231,-188129.9394406319479458034 7480289.8218887671828269958,-182347.5598108662525191903 7475857.3263881225138902664,-181883.6914927319739945233 7475418.5295847235247492790,-176039.4182260840607341379 7456677.5100805619731545448,-175946.6890902519808150828 7455914.6764129670336842537,-175142.7397277450072579086 7443754.2639988819137215614,-175390.0916362856514751911 7442011.1962630506604909897,-175853.8486349315207917243 7440540.6407940434291958809,-176472.3397257800679653883 7439233.4532788544893264771,-177338.1827251692302525043 7438253.9913258319720625877,-178049.4029518470051698387 7437110.0377145763486623764,-177956.5624965265451464802 7436349.1286853188648819923,-174307.8435467940871603787 7417809.6096018599346280098,-169267.5196426559996325523 7393792.0168038709089159966,-168649.0285518103046342731 7390977.9059561742469668388,-165309.4438280122994910926 7379298.8314456325024366379,-164845.6868293664301745594 7377946.8761517070233821869,-157269.7275639401341322809 7361758.0114582711830735207,-153775.5200674307125154883 7357284.6592334136366844177,-151951.1049328174558468163 7347860.9245307389646768570,-149786.6087538343272171915 7336944.2758245058357715607,-145210.1531678305182140321 7317622.4263066947460174561,-144437.0393042733776383102 7316122.3325622798874974251,-129718.1535926052310969681 7294178.3575482182204723358,-118153.2830135825643083081 7287927.5555360559374094009,-116205.1919246989855309948 7287126.6845920952036976814,-97497.2835799460444832221 7278907.7899014819413423538,-62864.5654797602182952687 7261591.0334556419402360916,-58040.7579852129128994420 7255897.8630520133301615715,-57391.4313954173485399224 7254834.3475495018064975739,-50928.6670379237402812578 7241869.5298562953248620033,-48547.6544493441469967365 7236774.8955561565235257149,-46939.7557243273477070034 7233327.9344458756968379021,-43785.6292721906356746331 7221986.6204860499128699303,-43878.4697275110956979915 7221298.6330675203353166580,-43723.8469547996719484217 7220663.1227116677910089493,-42517.8115915465459693223 7218704.4958674106746912003,-29221.3663332339092448819 7203262.2267202856019139290,-22322.7861692835504072718 7198941.0594396125525236130,-16883.4932101425183645915 7196979.0933433379977941513,-13018.1465313280514237704 7195236.4839738700538873672,-9183.8579904459147655871 7192704.6780755193904042244,-8719.9896723088095313869 7192281.5234471354633569717,-8348.9618094949983060360 7191384.5792159922420978546,-9060.1820361727550334763 7190277.9254292584955692291,-12523.4427142439144517994 7188800.4774872828274965286,-15151.8072113641555915819 7188273.3185697188600897789,-15739.3514837687544059008 7188009.6574359694495797157,-17934.7944811958222999237 7186849.7671871241182088852,-19357.2349345513357548043 7185637.8634417597204446793,-22573.1437040790115133859 7180526.0465760026127099991,-23222.4702938745722349267 7179315.1152911577373743057,-23748.1209294026375573594 7176944.6104523604735732079,-24026.4196563843906915281 7175207.9441003389656543732,-24057.3664748254886944778 7174418.4873064290732145309,-23933.6905205551593098789 7172997.4369678078219294548,-23624.4449751294814632274 7171734.2435828270390629768,-19047.9893891284882556647 7155333.7633468983694911003,-18615.0678894324810244143 7154126.8833253327757120132,-14193.2350761429133854108 7144419.5832378705963492393,-12554.3895326850124547491 7141903.8511276524513959885,-11070.0554424473011749797 7139859.8072505984455347061,-5998.8960393593388289446 7132945.1507118921726942062,-1453.3872717966128220723 7126454.6479702983051538467,6802.8454018663251190446 7114535.5168764470145106316,7143.0377657331036971300 7114013.2940712310373783112,14069.5591218697554722894 7103259.0684288404881954193,15708.4046653304885694524 7099869.9713335065171122551,16636.0299821106345916633 7097523.1800195593386888504,17099.8983002449058403727 7095647.7857606485486030579,17161.7919371271018462721 7094917.8248328994959592819,16914.3287090923840878531 7093250.6669076913967728615,16357.7312551260511099827 7091948.5844065481796860695,15986.7033923122398846317 7091374.9708274621516466141,15646.6223479395266622305 7090854.2729036491364240646,13141.9338050896112690680 7088093.8515161555260419846,12554.3895326850124547491 7087833.5123058557510375977,12678.0654869553382013692 7088511.4617120213806629181,13296.4452583126330864616 7089708.3704364756122231483,13729.3667580058081512107 7090177.6229659169912338257,15120.8603929230575886322 7092416.4693086668848991394,15337.3768025166773441015 7093146.1998539250344038010,15461.0527567870049097110 7093980.4742726171389222145,15399.1591199048089038115 7094708.8481437452137470245,14533.4274400068607064895 7096324.9029302448034286499,12152.4148514301032264484 7098357.9154378520324826241,9926.0250355647694959771 7099765.4169691428542137146,7514.0656285469149224809 7101017.1566209914162755013,4916.6479498677708761534 7102111.5584062384441494942,3308.6379053569012285152 7102269.0941189657896757126,1886.1974520013864093926 7102164.5080816512927412987,1236.8708622029926118557 7102008.4765428882092237473,0.0000000000000000000 7101431.1302505927160382271,-3865.2353593232346611330 7099556.3123803008347749710,-5751.5441308158542597084 7099088.1933918865397572517,-7946.9871282400890777353 7098983.6493452610448002815,-8781.8833091910055372864 7099139.6213106811046600342,-9524.0503543098602676764 7099348.7149004405364394188,-11781.2756691250579024199 7100703.4522398170083761215,-14440.5869846863988641417 7102529.9080452322959899902,-15801.1338011597163131228 7103780.5822782544419169426,-19109.8830260078502760734 7108896.1305201593786478043,-25758.1056551655856310390 7117618.0292343124747276306,-27767.9790614372977870516 7119133.9567664228379726410,-30334.5612411665752006229 7120230.8664008406922221184,-32715.5738297433344996534 7120440.5178420627489686012,-39772.0050316395572735928 7119290.5144223310053348541,-42406.4921007521479623392 7118537.4854326220229268074,-47044.8413236353298998438 7116844.4668464977294206619,-61132.9908004702592734247 7114691.9835692644119262695,-62431.6439800642110640183 7115265.9720149785280227661,-63359.3806163355911849067 7116102.5801986549049615860,-64317.9527515568333910778 7117618.0292343124747276306,-65245.5780683369812322780 7118558.3705776063725352287,-67441.0210657612187787890 7119133.9567664228379726410,-68987.1374733898846898228 7119185.7032449971884489059,-70378.6311083042965037748 7119029.1476632580161094666,-72852.3728327131248079240 7118297.0280052861198782921,-75047.9271496285946341231 7117147.3361070714890956879,-76068.2816022407932905480 7116520.1650381507351994514,-80026.3574168844934320077 7113489.7894850960001349449,-79500.7067813592584570870 7112237.3926360346376895905,-77150.6410112235898850486 7110930.8694202639162540436,-57855.1883940603875089437 7110774.4770652083680033684,-56203.9863871237321291119 7110994.0287472624331712723,-50971.9703198407660238445 7112811.2030738070607185364,-43439.3143363315102760680 7115037.0863102758303284645,-34014.3383288285258458927 7116102.5801986549049615860,-33148.4953294393417309038 7115946.0856116190552711487,-32035.3004215066757751629 7115162.7187287099659442902,-29437.7714233362967206631 7111818.5297639798372983932,-26891.2267519497254397720 7107538.4878966500982642174,-25665.3765193335239018779 7104772.2481803689152002335,-24799.5335199443434248678 7103103.0153247127309441566,-23067.8475211631484853569 7100547.4494800595566630363,-13049.0933497691494267201 7091374.9708274621516466141,-11224.6782151587249245495 7089708.3704364756122231483,2999.3923599340528198809 7077844.5566999809816479683,9493.1035358687622647267 7073477.1131620751693844795,13667.5844406176784104900 7071294.8535670163109898567,14316.9110304132391320309 7071190.6753119593486189842,15584.7287110573306563310 7071503.2141794422641396523,16357.7312551260511099827 7071294.8535670163109898567,18769.6906621439047739841 7068542.7249169526621699333,24150.0956106547164381482 7060495.8343626335263252258,26221.9739732998568797484 7057227.4555320134386420250,27551.5739713349103112705 7053702.6916324980556964874,28386.4701522858267708216 7051318.2075183065608143806,30056.2625141848257044330 7047328.0035564331337809563,37724.9509154403349384665 7026842.4494658457115292549,37879.5736881545890355483 7026223.6754355067387223244,39394.7432773421678575687 7017703.5450261589139699936,39673.0420043239137157798 7015691.0640749577432870865,39827.6647770353374653496 7012802.3316875854507088661,39703.9888227650153567083 7010533.2681854963302612305,39425.6900957804318750277 7008832.6830489663407206535,38034.1964608660127851181 7000744.1420908663421869278,37477.5990068996834452264 6999455.6372812390327453613,36642.7028259487633476965 6998529.4734485764056444168,36086.1053719824340078048 6998272.2350947232916951180,30798.4295593036804348230 6995389.6200752491131424904,22696.8196583493372600060 6988188.6528307721018791199,19295.3412976691397489049 6984693.0072716111317276955,3586.9366323414842554484 6967700.7002976760268211365,204.8278630594293474587 6961354.9044529246166348457,4.1188211586421843435 6960748.3811523709446191788,730.4784985846645213314 6960382.4234024332836270332,3123.1796336956126651785 6960984.8607404958456754684,6431.8175390525138936937 6962521.7795065864920616150,7668.6884012583386720507 6962830.0843350710347294807,8379.9086279360963089857 6962880.8238509008660912514,9616.7794901390898303362 6962624.5464622424915432930,13605.6908037354824045906 6960369.8807924287393689156,16419.6248920082471158821 6958730.6571325277909636497,17501.8729814998150686733 6958013.1413981756195425987,18398.6627993300935486332 6957194.2821242287755012512,19109.8830260078502760734 6956221.0161899179220199585,19604.5868430948212335352 6955145.0120533993467688560,20810.5108868567149329465 6953047.0633922005072236061,24242.9360659780104469974 6947366.2139230202883481979,26036.4043821473351272289 6945729.6511862622573971748,26562.0550176725701021496 6945371.3754870649427175522,27149.5992900800010829698 6945218.3859691601246595383,40446.0445483926378074102 6942457.3447160013020038605,42177.7305471738291089423 6942662.3653290001675486565,47589.0823141257424140349 6950231.5565550718456506729,47929.2746779896842781454 6950795.0945429606363177299,48207.5734049714374123141 6951409.3390278974547982216,48640.3835851762123638764 6952892.4528996841982007027,48795.0063578876361134462 6954479.9785400191321969032,49351.6038118539654533379 6957809.0179931297898292542,49568.1202214475852088071 6958577.2276446707546710968,54391.9277159920675330795 6970264.7331179445609450340,55010.4188068405928788707 6971443.9915941450744867325,60205.2541641988791525364 6976475.1092778826132416725,60761.8516181652157683857 6976731.6462687784805893898,61473.0718448429688578472 6976885.4245519321411848068,74274.8132860686309868470 6978579.2146231736987829208,97713.7999895368266152218 6977039.0209598978981375694,98270.3974435031559551135 6976936.0685286289080977440,98981.6176701837539440021 6976576.5771750826388597488,100032.9189412342238938436 6975190.5151413669809699059,101022.4378948965604649857 6974317.9738076440989971161,101486.3062130308389896527 6974164.2449267618358135223,104887.6732542198005830869 6973907.7903467118740081787,107948.9592510360380401835 6973754.0693561760708689690,110731.9465208677138434723 6973959.8935649544000625610,111968.8173830707091838121 6974267.1616508271545171738,131789.9206357591319829226 6971702.0295296050608158112,142736.3001239332952536643 6969598.2566089574247598648,145364.6646210535254795104 6968520.4514044271782040596,150559.6112979030585847795 6966316.1764631457626819611,152816.9479322094703093171 6964982.6451792847365140915,155105.1200654629501514137 6963342.4759793952107429504,159434.2237429232918657362 6959857.6795340720564126968,163485.0286933990428224206 6956118.3314620824530720711,167381.2108711662003770471 6952995.0962426476180553436,168896.3804603509488515556 6951869.0399288795888423920,170411.5500495385203976184 6950743.1420124322175979614,183151.5091733760491479188 6940771.3688324391841888428,184450.1623529700154904276 6939492.4941559778526425362,186491.0938971740542910993 6936530.2116574672982096672,187727.9647593770641833544 6934435.6625089757144451141,189521.4330755491973832250 6930605.8057489953935146332,190047.0837110744323581457 6928819.8021876635029911995,191593.2001187031273730099 6923412.4016031753271818161,194128.8354799913067836314 6914339.2039271770045161247,194407.1342069730453658849 6913066.0045715495944023132,194437.9697059229074511677 6912402.9866241440176963806,194695.7856465998629573733 6896365.8756385929882526398,194747.2152513485925737768 6882968.0210112361237406731,192613.6658908037352375686 6874853.6870781192556023598,192922.8001167353359051049 6871508.5254627736285328865,192211.5798900575900916010 6869430.8607410807162523270,187656.9429242505284491926 6859341.4470536075532436371,181450.7699930359667632729 6835510.8029802357777953148,181203.4180844924703706056 6833190.3922358248382806778,180894.1725390696374233812 6831576.9023750508204102516,176843.3675885938573628664 6815608.4350056154653429985,176565.0688616092957090586 6815005.9218393582850694656,175853.8486349315207917243 6813948.6433141799643635750,154765.0390210902260150760 6794701.8020724281668663025,148487.8442547491285949945 6787326.9754436798393726349,147900.2999823417048901320 6787176.7781377770006656647,146694.3759385798184666783 6789934.7506418079137802124,142581.6773512218496762216 6797011.1910111699253320694,141839.6216255942417774349 6798015.1014129379764199257,140788.2090350497164763510 6798768.2063146326690912247,130274.7510465715749887750 6804797.7405615402385592461,129656.2599557230423670262 6804746.7220581844449043274,129161.5561386389017570764 6804345.0982658406719565392,128975.9865474863763665780 6803692.2307787518948316574,134511.0142687086190562695 6791239.5852784086018800735,142272.4318057961936574429 6778656.0165675682947039604,142921.8697150858060922474 6778606.4248035736382007599,143478.4671690521354321390 6778355.9459475129842758179,144004.1178045773704070598 6778005.2164214923977851868,144189.6873957299103494734 6776603.3514611618593335152,143200.1684420675737783313 6773949.2166879503056406975,142519.8950338308932259679 6772997.6602753000333905220,140942.8318077611329499632 6771096.5037884619086980820,136211.7534490487014409155 6765642.7242298284545540810,134511.0142687086190562695 6764042.4606979787349700928,133552.4421334873768500984 6763342.0733725214377045631,131511.6219087745703291148 6762192.8922931132838129997,126038.3765049404610181227 6759893.2189029902219772339,123750.0930521957634482533 6759244.1019327081739902496,121152.6753735166130354628 6758744.5259030275046825409,116885.4653329384746029973 6758744.5259030275046825409,115463.0248795801308006048 6759143.8964259419590234756,115122.9438352074066642672 6759643.4970304546877741814,111041.1920662905613426119 6765992.9195979200303554535,109835.2680225258372956887 6768744.9549126578494906425,108969.4250231366459047422 6768443.9900079276412725449,104269.2934828625002410263 6765443.4166989130899310112,103310.7213476412580348551 6764742.9084688900038599968,95969.7575272797112120315 6759293.3955801734700798988,80613.9016892919171368703 6750852.5330523196607828140,78387.5118734265852253884 6750054.4468685286119580269,77892.6967368512123357505 6749555.4357385458424687386,77954.5903737334156176075 6748956.3753528706729412079,84448.1902301768859615549 6744514.7519229725003242493,85068.5737523678835714236 6744607.9531195899471640587,96074.8431265876861289144 6748157.2177806487306952477,97157.2025355704972753301 6748706.9895085664466023445,97961.1518980803084559739 6749504.7636540355160832405,100125.6480770662747090682 6752040.7322811791673302650,100366.8774136145802913234 6752250.4959800848737359047,100700.8358859949512407184 6752430.7856468530371785164,102290.2555755406501702964 6753349.7319445507600903511,103094.2049380504759028554 6753299.0361882746219635010,103712.6960288961709011346 6752999.9017133433371782303,104609.3745272380474489182 6752251.2149737980216741562,105413.3238897450355580077 6750653.5882240496575832367,106310.1137075753067620099 6730660.1351233590394258499,106186.4377533049846533686 6729912.0686262054368853569,105815.2985699999990174547 6729265.1621820526197552681,101300.7366218783135991544 6724785.1914182389155030251,97435.5012625550734810531 6721153.1737744640558958054,91374.7115863163780886680 6716579.2334731891751289368,89952.3824524492665659636 6715585.3395559396594762802,87880.5040898041188484058 6714540.9331694291904568672,86736.4736829215835314244 6714143.7467630458995699883,85375.8155469570338027552 6714044.0953743606805801392,75759.0360568151081679389 6715982.5962533196434378624,72914.2664695953135378659 6716827.8153850547969341278,72233.9930613558244658634 6716777.1677711084485054016,51763.5632188718227553181 6710764.8671130761504173279,50279.3404481253455742262 6709871.2812930857762694359,42610.6520468670059926808 6701381.8283900208771228790,43064.2789718502899631858 6701007.0742961307987570763,43754.6824537495340337045 6700735.9331194628030061722,45888.3431337856382015161 6700735.9331194628030061722,49413.4974487361614592373 6701579.3939626310020685196,50372.0695839574036654085 6702225.3560402169823646545,50774.0442652094789082184 6702671.0925624407827854156,51639.8872646014933707193 6704409.2511069858446717262,52103.7555827357718953863 6705749.3941652383655309677,52443.8366271113118273206 6706294.8788479296490550041,53309.6796265004959423095 6707039.5279338592663407326,54453.8213528742635389790 6707437.7983302054926753044,59679.6035286736441776156 6708082.7945359824225306511,61102.0439820291576324962 6708082.7945359824225306511,65492.9299768776327255182 6708082.7945359824225306511,66296.9906588786834618077 6707983.2172337938100099564,77645.3448283077304949984 6705005.0416132481768727303,78789.4865546814980916679 6704409.2511069858446717262,79717.1118714616459328681 6703664.8422080753371119499,80150.0333711576531641185 6703266.7560733892023563385,80737.5776435650768689811 6702225.3560402169823646545,80892.2004162765078945085 6701629.7685904707759618759,80923.0359152263699797913 6700983.8536843284964561462,80737.5776435650768689811 6700239.7573901163414120674,80459.2789165805006632581 6699644.3149194410070776939,80057.3042353255877969787 6699198.9223657045513391495,78913.1625089518202003092 6698801.0541552761569619179,76903.1777831888757646084 6698652.4835701789706945419,76161.1220575612533139065 6698703.0186938904225826263,74120.1905133572145132348 6699347.3243200285360217094,71770.1247432215459411964 6699694.8561883503571152687,71244.4741076963109662756 6699446.7962147807702422142,62122.5097541297654970549 6694339.0450256969779729843,61936.9401629800777300261 6693793.1100867912173271179,62184.2920715207292232662 6693248.4608231084421277046,62864.5654797602182952687 6692306.2962902672588825226,64379.7350689449667697772 6690572.1413981299847364426,64812.6565686409740010276 6690174.8719813767820596695,65431.1476594894993468188 6689927.1012506941333413124,68987.1374733898846898228 6689332.4108338663354516029,78387.5118734265852253884 6688341.8319851234555244446,100744.1391679119842592627 6681805.1349956169724464417,101424.4125761514733312652 6681756.1299759773537516594,109247.7237501212366623804 6682646.8198239812627434731,111226.7616574430867331102 6683042.4755222806707024574,112680.0376097484549973160 6683786.2531834430992603302,115030.1033798869466409087 6685567.8164087301120162010,117194.7108783613221021369 6686509.2034763842821121216,118307.9057862939953338355 6686807.1536996848881244659,119080.9083303627121495083 6686905.0463614761829376221,132686.5991340981854591519 6688540.5088516287505626678,154239.3883855649910401553 6690174.8719813767820596695,158908.5731073980568908155 6680073.2087286934256553650,157084.1579727876232936978 6656945.3261903403326869011,156836.8060642441269010305 6654675.3291309745982289314,156558.5073372623883187771 6653096.7032059691846370697,155661.7175194292794913054 6650334.7957205474376678467,154765.0390210902260150760 6648608.2778596915304660797,152384.0264325134630780667 6645355.0285463603213429451,151177.9910692603443749249 6644074.7730702105909585953,150683.2872521733806934208 6643729.5977886170148849487,140540.8571265090431552380 6639296.2930648280307650566,137139.3787658288492821157 6639148.7998142978176474571,135871.6724046760064084083 6638951.3190174130722880363,122482.3866910429060226306 6634324.4684879593551158905,120998.0526008051965618506 6633437.9595231357961893082,116390.6501963602640898898 6630043.3960066903382539749,114659.0755170731426915154 6628519.7884833589196205139,111443.1667475454742088914 6625126.0206480631604790688,110700.9997024266194785014 6624193.0823077233508229256,109773.3743856464716373011 6622570.4257791340351104736,108969.4250231366459047422 6620703.9979277048259973526,108289.1516148999944562092 6618689.4539327388629317284,107948.9592510360380401835 6617216.3961567794904112816,107763.3896598835272016004 6615693.5772022884339094162,107979.9060694743093336001 6611619.0300757139921188354,108536.5035234406386734918 6609263.1620539985597133636,95363.7342194011725950986 6608135.2838680362328886986,90632.6558606859180144966 6609705.2774426080286502838,88313.4255895001260796562 6609999.0332648092880845070,86983.8255914650653721765 6609902.0555871129035949707,79748.0586899027402978390 6604506.3188261361792683601,78572.9701450878783361986 6603231.0086531080305576324,75975.5524664087279234082 6599947.3558878600597381592,75171.6031038989167427644 6599114.1223738146945834160,73718.2158321023016469553 6598231.2304569184780120850,63142.8642067419714294374 6594410.2218072060495615005,55505.1226239247334888205 6592696.6615649545565247536,50403.0164023956676828675 6591962.3375045266002416611,47960.1101769395536393858 6591472.8832935625687241554,44589.6899541916864109226 6590297.7480696095153689384,41404.6166836138800135814 6588878.0462712580338120461,38899.9281407639646204188 6587508.4432779615744948387,34199.7966004898116807453 6582566.6249391864985227585,32993.8725567279179813340 6581296.1196682117879390717,31942.4599661862121138256 6579829.5697773406282067299,30829.2650582535461580846 6577482.7885235277935862541,30179.9384684551514510531 6576456.0180761599913239479,29345.0422875042349915020 6575674.7659990629181265831,28262.7941980126670387108 6575186.2792139910161495209,26964.0296989274793304503 6575088.3038974460214376450,25356.1309739106764027383 6575332.6287153083831071854,23964.6373389934269653168 6575723.1413118075579404831,13729.3667580058081512107 6578850.9004119643941521645,10946.3794881741414428689 6580318.3320494303479790688,6338.9770837320520513458 6582616.2731604762375354767,-10513.4579884809681971092 6587801.4071046337485313416,-15368.3236209549431805499 6589220.9231394883245229721,-17470.9261630587170657236 6589759.9347267141565680504,-21243.4323865527221641969 6590542.5246952790766954422,-22944.1715668928191007581 6590688.9712123088538646698,-28386.4701522858267708216 6590640.5078765731304883957,-64256.0591146746373851784 6585159.4724248666316270828,-70966.1753807117347605526 6583740.6566685084253549576,-75171.6031038989167427644 6582664.6892674667760729790,-79191.4612359364109579474 6581394.1688646301627159119,-83056.6965952596365241334 6579976.0018110657110810280,-85499.6028207185881910846 6577678.7972948485985398293,-85344.9800480071717174724 6576603.6216455642133951187,-85344.9800480071717174724 6575967.4849601816385984421,-85623.2787749889248516411 6574550.2519700163975358009,-86427.2281374987360322848 6573525.0780129041522741318,-86891.0964556358376285061 6573231.2148195700719952583,-87416.7470911610726034269 6572987.1281669437885284424,-88653.6179533640679437667 6572840.8203256204724311829,-89333.8913616007339442149 6573183.0299939680844545364,-100558.5695767594588687643 6580366.7346454048529267311,-101331.6834403194079641253 6581100.0247781453654170036,-101393.4657577103789662942 6581784.9690047213807702065,-101146.1138491668825736269 6582371.9082093313336372375,-99569.0506230999453691766 6584132.9769384199753403664,-99136.2404428951704176143 6584474.2473884969949722290,-96755.2278543184220325202 6585159.4724248666316270828,-96384.0886720133712515235 6585600.1175798885524272919,-96136.7367634698894107714 6586137.6591189010068774223,-96260.4127177402115194127 6586823.0226584235206246376,-97064.4733997412695316598 6587556.8881699843332171440,-103186.9340738797036465257 6592941.3333000577986240387,-103515.6605301919189514592 6593018.0150606799870729446,-104547.5922098470764467493 6593186.1885973056778311729,-121833.0601012445113155991 6594068.5390184000134468079,-128543.0650477903691353276 6593773.9415692239999771118,-129068.8270028068363899365 6593529.0687755867838859558,-129377.9612287412892328575 6592991.2204331913962960243,-129037.8801843657420249656 6592451.8208384830504655838,-127646.3865494513302110136 6591472.8832935625687241554,-124461.4245983647560933605 6588095.7907635690644383430,-124275.8550072122307028621 6587556.8881699843332171440,-124214.0726898212742526084 6586921.1377092571929097176,-124399.5309614825528115034 6586285.4363561803475022316,-125481.7790509741316782311 6583740.6566685084253549576,-125729.2422790088457986712 6583203.2716363826766610146,-126100.2701418226497480646 6582762.7547638705000281334,-126564.1384599597658962011 6582371.9082093313336372375,-127089.7890954850008711219 6582175.7876954572275280952,-127677.3333678924245759845 6582029.3165351273491978645,-148209.5455277645669411868 6584474.2473884969949722290,-156744.0769284148991573602 6580855.5288042686879634857,-177307.2359067281358875334 6572644.9266350185498595238,-179688.2484953048988245428 6573623.0347399692982435226,-185192.3293980888556689024 6574990.3297461569309234619,-186166.3749425292480736971 6575124.5387979382649064064,-187511.5596692746621556580 6575234.6516581308096647263,-189397.7571212760522030294 6574892.3567600846290588379,-202199.4985625045374035835 6572450.4443286526948213577,-215310.4855491530615836382 6570643.5257586129009723663,-223690.3941770891542546451 6571033.8161583300679922104,-224927.2650392949872184545 6571522.0567965423688292503,-225700.2675833608664106578 6572304.1460169507190585136,-225947.7308113955950830132 6572890.4099519327282905579,-226195.0827199390623718500 6574111.4288358055055141449,-226658.8397185821086168289 6574403.9163672123104333878,-227308.2776278717501554638 6574355.7246823841705918312,-229936.6421249919803813100 6571131.7432721024379134178,-230214.9408519737189635634 6570691.8715392947196960449,-231482.7585326178232207894 6568495.1498176343739032745,-231699.1636227201961446553 6567957.5282158264890313148,-231513.5940315676853060722 6567371.4393543517217040062,-219825.0474982746818568558 6563174.7845095358788967133,-218544.8733541518449783325 6562004.3818928413093090057,-218081.1163555059756617993 6562530.9020095784217119217,-217765.6369185995426960289 6562735.8640607446432113647,-217394.6090557857241947204 6562619.2440684689208865166,-217227.6298195955459959805 6562325.9450527308508753777,-217134.9006837634951807559 6561975.5805613631382584572,-215465.1083218644780572504 6557761.3089584950357675552,-218000.6323636642773635685 6550597.2664701351895928383,-218464.5006817985267844051 6550255.9927347600460052490,-219113.8272715969360433519 6550110.2629772489890456200,-228359.5788989222201053053 6548356.4466876452788710594,-229039.8523071616946253926 6548306.8290311563760042191,-236213.8368913330778013915 6552106.6863311016932129860,-237543.4368893681094050407 6553178.6718548554927110672,-238749.3609331300249323249 6553813.0525497095659375191,-244593.6341997779381927103 6555420.8014626977965235710,-265837.1779058218817226589 6558151.1655460912734270096,-266486.5044956202618777752 6558151.1655460912734270096,-270630.0385819252696819603 6556980.7734949644654989243,-272361.7245807064464315772 6555470.4619060028344392776,-274340.7624880283256061375 6550547.8107175137847661972,-274680.8435324038728140295 6549086.7198353800922632217,-274495.3852607397129759192 6548015.2655762238427996635,-273691.3245787415071390569 6546457.1623187074437737465,-269918.9296747387270443141 6543828.4419597163796424866,-271001.8456811748910695314 6540638.4371511004865169525,-273315.0646998595329932868 6541884.1233676308766007423,-273649.9137281669536605477 6544135.2785705877467989922,-277698.1583303528605028987 6550078.8726683063432574272,-285318.0887946435250341892 6556493.2176425578072667122,-296326.3619196998770348728 6563712.0942877801135182381,-300624.5187787191243842244 6566103.2751583149656653404,-302912.6909119753981940448 6567225.4069060487672686577,-308509.6122700798441655934 6569911.3416102193295955658,-314261.1564008957357145846 6571962.1486793523654341698,-318806.6651684584212489426 6573378.9364791419357061386,-320909.3790300505934283137 6573867.3161043440923094749,-323599.6371640530414879322 6574257.7592431930825114250,-325578.6750713749206624925 6574013.4663015846163034439,-326197.0548427322064526379 6573769.3564727697521448135,-327774.1180687991436570883 6572498.8007120154798030853,-329443.9104306981316767633 6570643.5257586129009723663,-331515.6774738520616665483 6569519.6981390863656997681,-344472.0416877919342368841 6567029.6466476824134588242,-352944.6794515572837553918 6566395.4785974835976958275,-353594.0060413556639105082 6566297.6077411789447069168,-361139.0184883408364839852 6563906.3705149488523602486,-362252.2133962735533714294 6563468.2893692338839173317,-379754.0863777761696837842 6553665.8564450433477759361,-382629.8027834370732307434 6551717.2906227651983499527,-385010.8153720138361677527 6549184.3867120612412691116,-385722.0355986944050528109 6548258.7896054740995168686,-388968.8911866603302769363 6541443.7060142951086163521,-389649.1645948970108292997 6539594.1744699953123927116,-395338.8150888335076160729 6522536.1427857801318168640,-395555.3314984271419234574 6520302.7428543651476502419,-406532.6578050423413515091 6486396.6404237886890769005,-409099.1286652804119512439 6484511.5183053677901625633,-412624.2829802309279330075 6482674.6582738365978002548,-413737.4778881635866127908 6482143.8033350799232721329,-415098.0247046397416852415 6482094.7548869289457798004,-421838.9764696267084218562 6483109.8733296152204275131,-423354.2573783026891760528 6483448.3897058935835957527,-424498.3991046764422208071 6483979.1541199050843715668,-426724.6776010505273006856 6485719.5549450898543000221,-427436.0091472195344977081 6486589.1091404585167765617,-428023.4421001357259228826 6487604.7774988533928990364,-428796.4446442044572904706 6489974.0513618225231766701,-431517.6495966479997150600 6494908.6856395686045289040,-431795.9483236297382973135 6495392.4503537956625223160,-438691.6341808193828910589 6500570.8665431030094623566,-439371.9075890560052357614 6500910.0963339135050773621,-439928.5050430223345756531 6501104.2968212217092514038,-440670.5607686528237536550 6501104.2968212217092514038,-442587.7050390924559906125 6500814.3923815786838531494,-443670.0644480752525851130 6500378.0730656441301107407,-445834.5606270612333901227 6499457.9207693589851260185,-446576.7276721801026724279 6498586.8420996433123946190,-446855.0263991618412546813 6498102.8909656442701816559,-447813.7098538743448443711 6497424.8492318037897348404,-448401.1428067905362695456 6497182.9911710228770971298,-449019.5225781478220596910 6497133.6794784562662243843,-451153.1832581810886040330 6497666.8886055145412683487,-452111.7553934023017063737 6498392.7005846193060278893,-457677.7299330656533129513 6503671.5180009743198752403,-463707.5727908660774119198 6510649.9604480331763625145,-464758.8740619165473617613 6511959.0404964704066514969,-487579.3696745390770956874 6509534.2639653859660029411,-492650.6403971154359169304 6507693.2483937256038188934,-493578.2657138984068296850 6507014.4215451674535870552,-494368.0775010754005052149 6506141.3271888131275773048,-495464.5744853881769813597 6504833.1662141699343919754,-495742.8732123727677389979 6504348.8502035420387983322,-497196.2604841693537309766 6503477.4317235788330435753,-499020.6756187797873280942 6502896.0720003144815564156,-502112.7971145459450781345 6502314.7530601238831877708,-505761.6273837667540647089 6502266.8056578664109110832,-517109.9815531958010978997 6502508.8121071672067046165,-520820.7054592988570220768 6502896.0720003144815564156,-530159.1862224561627954245 6500378.0730656441301107407,-540734.4265283253043889999 6486493.5703135132789611816,-551340.7249721267726272345 6479824.5810984466224908829,-561947.0234159281244501472 6475960.9393837330862879753,-563214.8410965750226750970 6470410.8720269836485385895,-564977.3625943033257499337 6460960.2800550907850265503,-564513.6055956602795049548 6460575.1449917182326316833,-564235.1955491844564676285 6460093.4043032908812165260,-562689.1904610469937324524 6457010.1961755231022834778,-562565.5145067767007276416 6456383.1840558908879756927,-562781.9195968790445476770 6454891.0991322873160243034,-563214.8410965750226750970 6453638.6548310611397027969,-563895.1145048117032274604 6452772.3683655355125665665,-566183.3979575593257322907 6450220.8311815606430172920,-574315.8433574606897309422 6447044.8063102504238486290,-574656.0357213246170431376 6446611.4752407195046544075,-576263.9344463442685082555 6443292.9624209115281701088,-576758.6382634283509105444 6442187.4891776861622929573,-578088.3495809546438977122 6438534.0316680874675512314,-578737.7874902414623647928 6438581.6127114910632371902,-579170.5976704462664201856 6438965.7314497744664549828,-582046.4253955984022468328 6441658.0011166380718350410,-583035.8330297694774344563 6443005.0682396311312913895,-585076.7645739734871312976 6446419.9300056621432304382,-585231.3873466849327087402 6446997.3500893814489245415,-585200.4405282466905191541 6447622.2697355588898062706,-584551.1139384482521563768 6449450.1864814916625618935,-584520.1671200072159990668 6450268.3057826254516839981,-584798.4658469917485490441 6451472.7611525235697627068,-587395.9948451621457934380 6456623.8567919051274657249,-588849.1594779762672260404 6458310.8421387113630771637,-593734.8606094029964879155 6462117.1803809301927685738,-608979.5082760875811800361 6467950.4582427898421883583,-610309.1082741225836798549 6468288.3572993138805031776,-611082.2221376826055347919 6468337.3242838028818368912,-612380.9866367677459493279 6468143.8890122128650546074,-613525.1283631415572017431 6467758.4210167797282338142,-615689.6245421246858313680 6466745.1769030587747693062,-616431.6802677522646263242 6465925.2618505218997597694,-616864.6017674482427537441 6464672.5433970680460333824,-616926.4954043276375159621 6463900.5587546769529581070,-616493.5739046344533562660 6461297.7216064278036355972,-616462.6270861934171989560 6460575.1449917182326316833,-616648.1966773458989337087 6459080.9204000355675816536,-617142.9004944300977513194 6457924.4423432126641273499,-618039.6903122602961957455 6456576.3459163447842001915,-618843.6396747701801359653 6455757.4328499510884284973,-619740.4294926003785803914 6455131.7287068972364068031,-621472.0041718903230503201 6454553.7440260341390967369,-629388.1558011806337162852 6452626.9561679409816861153,-630037.3710714877815917134 6452482.9330037198960781097,-630810.4849350476870313287 6452579.4681103099137544632,-632047.3557972507551312447 6452964.2331628119572997093,-633129.6038867422612383962 6453542.1074361680075526237,-635912.7024760651402175426 6455564.2900025611743330956,-636469.2999300315277650952 6456576.3459163447842001915,-636592.9758843046147376299 6457201.9853167394176125526,-636592.9758843046147376299 6457924.4423432126641273499,-635634.2924295922275632620 6468675.2390704853460192680,-634057.3405230136122554541 6472824.2370138922706246376,-633191.4975236244499683380 6473548.0531108947470784187,-617421.3105409058043733239 6481902.2164400974288582802,-613555.9638620885089039803 6483448.3897058935835957527,-611793.3310448691481724381 6483931.4865742065012454987,-610432.8955478841671720147 6483592.9504946963861584663,-609907.1335928677581250668 6483303.6574793346226215363,-609350.5361389013705775142 6482288.5153391053900122643,-609041.2905934784794226289 6481080.9779284242540597916,-608793.9386849350994452834 6480598.0469208294525742531,-608144.6120951366610825062 6480356.5050043519586324692,-605516.2475980191957205534 6479679.9120374908670783043,-604774.0805529003264382482 6479727.7290563872084021568,-604217.4830989340553060174 6479921.4342703213915228844,-586437.3113904496422037482 6493844.2239179955795407295,-573728.4104045444400981069 6505803.2786012487486004829,-573326.3244037983240559697 6506238.4881842527538537979,-562379.9449156242189928889 6519720.2007216988131403923,-562039.7525517601752653718 6520205.4166752127930521965,-561514.2132357262307778001 6521322.7271694773808121681,-559318.6589188107755035162 6527488.8217860972508788109,-559256.8766014197608456016 6528121.2326617483049631119,-559627.9044642335502430797 6529287.5419454993680119514,-560988.4512807097053155303 6532591.2870305376127362251,-560926.5576438275165855885 6533369.6954847564920783043,-558947.5197365056956186891 6540373.2428789725527167320,-558143.5703739959280937910 6541200.3828694252297282219,-547228.0263847715687006712 6546846.3099766355007886887,-545187.2061600559391081333 6547139.2345692599192261696,-542558.8416629385901615024 6547139.2345692599192261696,-534488.2898999165045097470 6548989.0541141862049698830,-532756.4925816440954804420 6549573.6629438968375325203,-532261.7887645600130781531 6549866.6843251455575227737,-531179.5406750683905556798 6551181.9892134414985775948,-530066.3457671357318758965 6553422.3482725294306874275,-528953.1508592030731961131 6558102.8932447293773293495,-528056.3610413699643686414 6562687.0380198908969759941,-523294.4471837076707743108 6566347.3337222440168261528,-517790.2549614353338256478 6571082.1642288379371166229,-508142.6399723435170017183 6581344.3519815802574157715,-507400.4729272246477194130 6582273.8473684089258313179,-506905.7691101405071094632 6583398.1841057818382978439,-506441.9007920034346170723 6586431.9832690181210637093,-506225.4957019010325893760 6588878.0462712580338120461,-506194.5488834627903997898 6590395.9045603703707456589,-506101.8197476307395845652 6595635.6715642046183347702,-506132.6552465806016698480 6596370.3236654140055179596,-506318.2248377330834046006 6596958.3042418546974658966,-505761.6273837667540647089 6608380.5921446401625871658,-503968.1590675974148325622 6623061.9129843506962060928,-503566.1843863425310701132 6623995.9613394439220428467,-502854.8528401735238730907 6624094.5212316401302814484,-498092.9389825084363110363 6623798.8451065737754106522,-493207.2378510817652568221 6623061.9129843506962060928,-492496.0176244039903394878 6622914.7114246273413300514,-491135.4708079306874424219 6621784.9565870827063918114,-487177.5063127753674052656 6619720.2741699311882257462,-485105.7392696214956231415 6619426.1703810859471559525,-483064.8077254174277186394 6619623.1816156450659036636,-481425.8508624683017842472 6620458.3212403720244765282,-479075.8964118210133165121 6622030.6729902913793921471,-472149.2637361931265331805 6629011.4551815437152981758,-471809.1826918176375329494 6629355.8341757701709866524,-471345.3143736833590082824 6629896.2461983226239681244,-468840.6258308333926834166 6633782.5244615664705634117,-468531.3802854105597361922 6634423.1525730984285473824,-469799.1979660546639934182 6640872.0837478553876280785,-470850.6105565963662229478 6654329.7067407211288809776,-470479.4713742913445457816 6654822.9246394680812954903,-468500.4334669694653712213 6656007.3373144594952464104,-467387.2385590368066914380 6656450.5591909773647785187,-458265.2742054730770178139 6658919.0068900147452950478,-453255.8971197761129587889 6659214.5313891069963574409,-443577.3353122460539452732 6659758.3385743526741862297,-432352.5457775960676372051 6661041.0492092268541455269,-422179.1688334906357340515 6663708.9564904095605015755,-411913.0627535531530156732 6662078.6332148471847176552,-404337.2148076153243891895 6659954.8999988278374075890,-402821.9338989393436349928 6659708.2214982789009809494,-400410.0858114127186127007 6660350.8807510212063789368,-399358.6732208681642077863 6660893.3415082683786749840,-397936.2327675126725807786 6661880.5986261591315269470,-396915.8783149004448205233 6662375.6940869214013218880,-390793.3063212707638740540 6660992.3462550239637494087,-390051.1392761518945917487 6660794.3379595801234245300,-383217.3470558444969356060 6658277.8815929768607020378,-382691.6964203192619606853 6657980.9694618340581655502,-382320.6685575054725632071 6657635.2074515735730528831,-381083.7976952996104955673 6655514.0480420421808958054,-380341.6306501807994209230 6654625.0655641015619039536,-379444.8408323504845611751 6653984.2765384409576654434,-378177.1344711976125836372 6653491.1091670170426368713,-377373.0737891966127790511 6653392.1951259449124336243,-375177.6307917723315767944 6653392.1951259449124336243,-367508.9423905140138231218 6653392.1951259449124336243,-337112.5988319564494304359 6657833.1393120000138878822,-335009.8849703642772510648 6665585.3860762137919664383,-335442.8064700574614107609 6666721.1978914896026253700,-335535.5356058895122259855 6667364.4083972629159688950,-335442.8064700574614107609 6669686.8621351048350334167,-334731.5862433796864934266 6675818.3632567543536424637,-329505.6927480890881270170 6687895.4508803253993391991,-317569.7943062526173889637 6700339.2401692178100347519,-308416.8831342477933503687 6708281.9527898402884602547,-295367.7897844786639325321 6721651.8780053677037358284,-293821.6733768528210930526 6724338.2434999896213412285,-284204.8938867108663544059 6738381.9839703468605875969,-275144.7118505381513386965 6750453.5700151724740862846,-273753.2182156209019012749 6751500.9796426724642515182,-272794.6460804024827666581 6752001.5482317414134740829,-270382.6866733846254646778 6752699.5199452098459005356,-267877.9981305347173474729 6753198.9040106711909174919,-267321.4006765683880075812 6753448.4277819134294986725,-266270.0994055179180577397 6753998.5576991466805338860,-265466.0387235168600454926 6754897.0035624913871288300,-264940.3880879916250705719 6756046.5910426154732704163,-264816.7121337184216827154 6756895.9841181430965662003,-264909.5525890417629852891 6757544.9133904082700610161,-265095.0108607030124403536 6758144.7874324973672628403,-265713.5019515515887178481 6759143.8964259419590234756,-266146.3121317534823901951 6759543.2865942474454641342,-266795.7500410431530326605 6759793.0053844498470425606,-273907.8409883351414464414 6754448.3974975654855370522,-287080.6102923746220767498 6742071.0710735591128468513,-292677.5316504790680482984 6736039.8856896292418241501,-296666.4429640754242427647 6730809.2905607484281063080)),((-575738.2838108161231502891 7445662.3215868351981043816,-581489.8279416320146992803 7445770.2241512890905141830,-582386.6177594651235267520 7445825.5493218069896101952,-582974.0507123813731595874 7446151.8189138257876038551,-588138.0505707897245883942 7449095.3408749038353562355,-589189.3518418401945382357 7449914.4276374652981758118,-594477.1389740130398422480 7454604.6953015578910708427,-594693.5440641153836622834 7455040.9381962884217500687,-596363.3364260144298896194 7460335.7556891534477472305,-601156.1971021178178489208 7479304.9535188870504498482,-601218.0907390000065788627 7480946.8646880760788917542,-601156.1971021178178489208 7482698.7262673750519752502,-600723.2756024218397215009 7484888.7174664484336972237,-599022.5364220845513045788 7492726.1545226909220218658,-598682.4553777090040966868 7493328.3919804077595472336,-597878.3946957107400521636 7494370.7231308631598949432,-593332.9972476392285898328 7498868.1786772878840565681,-592838.1821110638557001948 7499308.1117766723036766052,-589467.7618883160175755620 7502216.2121196007356047630,-586159.0126634679036214948 7502326.0925307748839259148,-584643.8430742803029716015 7502107.5190071118995547295,-583221.4026209219591692090 7501613.0834732400253415108,-582015.3672576688695698977 7500954.2805386865511536598,-578675.8938533620676025748 7498759.5326921911910176277,-577222.5065815654816105962 7497496.7834186805412173271,-575305.3623111229389905930 7495797.2470681583508849144,-574439.5193117309827357531 7494809.0180133040994405746,-573821.1395403764909133315 7493658.2508851466700434685,-566090.6688217271585017443 7470661.5068180840462446213,-565935.9347295245388522744 7469951.2167844949290156364,-564915.4689574239309877157 7452313.8906287150457501411,-564884.6334584740689024329 7451494.5500295581296086311,-565039.2562311855144798756 7450732.0305376620963215828,-565379.3372755582677200437 7450131.5305778924375772476,-566987.3473200662992894650 7448060.6635185629129409790,-568131.4890464399941265583 7447460.1744271237403154373,-575738.2838108161231502891 7445662.3215868351981043816)),((-559164.0361460965359583497 7503149.4548785416409373283,-559967.9855086063034832478 7503315.2842772519215941429,-560462.6893256932962685823 7503698.3460194766521453857,-569832.2282267800765112042 7514194.0817999914288520813,-570172.3092711528297513723 7514797.9966831970959901810,-578706.7293523119296878576 7533405.3470081146806478500,-578923.2457619027700275183 7534066.9279908332973718643,-579510.6787148189032450318 7536933.3768309159204363823,-579603.5191701422445476055 7537705.8451215503737330437,-579572.5723517010919749737 7538422.3980714641511440277,-576820.5319003105396404862 7542119.5918227070942521095,-576294.8812647825106978416 7542505.9771181223914027214,-575583.6610381047939881682 7542284.6723198099061846733,-574470.4661301720188930631 7541567.9585998225957155228,-565533.9600482695968821645 7533956.3959615742787718773,-560648.2589168429840356112 7526188.9937526304274797440,-557586.9729200295405462384 7515347.9150803945958614349,-557215.9450572157511487603 7513973.5884437793865799904,-557185.1095582658890634775 7512324.6404690435156226158,-557370.5678299271967262030 7504962.3078959751874208450,-557617.9197384706931188703 7504248.8576145675033330917,-558421.8691009776666760445 7503259.5462272781878709793,-559164.0361460965359583497 7503149.4548785416409373283)),((-664793.8764454351039603353 7516393.0264545530080795288,-670947.3952575059374794364 7517053.1489270376041531563,-671534.8282104220706969500 7517328.1836316678673028946,-673823.1116631695767864585 7518758.9579390995204448700,-674936.3065711023518815637 7519473.5533458655700087547,-675771.2027520503615960479 7520353.2224902408197522163,-676111.2837964259088039398 7521013.6840607589110732079,-676327.8002060195431113243 7521784.5582381589338183403,-676729.7748872715746983886 7523985.6687162136659026146,-677719.2938409339403733611 7534066.9279908332973718643,-677688.3470224929042160511 7534893.4879543641582131386,-677317.3191596789984032512 7537375.6853555440902709961,-676698.8280688333325088024 7539635.8654353143647313118,-664360.9549457419198006392 7563116.1797363320365548134,-663000.2968097745906561613 7564499.3038628213107585907,-657496.2159069934859871864 7570035.6849991707131266594,-656970.6765909566311165690 7570534.5317315002903342247,-646673.5123730896739289165 7580288.4833432398736476898,-640643.6695152892498299479 7585004.0025708498433232307,-639530.5859268477652221918 7585836.2537989197298884392,-635789.0265217948472127318 7588389.7062006164342164993,-635139.5886125080287456512 7588501.0231859032064676285,-633902.7177503022830933332 7587778.3908720752224326134,-632851.3051597605226561427 7583506.1376072801649570465,-632789.5228423695079982281 7582729.0567086441442370415,-632789.5228423695079982281 7581840.8322850158438086510,-632974.9811140308156609535 7580954.1081317737698554993,-633500.7430690473411232233 7579679.4093241794034838676,-634799.3962486413074657321 7578126.0998761691153049469,-635417.8873394897673279047 7577571.6791742900386452675,-636407.4062931521330028772 7576241.0730336820706725121,-637891.6290638985810801387 7573968.7197279091924428940,-639654.2618811180582270026 7571088.2456444287672638893,-640303.5884709164965897799 7569758.7612766847014427185,-650755.2641420065192505717 7548357.1669921502470970154,-657248.8639984499895945191 7535060.1998698618263006210,-662010.8891756035154685378 7524646.4415445849299430847,-664793.8764454351039603353 7516393.0264545530080795288)),((-638417.3910189151065424085 7589943.6933768726885318756,-639252.1758803747361525893 7589943.6933768726885318756,-639870.6669712205184623599 7590277.5168613130226731300,-640087.0720613228622823954 7590887.6317691532894968987,-639437.7454715273343026638 7592165.9916837643831968307,-638479.1733363060047850013 7593886.9354436751455068588,-633562.5253864382393658161 7597221.5368494736030697823,-632913.1987966427113860846 7597442.8280005119740962982,-632263.8722068442730233073 7597276.3588367188349366188,-631769.1683897601906210184 7596720.7541195014491677284,-631614.4342975575709715486 7596165.1896145483478903770,-631583.5987986077088862658 7595332.0182546870782971382,-631583.5987986077088862658 7592942.4283110080286860466,-631830.9507071482948958874 7592220.7775808572769165039,-632882.2519782016752287745 7591442.8144898163154721260,-634799.3962486413074657321 7590610.1554144909605383873,-636005.4316118971910327673 7590221.1456497646868228912,-638417.3910189151065424085 7589943.6933768726885318756)),((-627625.5229839611565694213 7595664.6758304946124553680,-628305.7963921978371217847 7595887.5224618297070264816,-628738.7178918938152492046 7596276.6193867400288581848,-630439.4570722338976338506 7600834.0224557518959045410,-630655.8621623362414538860 7601501.4306786367669701576,-628305.7963921978371217847 7609568.5682490030303597450,-628058.4444836571346968412 7610404.8903153417631983757,-624440.5610328745096921921 7609848.3341843830421566963,-624100.3686690105823799968 7609235.1069969609379768372,-624155.3604974618647247553 7608345.4135369891300797462,-626728.7331661308417096734 7597720.7515125907957553864,-627625.5229839611565694213 7595664.6758304946124553680)),((-625924.7838036210741847754 7611796.5575865879654884338,-626821.5736214512726292014 7611963.1416269848123192787,-627347.2242569765076041222 7612353.2551147406920790672,-628738.7178918938152492046 7616976.7891307510435581207,-628460.4191649092826992273 7618592.9562537530437111855,-628212.9559368772897869349 7619261.9116027252748608589,-626821.5736214512726292014 7621603.1129946140572428703,-625832.0546677918173372746 7622550.9550202786922454834,-624997.1584868408972397447 7622606.1590891517698764801,-622245.1180354503449052572 7622662.7687654243782162666,-621595.6801261607324704528 7622327.3324233889579772949,-621564.7333077224902808666 7621491.3147064875811338425,-621750.4142183633521199226 7620655.1874711913987994194,-623729.3408061967929825187 7614190.4231539648026227951,-624347.8318970424588769674 7613021.6662818389013409615,-625244.5103953843936324120 7612129.9297544891014695168,-625924.7838036210741847754 7611796.5575865879654884338)),((-618132.4194480923470109701 7634046.1933538932353258133,-621657.5737630429212003946 7634883.5821420866996049881,-621904.9256715864175930619 7635442.1603071531280875206,-622028.6016258567105978727 7636279.7017440479248762131,-621781.2497173132142052054 7636895.0966819357126951218,-619802.3231294825673103333 7639688.1229722509160637856,-617359.4169040236156433821 7640358.9228347931057214737,-613370.3942709360271692276 7640695.1496747722849249840,-616308.0043134819716215134 7636000.5781193887814879417,-617112.0649954801192507148 7634995.5760108726099133492,-618132.4194480923470109701 7634046.1933538932353258133)),((-680904.2557920205872505903 7688971.5201006643474102020,-675585.7444803890539333224 7686665.2835735436528921127,-674256.0331628656713292003 7686046.0731141436845064163,-672307.9420739820925518870 7683404.2137830695137381554,-667484.1345794375520199537 7675427.1118102883920073509,-667143.9422155736247077584 7674753.2296862173825502396,-665072.1751724197529256344 7668860.4051676299422979355,-664701.0359901146730408072 7667570.8004183741286396980,-651930.2413673300761729479 7662468.6353409998118877411,-645714.9402378683444112539 7661347.9715940030291676521,-644972.7731927494751289487 7661178.6958307139575481415,-643828.6314663757802918553 7660395.1128663672134280205,-636160.0543846086366102099 7654344.2149505326524376869,-629078.9102557577425613999 7648748.2799130156636238098,-628831.5583472142461687326 7647293.0925275487825274467,-628769.6647103348514065146 7645670.6582383047789335251,-629913.6951172174885869026 7639074.1129465876147150993,-630408.5102537928614765406 7637565.6519075101241469383,-634737.6139312532031908631 7630249.2361390953883528709,-636160.0543846086366102099 7628799.1669095354154706001,-646611.7300556986592710018 7621881.7125919712707400322,-650136.8843706492334604263 7620153.0732755511999130249,-650878.9400962768122553825 7619986.1105338940396904945,-652332.3273680733982473612 7620488.4144801758229732513,-653661.9273661085171625018 7621212.5285919252783060074,-654342.2007743450812995434 7622215.5235715247690677643,-653600.0337292291224002838 7623442.4991875784471631050,-652981.6539578718366101384 7623832.9964729128405451775,-652301.3805496323620900512 7624893.1614138623699545860,-651651.9426403455436229706 7626232.2308252081274986267,-651466.3730491930618882179 7626956.8878510380163788795,-651497.4311871253885328770 7627960.8358448566868901253,-651744.7830956659745424986 7628351.5634496649727225304,-652363.1628670232603326440 7628799.1669095354154706001,-653198.0590479741804301739 7628909.4645963711664080620,-654837.0159109233645722270 7628799.1669095354154706001,-655981.0463178058853372931 7628239.6666702739894390106,-666092.6409445203607901931 7622103.9170802831649780273,-669710.4130758116953074932 7619317.0919895535334944725,-697385.7743209081236273050 7610126.5069936262443661690,-703662.9690872520441189408 7611183.1746214488521218300,-704095.8905869452282786369 7611684.9021986331790685654,-705703.7893119648797437549 7613633.3917428404092788696,-706445.8450375924585387111 7614692.1469864565879106522,-706693.1969461359549313784 7615249.0547652505338191986,-708672.3461729490663856268 7620153.0732755511999130249,-708703.1816718989284709096 7620988.9450392834842205048,-708393.9361264732433483005 7624390.5687281526625156403,-706909.7133557267952710390 7625784.7772830910980701447,-705703.7893119648797437549 7626232.2308252081274986267,-701065.4400890816468745470 7626566.2313562659546732903,-700261.4907265718793496490 7626399.1288499012589454651,-699271.9717729095136746764 7625617.6918163308873772621,-698869.9970916574820876122 7625060.0302770650014281273,-698189.7236834180075675249 7623889.6152115678414702415,-693736.9440516872564330697 7622048.7166286185383796692,-675245.5521165251266211271 7629412.3537727268412709236,-670947.3952575059374794364 7631477.4989179829135537148,-670390.7978035395499318838 7631923.6771551398560404778,-669988.8231222846079617739 7632371.2883725911378860474,-669895.9826669641770422459 7633375.9461823049932718277,-670050.6054396756226196885 7634101.2792629031464457512,-670359.8509851013077422976 7634659.8003760473802685738,-670978.2307564557995647192 7634995.5760108726099133492,-671751.3446200157050043344 7634883.5821420866996049881,-676698.8280688333325088024 7632650.4807734247297048569,-679574.6557939854683354497 7630976.0818669088184833527,-680811.5266561885364353657 7630416.4226354788988828659,-686748.6403781569097191095 7629636.1832222435623407364,-688511.0505563967162743211 7629748.1003992343321442604,-689469.6226916179293766618 7630695.5437986664474010468,-690026.2201455842005088925 7631980.3557103760540485382,-690521.0352821595733985305 7634046.1933538932353258133,-690211.7897367367986589670 7635777.9702437324449419975,-683841.8658345636213198304 7644105.5374458534643054008,-677162.6963869676692411304 7647068.7463837368413805962,-675894.8787063235649839044 7647740.1933232387527823448,-671473.0458930339664220810 7650426.5703271478414535522,-670297.9573482191190123558 7651210.5432822806760668755,-669339.3852129979059100151 7652050.0132930614054203033,-668690.0586231994675472379 7653392.2282215254381299019,-668535.4358504880219697952 7654121.2745546055957674980,-668164.4079876742325723171 7656585.2678927034139633179,-668226.1903050652472302318 7657425.3278924627229571342,-668628.2763058113632723689 7657817.5622460534796118736,-669886.2978712664917111397 7657906.9017260335385799408,-678925.2178846986498683691 7654793.3406463116407394409,-681058.8785647319164127111 7653224.7407988160848617554,-681986.6152010033838450909 7653337.0033232029527425766,-682604.9949723606696352363 7653616.7598327985033392906,-704126.7260858950903639197 7666561.8454829007387161255,-705518.2197208123980090022 7668019.0880806446075439453,-703879.3741773545043542981 7678797.4141988204792141914,-703539.2931329789571464062 7679414.6216089874505996704,-688696.6201475491980090737 7687508.6565122948959469795,-683563.5671075818827375770 7689476.7540199654176831245,-682821.4000624630134552717 7689589.5502855889499187469,-680904.2557920205872505903 7688971.5201006643474102020)),((-690180.9542377869365736842 7564002.2430113358423113823,-694695.5161869085859507322 7564777.4328891141340136528,-697200.2047297556418925524 7565607.4966493323445320129,-693984.2959602307528257370 7573802.7585255075246095657,-693365.9161888734670355916 7574910.6972285006195306778,-691355.9314631104934960604 7577516.9976580627262592316,-690459.1416452802950516343 7578459.4101906763389706612,-687799.9416492101736366749 7580954.1081317737698554993,-687243.3441952437860891223 7581397.5572497285902500153,-683377.9975164294010028243 7582674.1387760406360030174,-682481.3190180874662473798 7582506.5915821185335516930,-682357.6430638171732425690 7581785.9207013463601469994,-687738.0480123279849067330 7566215.6559154046699404716,-689036.8125114131253212690 7564555.4867092901840806007,-689500.6808295502560213208 7564222.7786708874627947807,-690180.9542377869365736842 7564002.2430113358423113823)),((-713558.0473043757956475019 7618759.8886554623022675514,-714485.6726211558561772108 7618815.0654288716614246368,-716124.5181646137498319149 7620041.2961338991299271584,-716495.5460274276556447148 7620488.4144801758229732513,-714269.1562115622218698263 7624669.4704670021310448647,-711764.4676687123719602823 7626622.8702922686934471130,-711084.1942604756914079189 7626845.0114367129281163216,-710434.8676706772530451417 7626566.2313562659546732903,-710311.1917164069600403309 7625896.6382192233577370644,-711238.8170331871369853616 7622941.6076344205066561699,-711702.5740318329771980643 7621546.3104290068149566650,-712568.5283507134299725294 7619595.8091768035665154457,-712970.5030319682555273175 7619093.5637132814154028893,-713558.0473043757956475019 7618759.8886554623022675514)),((-690057.2782835166435688734 7650259.1478064712136983871,-693829.7845070078037679195 7651545.6351253492757678032,-694478.9997773149516433477 7651770.1125533133745193481,-696736.3364116213051602244 7654121.2745546055957674980,-696860.0123658917145803571 7654960.8624124936759471893,-696705.5009126714430749416 7655688.4453540705144405365,-694819.1921411788789555430 7657537.4504156326875090599,-694262.5946872124914079905 7657929.8921899255365133286,-692932.9946891774889081717 7658433.0779828028753399849,-691912.5289170768810436130 7658152.9436111617833375931,-690737.5516917532077059150 7657537.4504156326875090599,-686037.3088319879025220871 7653616.7598327985033392906,-685697.1164681239752098918 7653112.4799216054379940033,-685480.7113780215149745345 7652441.9721416393294930458,-685789.9569234472000971437 7651713.2873035967350006104,-686377.3898763634497299790 7651490.4235533857718110085,-690057.2782835166435688734 7650259.1478064712136983871)),((-767300.5371110781561583281 7646229.8225794527679681778,-770887.4737434196285903454 7646509.5207542572170495987,-776144.2027376601472496986 7648356.3087900653481483459,-778216.0811003053095191717 7657480.5823117960244417191,-777968.6178722705226391554 7658210.0183924464508891106,-776267.8786919304402545094 7660225.8581922017037868500,-774567.2508310816483572125 7661235.5914379972964525223,-752952.6792622238863259554 7668356.5327569469809532166,-752303.3526724282419309020 7668524.5538538070395588875,-751684.9729010709561407566 7668187.1017607403919100761,-750819.0185821906197816133 7667122.5390420444309711456,-749551.3122210348956286907 7664822.8869000077247619629,-749087.4439029005588963628 7663532.5502214934676885605,-749056.4970844595227390528 7662691.8191753616556525230,-750200.6388108332175761461 7661795.8945489320904016495,-752365.2463093076366931200 7661458.7392884017899632454,-754096.8209885975811630487 7661795.8945489320904016495,-755797.5601689376635476947 7661683.5078104771673679352,-756508.7803956154966726899 7661571.1227236548438668251,-757158.1069854138186201453 7661235.5914379972964525223,-757652.8108024980174377561 7660898.4608781281858682632,-765476.2332959589548408985 7653561.3317578630521893501,-766094.6130673162406310439 7652329.7227487266063690186,-767300.5371110781561583281 7646229.8225794527679681778)),((-743428.7402274052146822214 7669814.3195266835391521454,-744263.6364083561347797513 7670039.3338285284116864204,-745222.2085435773478820920 7672621.0086415195837616920,-745717.0236801527207717299 7674753.2296862173825502396,-745407.7781347299460321665 7675427.1118102883920073509,-736254.8669627250637859106 7685765.1272280197590589523,-731802.0873309944290667772 7689589.5502855889499187469,-728678.9076972960028797388 7692177.8419745219871401787,-724659.0495652613462880254 7695442.6619901461526751518,-724102.4521112949587404728 7695779.9016436329111456871,-722834.6344306508544832468 7696455.6417993549257516861,-719742.4016153935808688402 7696794.3451119465753436089,-718814.7762986134039238095 7696849.8849710999056696892,-718505.5307531905127689242 7696229.8479412319138646126,-718350.9079804790671914816 7695498.3946791412308812141,-718505.5307531905127689242 7694767.0115071609616279602,-721907.0091138707939535379 7686609.8177584111690521240,-722834.6344306508544832468 7684866.2912695864215493202,-723731.4242484811693429947 7683853.4313346464186906815,-731863.8696483825333416462 7675875.8605245305225253105,-732358.5734654695261269808 7675427.1118102883920073509,-733502.8265113317174836993 7674753.2296862173825502396,-743428.7402274052146822214 7669814.3195266835391521454)),((-685635.3341507357545197010 7733423.8133532665669918060,-687738.0480123279849067330 7734216.3565954193472862244,-691448.7719184310408309102 7737101.2504727263003587723,-691696.1238269745372235775 7737780.6634278474375605583,-691788.8529628065880388021 7739480.2764370609074831009,-691819.7997812448302283883 7740329.3078640596941113472,-691634.2301900923484936357 7741122.5701952194795012474,-691015.7390992465661838651 7742424.6700222818180918694,-685573.4405138535657897592 7748033.6259813448414206505,-684584.0328796824906021357 7748885.4488587854430079460,-683223.4860632063355296850 7748488.3005979405716061592,-680564.0634281565435230732 7746333.7345695439726114273,-679883.7900199170690029860 7745200.6845221538096666336,-679945.6836567992577329278 7744010.0824914481490850449,-680687.7393824269529432058 7738800.7118177721276879311,-681244.3368363932240754366 7736423.3244223734363913536,-681646.4228371393401175737 7735856.0891268383711576462,-683377.9975164294010028243 7733875.9867491759359836578,-684027.4354257161030545831 7733593.0697105331346392632,-685635.3341507357545197010 7733423.8133532665669918060)),((-703260.8830865059280768037 7746674.6618245746940374374,-704157.6729043362429365516 7746674.6618245746940374374,-704714.2703583025140687823 7746958.0654037864878773689,-708208.4778548119356855750 7749168.7306836582720279694,-717825.2573449538322165608 7759890.9843808347359299660,-718258.1788446470163762569 7760572.4351671617478132248,-718196.2852077677380293608 7761366.3798615131527185440,-714176.4270757301710546017 7766820.7554353112354874611,-711702.5740318329771980643 7769093.4759708065539598465,-710682.2195792207494378090 7769946.0237154727801680565,-706600.5791297951946035028 7770970.3944133082404732704,-704219.5665412184316664934 7771196.7774734115228056908,-698777.2679558254312723875 7768979.4963971134275197983,-698035.1009107065619900823 7768639.2048939717933535576,-697045.5819570441963151097 7767843.0682328445836901665,-696210.7970955873606726527 7766650.5538066504523158073,-695715.8706395179033279419 7765400.5477899685502052307,-695654.0883221297990530729 7764660.9911908647045493126,-695468.5187309773173183203 7759209.5946496361866593361,-696303.4149119253270328045 7752797.8342572487890720367,-696519.9313215189613401890 7752059.2978702094405889511,-697602.1794110105838626623 7750244.8079027310013771057,-698900.9439100957242771983 7748771.7717254515737295151,-701158.2805444021942093968 7747297.3882506629452109337,-702487.8805424371967092156 7746730.7704066531732678413,-703260.8830865059280768037 7746674.6618245746940374374)),((-831216.5151813800912350416 7748033.6259813448414206505,-832144.1404981602681800723 7748033.6259813448414206505,-836596.9201298909028992057 7748488.3005979405716061592,-838297.6593102309852838516 7748714.0154124023392796516,-840493.2136271464405581355 7749394.6697850516065955162,-841513.5680797586683183908 7750188.6734056873247027397,-841637.2440340289613232017 7751039.1043900754302740097,-841575.3503971467725932598 7752003.3541585551574826241,-840864.2414899603463709354 7753251.1599986683577299118,-828835.5025928033282980323 7771481.0963116846978664398,-828186.1760030050063505769 7771254.5000919839367270470,-825836.1102328692795708776 7769093.4759708065539598465,-825403.0774136820109561086 7768581.5021328534930944443,-821321.4369642564561218023 7756484.6460112733766436577,-821105.0318741539958864450 7755747.3858311343938112259,-827629.5785490386188030243 7748885.4488587854430079460,-828124.2823661256115883589 7748430.7505138125270605087,-831216.5151813800912350416 7748033.6259813448414206505)),((-732606.0366935014026239514 7769093.4759708065539598465,-734987.0492820809595286846 7769605.6889409003779292107,-734863.2620083165820688009 7770572.0912204943597316742,-733966.5835099775576964021 7771367.0809285240247845650,-729761.1557867904193699360 7773698.9026612853631377220,-729080.8823785508284345269 7774041.0589115936309099197,-728431.5557887553004547954 7773927.0051164235919713974,-723545.7433378373971208930 7772846.1385993212461471558,-722927.3635664801113307476 7772505.6739396685734391212,-722680.0116579394089058042 7771822.9480177508667111397,-723020.2040218034526333213 7771196.7774734115228056908,-723483.9610204464988783002 7770798.6670958334580063820,-732606.0366935014026239514 7769093.4759708065539598465)),((-803139.2905745200114324689 7783545.4806583048775792122,-803201.1842113992897793651 7782691.4057300575077533722,-804376.1614367229631170630 7779958.0274736341089010239,-804870.8652538099559023976 7779503.1064727054908871651,-808797.9942500125616788864 7781324.5936553860083222389,-811209.9536570303607732058 7781837.6318025905638933182,-813003.4219732024939730763 7781837.6318025905638933182,-816373.8421959504485130310 7780584.9204607559368014336,-818878.6420582886785268784 7780412.9757186239585280418,-819651.6446023574098944664 7780584.9204607559368014336,-821692.5761465615360066295 7782348.8596929777413606644,-822280.0090994776692241430 7783601.8620031997561454773,-822929.4470087644876912236 7785708.9791850121691823006,-823331.3103705309331417084 7787134.6291069323197007179,-825712.3229591076960787177 7797510.9036505632102489471,-825866.9457318191416561604 7798366.6473784530535340309,-826083.4621414127759635448 7804874.9768173359334468842,-826547.2191400558222085238 7838645.1134567204862833023,-826238.0849141242215409875 7839277.0828192858025431633,-825124.8900061915628612041 7840194.1517706615850329399,-820208.2420563237974420190 7842604.0526296412572264671,-816590.3586055439664050937 7841456.5425315601751208305,-811055.2195648277411237359 7838415.2631706120446324348,-809509.2144766930723562837 7837211.6059288652613759041,-808983.4525216766633093357 7836579.8089068550616502762,-804221.5386640115175396204 7830274.0186152849346399307,-804066.9158913000719621778 7829414.6665635276585817337,-804963.7057091303868219256 7828440.5358255095779895782,-806633.4980710294330492616 7827410.0282281637191772461,-807313.7714792659971863031 7827180.3123987615108489990,-808055.8272048936923965812 7827295.1694429172202944756,-810993.4372474396368488669 7829128.1687867753207683563,-811302.6827928624115884304 7826321.5173279317095875740,-808086.7740233347285538912 7791351.6153712701052427292,-803139.2905745200114324689 7783545.4806583048775792122)),((-661918.1600397742586210370 7817105.8563358392566442490,-664453.6840815711766481400 7817564.6846419069916009903,-665721.5017622152809053659 7818136.6114055309444665909,-667514.9700783874141052365 7819280.5943950461223721504,-668844.6813959137070924044 7820710.9189692307263612747,-669926.9294854053296148777 7822370.8705623568966984749,-670050.6054396756226196885 7823287.5438298834487795830,-669895.9826669641770422459 7824088.0749338585883378983,-669586.7371215414023026824 7824660.4941557068377733231,-668566.3826689291745424271 7825519.3071479909121990204,-666556.5092626574914902449 7825920.2985368277877569199,-663093.1372650979319587350 7825519.3071479909121990204,-662381.9170384173048660159 7825232.9566187048330903053,-660557.5019038068130612373 7824088.0749338585883378983,-660124.5804041108349338174 7823686.9743719510734081268,-659104.2259515015175566077 7821913.3997483728453516960,-658856.8740429580211639404 7821226.4211233425885438919,-658764.0335876375902444124 7820483.0542908245697617531,-658825.9272245168685913086 7819510.0708995973691344261,-659382.5246784832561388612 7818251.3314756704494357109,-659908.1753140112850815058 7817735.8256493629887700081,-661145.0461762142367660999 7817163.9291197257116436958,-661918.1600397742586210370 7817105.8563358392566442490)),((-664732.0941280469996854663 7866173.6906613605096936226,-665504.9853526245569810271 7866289.1409389581531286240,-666680.1852169277844950557 7868016.1309835584834218025,-666834.6966701479395851493 7868764.3877861686050891876,-666896.5903070301283150911 7871473.1617560992017388344,-665938.0181718089152127504 7875966.9064698787406086922,-664175.4966740807285532355 7878101.7436213130131363869,-663649.7347190642030909657 7878562.6237914720550179482,-663000.2968097745906561613 7878331.3497779136523604393,-662783.8917196722468361259 7877639.2308822218328714371,-662722.1094022812321782112 7876947.1753730922937393188,-664330.0081273008836433291 7867728.3755705067887902260,-664732.0941280469996854663 7866173.6906613605096936226)),((-327217.5206148328143171966 8122256.1049753436818718910,-328176.0927500540274195373 8122375.5594781693071126938,-329258.3408395484439097345 8123268.8481088783591985703,-330680.7812929039355367422 8124818.6841298276558518410,-331732.0825639544054865837 8126606.3827060386538505554,-338071.1709671777207404375 8141164.8613730743527412415,-337700.0317848726408556104 8141761.9490661276504397392,-337174.3811493474058806896 8142239.3096001911908388138,-335071.6672877524397335947 8144091.8439540863037109375,-334453.2875163979479111731 8144390.6066143354400992393,-331144.5382915498339571059 8144868.1355154002085328102,-324619.9916166624170728028 8145644.2928006444126367569,-322733.7941646638792008162 8145526.1860490273684263229,-322177.1967106975498609245 8145225.8720280602574348450,-321311.3537113055353984237 8144150.9932781057432293892,-320352.7815760842640884221 8142478.8614945756271481514,-320229.1056218139710836112 8141642.1836746307089924812,-324372.6397081217728555202 8123090.6107930103316903114,-324867.4548446971457451582 8122673.2389830201864242554,-327217.5206148328143171966 8122256.1049753436818718910)),((-359716.5780349853448569775 8132032.1771923750638961792,-366117.4487555996165610850 8132032.1771923750638961792,-366890.5626191567280329764 8132151.7883329940959811211,-367447.1600731230573728681 8132569.4760318957269191742,-381949.5293752003926783800 8152639.8082044757902622223,-382166.0457847940269857645 8153357.7678188877180218697,-382166.0457847940269857645 8154254.7761839749291539192,-380496.2534228921867907047 8159700.9549947949126362801,-379846.8155136054265312850 8161079.0209944993257522583,-378269.8636070268694311380 8162516.6373222228139638901,-372827.5650216366630047560 8164612.7600420080125331879,-371312.3954324491205625236 8164732.8952381443232297897,-359902.1476261378847993910 8156049.5478410180658102036,-359376.3856711214175447822 8155571.3019034899771213531,-357861.3274014250491745770 8153237.8159430967643857002,-349141.2264096250291913748 8138178.4206436891108751297,-348955.6568184724892489612 8137461.9215934518724679947,-349203.1200465043657459319 8134417.8906672578305006027,-349728.7706820324528962374 8134001.6030156211927533150,-351800.5377251863828860223 8133224.9385366979986429214,-354088.7098584398045204580 8132807.2141293892636895180,-359716.5780349853448569775 8132032.1771923750638961792)),((-650940.8337331590009853244 7781665.4537310823798179626,-652239.4869127529673278332 7779104.3557837270200252533,-655579.0716365509433671832 7773188.2564562307670712471,-656011.9931362469214946032 7772618.0685839969664812088,-659908.1753140112850815058 7768013.2965631103143095970,-661175.9929946553893387318 7767388.8718715608119964600,-665690.5549437770387157798 7765626.7650331659242510796,-667731.4864879810484126210 7764944.8611572692170739174,-669246.6560771686490625143 7764660.9911908647045493126,-669772.3067126938840374351 7765172.7010596394538879395,-671503.9927114722086116672 7769889.7436121432110667229,-671782.1801189655670896173 7771310.7903017997741699219,-671627.6686657454119995236 7772049.5611863331869244576,-666896.5903070301283150911 7782976.1550377663224935532,-665041.2283539786003530025 7786791.8826122293248772621,-671318.4231203197268769145 7796484.5903450027108192444,-676111.2837964259088039398 7785995.4823604188859462738,-676513.3697971692308783531 7785538.5590157033875584602,-677440.9951139522017911077 7785708.9791850121691823006,-678677.8659761551534757018 7786392.7490999111905694008,-687583.4252396165393292904 7795058.8264241442084312439,-702920.8020421331748366356 7792377.4401110168546438217,-703353.7235418263589963317 7792776.6830875966697931290,-717670.6345722425030544400 7814132.3829986350610852242,-718289.0143435968784615397 7815390.4386823251843452454,-721257.5712045810651034117 7821627.3914837995544075966,-721504.9231131245614960790 7822314.4064546646550297737,-721443.1407957336632534862 7823229.4241122854873538017,-715351.4043010538443922997 7828440.5358255095779895782,-714856.7004839696455746889 7828841.8880936847999691963,-713898.1283487484324723482 7828900.0508747342973947525,-712259.2828052905388176441 7827581.5979150198400020599,-709538.0778528470546007156 7824660.4941557068377733231,-706415.0095386425964534283 7822141.5135000469163060188,-705240.0323133190395310521 7821341.1879651453346014023,-704590.5944040322210639715 7821111.6560192611068487167,-702889.8552236921386793256 7821168.3172089494764804840,-702456.9337239961605519056 7821683.8504735743626952171,-702395.1514066079398617148 7822542.3261973010376095772,-726576.1938357037724927068 7843695.1129715852439403534,-727287.4140623816056177020 7844040.4415323473513126373,-731771.0291930621024221182 7839047.0068216221407055855,-731771.0291930621024221182 7837326.6151750516146421432,-730750.6747404498746618629 7832795.7048286721110343933,-730626.9987861795816570520 7832050.9958565970882773399,-730657.9456046206178143620 7831076.7324042050167918205,-731059.9202858755597844720 7829414.6665635276585817337,-731369.1658312984509393573 7828900.0508747342973947525,-731987.5456026557367295027 7828555.4119750792160630226,-732791.4949651627102866769 7828670.2898664269596338272,-737491.6265054368413984776 7830790.1717676045373082161,-744634.6642711699241772294 7834230.0278706122189760208,-745840.6996344230137765408 7834859.9775811564177274704,-746922.9477239146362990141 7835719.9476449796929955482,-748159.8185861204983666539 7837383.1912799999117851257,-748376.2236762228421866894 7838071.6365813584998250008,-748592.7400858164764940739 7839677.3573641553521156311,-748561.7932673753239214420 7843064.4278551321476697922,-753942.3095353774260729551 7848059.0666098510846495628,-754375.2310350706102326512 7848576.4335481449961662292,-754622.5829436141066253185 7849150.7067721011117100716,-755395.4741681915475055575 7851850.8092466006055474281,-755271.7982139212545007467 7852655.8140139542520046234,-751128.2641276133945211768 7861511.7600345788523554802,-738883.2314598453231155872 7884909.0200083097442984581,-737244.2745968961389735341 7884965.9531716629862785339,-736471.2720528275240212679 7884736.1452953089028596878,-735512.6999176061945036054 7883870.5845353407785296440,-733193.5809659088263288140 7881388.9321941649541258812,-732296.7911480785114690661 7880235.5228319214656949043,-731616.5177398419473320246 7879139.1787895429879426956,-730874.3506947230780497193 7877178.4070259826257824898,-730750.6747404498746618629 7874584.8250432880595326424,-730626.9987861795816570520 7873834.5395373199135065079,-729946.6140584516106173396 7872625.2344198366627097130,-719495.0497068528784438968 7864850.1802075924351811409,-715784.3258007498225197196 7864043.8767328895628452301,-709723.6474439995363354683 7868016.1309835584834218025,-711980.9840783060062676668 7884332.0245320610702037811,-714702.1890307466965168715 7890971.9028529813513159752,-714825.8649850197834894061 7891549.4058504449203610420,-714578.5130764764035120606 7892472.8369492143392562866,-713248.8017589501105248928 7893918.7326100040227174759,-706384.0627202015602961183 7901375.5957725411280989647,-701776.6603157594799995422 7901664.8440894046798348427,-698746.3211373842786997557 7898773.6923043765127658844,-688109.1871946330647915602 7886756.0343129066750407219,-687336.0733310730429366231 7885542.9970754394307732582,-686068.2556504289386793971 7883177.9580443715676665306,-685511.6581964626675471663 7881908.5846560634672641754,-684181.9468789363745599985 7878562.6237914720550179482,-683934.5949703956721350551 7877121.3249458475038409233,-683130.6456078859046101570 7859786.2546087922528386116,-684243.8405158185632899404 7846279.7762876143679022789,-684367.6277895801467821002 7843178.0795488543808460236,-682914.2405177834443747997 7824374.1760758152231574059,-682079.3443368354346603155 7822943.3661821372807025909,-656692.2665444836020469666 7809731.3453689971938729286,-655300.7729095664108172059 7809160.0521987508982419968,-654558.6058644475415349007 7809331.2093394454568624496,-649363.7705070892116054893 7812131.7817999981343746185,-642189.7859229178866371512 7815962.2014425937086343765,-640148.9656982050510123372 7816763.4007026748731732368,-639314.0695172541309148073 7816706.7724281931295990944,-628862.3938461641082540154 7813446.1090800398960709572,-628522.2014823001809418201 7812759.6914533944800496101,-630222.9406626402633264661 7802989.8662061607465147972,-630501.2393896249122917652 7802133.6009787861257791519,-631212.5709357938030734658 7801048.7123676566407084465,-646055.1326017323881387711 7784000.6438278378918766975,-647477.4617355966474860907 7782691.4057300575077533722,-648652.5502804114948958158 7781951.8044139267876744270,-649332.8236886481754481792 7781723.4609073279425501823,-650940.8337331590009853244 7781665.4537310823798179626)),((-673606.5952535760588943958 7826722.5511717032641172409,-674627.0610256794607266784 7826779.2543275719508528709,-675276.4989349662791937590 7827123.6062415624037384987,-675523.8508435097755864263 7827696.4610423119738698006,-676760.7217057127272710204 7831764.6046938020735979080,-676791.5572046625893563032 7832624.0170202022418379784,-676637.0457514424342662096 7841112.9844846883788704872,-676049.5014790350105613470 7848633.3005767501890659332,-675183.6584796457318589091 7852886.3040355583652853966,-674998.2002079845406115055 7853633.0654424177482724190,-671442.0990745929302647710 7861339.4196276441216468811,-671009.2888943881262093782 7861856.2454785956069827080,-668257.1371235062833875418 7864791.7405201056972146034,-665690.5549437770387157798 7862143.9846739172935485840,-665350.4738994014915078878 7861453.3461393844336271286,-665319.5270809632493183017 7860590.4473247332498431206,-666803.8611712008714675903 7831994.2532315347343683243,-667298.5649882850702852011 7830503.6219702130183577538,-667607.8105337079614400864 7829874.2407008716836571693,-668628.2763058113632723689 7828900.0508747342973947525,-670916.4484390676952898502 7827581.5979150198400020599,-673606.5952535760588943958 7826722.5511717032641172409)),((-810777.0321573371766135097 7841801.7689958875998854637,-814147.5636995763052254915 7842260.4425088372081518173,-814889.6194252038840204477 7842604.0526296412572264671,-821321.4369642564561218023 7846508.4183434061706066132,-822434.6318721891148015857 7847254.5506350584328174591,-824289.9938252406427636743 7850989.0049488563090562820,-824815.6444607658777385950 7852310.0921453200280666351,-824877.5380976480664685369 7853115.1457784250378608704,-824475.4520969019504263997 7855587.7400682382285594940,-824228.1001883612480014563 7856334.5611695535480976105,-823795.1786886652698740363 7856852.7012757556512951851,-819064.2116494412766769528 7860590.4473247332498431206,-818321.9332848311169072986 7860821.1792277200147509575,-812694.1764277797192335129 7859612.2967918114736676216,-809973.0827948274090886116 7858980.4908027257770299911,-806416.9816614357987418771 7858117.8740360643714666367,-802025.9843470960622653365 7854897.6998469531536102295,-801778.6324385525658726692 7854207.7225553095340728760,-801747.7969396027037873864 7846221.2731461282819509506,-801995.1488481462001800537 7845416.9524751473218202591,-805891.3310259105637669563 7843064.4278551321476697922,-806478.7639788266969844699 7842777.6198937362059950829,-809169.0221128291450440884 7842088.7466019783169031143,-810777.0321573371766135097 7841801.7689958875998854637)),((-805242.0044361121254041791 7863756.2724463250488042831,-814271.2396538465982303023 7863871.6858208188787102699,-815013.2953794742934405804 7864100.8601484736427664757,-815539.0573344907024875283 7864619.3253505919128656387,-815693.5687877108575776219 7868707.3692137449979782104,-815446.2168791702715680003 7872164.7161779273301362991,-817950.9054220173275098205 7873546.3550861412659287453,-825062.9963693093741312623 7877121.3249458475038409233,-829948.6975007359869778156 7876256.8436097716912627220,-831309.2443172121420502663 7875852.9685768820345401764,-832329.7100893127499148250 7875966.9064698787406086922,-833350.1758614162681624293 7876717.6112698679789900780,-838946.9859000294236466289 7880869.1077992143109440804,-839379.9073997226078063250 7881273.2510386267676949501,-839720.0997635865351185203 7881850.0129269957542419434,-839720.0997635865351185203 7882831.0454295976087450981,-833937.6088143324013799429 7893398.0451419055461883545,-833102.8239528727717697620 7894553.4665402770042419434,-831865.8417711785295978189 7895190.1264471393078565598,-827382.2266404979163780808 7894842.4535497147589921951,-809849.4068405569996684790 7895478.9297529878094792366,-809416.3740213698474690318 7896114.0029696635901927948,-806478.7639788266969844699 7899756.9260776657611131668,-801191.1994856364326551557 7901839.5646060863509774208,-800232.6273504151031374931 7901896.8338320590555667877,-787616.3441808507777750492 7892531.6991114467382431030,-786750.5011814616154879332 7891549.4058504449203610420,-786565.0429098003078252077 7890683.0639442158862948418,-795656.0604449228849261999 7866116.6916231373324990273,-796367.3919910918921232224 7864964.1596209304407238960,-797975.2907161087496206164 7864676.3131026867777109146,-805242.0044361121254041791 7863756.2724463250488042831)),((-801995.1488481462001800537 7904963.7670821649953722954,-803695.8880284862825646996 7905253.1529692234471440315,-803633.9943916040938347578 7906178.2622187910601496696,-802922.7741649263771250844 7907394.6201119190081954002,-801345.7109388565877452493 7909709.9304114840924739838,-800356.3033046883065253496 7910578.4592534480616450310,-798593.6704874660354107618 7911563.0240568816661834717,-797789.7211249590618535876 7911795.3183526406064629555,-797047.6653993285726755857 7911679.1703105298802256584,-795346.9262189912842586637 7909479.3680197680369019508,-795408.7085363793885335326 7908494.8666790891438722610,-795934.4704913959139958024 7908030.6968740494921803474,-796491.0679453621851280332 7907683.8908374262973666191,-801129.3058487570378929377 7905021.0600231559947133064,-801995.1488481462001800537 7904963.7670821649953722954)),((-954626.5271892031887546182 7925129.6127497013658285141,-955399.5297332719201222062 7925187.0590954320505261421,-958028.0055498805595561862 7926406.2782483883202075958,-958677.4434591701719909906 7926755.4090198026970028877,-959728.6334107293514534831 7930180.2969843065366148949,-959419.4991847977507859468 7930877.2520577358081936836,-955523.3170070333871990442 7930411.4932919247075915337,-953513.3322812705300748348 7929540.6768891820684075356,-952307.5195569969946518540 7928786.5801868764683604240,-952029.1095105211716145277 7928322.6224836604669690132,-952369.3018743880093097687 7927683.1600318755954504013,-954626.5271892031887546182 7925129.6127497013658285141)),((-787430.7745896981796249747 7941686.0055912025272846222,-787894.6429078353103250265 7942150.7870398368686437607,-787770.9669535622233524919 7944826.7693878216668963432,-787430.7745896981796249747 7945525.2904836768284440041,-782885.2658221382880583405 7949598.1104610562324523926,-782297.7215497308643534780 7949889.2152951927855610847,-780040.4962349156849086285 7950471.2491385219618678093,-779174.6532355237286537886 7950530.3562806798145174980,-778525.2153262369101867080 7950239.4362448817119002342,-778587.1089631190989166498 7949249.5958655448630452156,-778958.1368259328883141279 7944419.4159345766529440880,-779112.7595986443338915706 7943662.1531161759048700333,-779947.7670990836340934038 7942441.6040757866576313972,-787430.7745896981796249747 7941686.0055912025272846222)),((-720670.0269321736413985491 7956470.2657709643244743347,-720546.3509779033483937383 7955596.4309239611029624939,-720670.0269321736413985491 7954782.0477885073050856590,-727813.0646979068405926228 7949772.4785993508994579315,-728493.4494256346952170134 7949481.3782793292775750160,-731430.9481486893491819501 7948725.2825949238613247871,-732203.9506927580805495381 7948608.5639465516433119774,-737058.8163252320373430848 7949366.3244531759992241859,-737831.8188693007687106729 7949540.6872226530686020851,-740614.8061391324736177921 7950530.3562806798145174980,-741635.2719112358754500747 7951403.6031045736744999886,-742099.0289098789216950536 7951985.7540601119399070740,-744943.9098165928153321147 7958511.1471517411991953850,-745685.9655422203941270709 7960493.1864522015675902367,-746366.2389504598686471581 7964516.5794345876201987267,-746613.5908590005710721016 7967083.5220405189320445061,-746520.9730426626047119498 7968893.2158791339024901390,-745562.2895879501011222601 7971811.1980586852878332138,-745222.2085435773478820920 7972452.7385530890896916389,-741697.0542286268901079893 7976659.0013084486126899719,-741295.0795473719481378794 7977125.9482289021834731102,-740583.9706401826115325093 7977477.4395719552412629128,-737986.4416420122142881155 7977477.4395719552412629128,-737151.5454610640881583095 7977360.2739647664129734039,-736378.5429169953567907214 7977008.7881000870838761330,-734554.1277823849814012647 7976833.9978371746838092804,-733997.5303284185938537121 7977068.3144723521545529366,-733626.5024656048044562340 7977827.2644502716138958931,-734182.9886000799015164375 7978353.8205026816576719284,-734739.5860540462890639901 7978645.8299302291125059128,-735574.5935544855892658234 7978820.8727197982370853424,-742501.0035911338636651635 7979522.3471052441745996475,-743428.7402274052146822214 7979580.2100927950814366341,-744109.0136356446892023087 7979405.1496189218014478683,-744634.6642711699241772294 7978878.7303074216470122337,-752674.4918547333218157291 7966966.5182834491133689880,-748716.4160400867694988847 7957752.2556198462843894958,-748376.2236762228421866894 7957054.4870444852858781815,-747943.3021765268640592694 7956646.2615891080349683762,-746428.1325873392634093761 7956179.1160528566688299179,-745438.6136336798081174493 7955247.6384827783331274986,-744170.7959530327934771776 7953558.2269180696457624435,-743583.3630001166602596641 7952276.7415983201935887337,-743336.0110915759578347206 7951694.5681378180161118507,-741913.4593187292339280248 7946455.3561793249100446701,-741449.7023200833937153220 7943139.6945446738973259926,-741635.2719112358754500747 7942267.4056255519390106201,-749767.7173111372394487262 7930818.2989529734477400780,-774876.4963765044230967760 7910288.8690335713326931000,-775402.1470120325684547424 7909884.8373241545632481575,-776175.2608755896799266338 7909709.9304114840924739838,-776824.4761458968278020620 7909942.1676476104184985161,-777443.0785562365781515837 7910463.9978594612330198288,-792594.8857675979379564524 7928379.8841866310685873032,-792934.9668119734851643443 7929076.6727590663358569145,-793089.5895846848143264651 7929773.5257634706795215607,-793336.9414932255167514086 7932386.1485914569348096848,-793182.3187205140711739659 7933316.2661125017330050468,-792409.3161764453398063779 7933373.7748991921544075012,-790801.3061319373082369566 7933083.3077363111078739166,-789502.6529523433418944478 7932269.6828811541199684143,-788543.9694976309547200799 7930585.4177934592589735985,-786565.0429098003078252077 7929773.5257634706795215607,-782452.4556419336004182696 7932154.8916895156726241112,-760343.0689365004654973745 7946514.4317998429760336876,-760775.9904361936496570706 7947096.2047789441421627998,-765816.3143403317080810666 7953150.1910642273724079132,-772000.6686513522872701287 7956761.6365282991901040077,-773237.6508330465294420719 7957403.3632442764937877655,-777566.7545105068711563945 7958802.3871165215969085693,-778432.4861904048593714833 7958743.2149719782173633575,-781679.3417783735785633326 7957228.8182288901880383492,-782545.1847777627408504486 7957286.5099434768781065941,-783194.5113675611792132258 7957520.2183597078546881676,-787616.3441808507777750492 7959442.8189643006771802902,-791790.9364050909643992782 7963933.4582474185153841972,-791914.6123593612574040890 7964633.5450868243351578712,-788605.8631345131434500217 7969008.5686866920441389084,-785420.9011834264965727925 7971577.0452094133943319321,-785668.2530919699929654598 7974964.7744164736941456795,-789688.1112240046495571733 7982152.3016326762735843658,-793336.9414932255167514086 7993269.8519307244569063187,-793522.5110843779984861612 7994733.5469595398753881454,-793491.5642659397562965751 7995671.4061694080010056496,-789750.0048608868382871151 8005634.5332959173247218132,-785822.8758646814385429025 8013733.6948617883026599884,-785420.9011834264965727925 8014378.8262904621660709381,-783751.1088215275667607784 8016493.4368419172242283821,-782792.5366863062372431159 8017491.9634408326819539070,-780133.2253707449417561293 8016904.7155841961503028870,-779360.2228266762103885412 8016728.9942470602691173553,-770300.0407905034953728318 8012853.1757603893056511879,-769001.2762914182385429740 8012148.5812776368111371994,-768753.9243828747421503067 8011502.1624122951179742813,-769001.2762914182385429740 8010739.5907647507265210152,-770361.9344273856841027737 8010210.9706140225753188133,-772124.3446056226966902614 8006924.9379145596176385880,-765228.7700679242843762040 7990693.9523769477382302284,-764548.4966596876038238406 7989641.0427555870264768600,-763868.2232514509232714772 7989171.6321176402270793915,-763651.8181613485794514418 7990050.8529532160609960556,-765129.9183600998949259520 8004928.2938264198601245880,-757549.5063150412170216441 8008575.9065968403592705727,-751448.1963441522093489766 8006331.3369063120335340500,-750819.6864991351030766964 8008857.0530969668179750443,-755478.8524667968740686774 8015665.8727857079356908798,-757869.3272120916517451406 8022075.6411512596532702446,-758951.6866210744483396411 8026368.0272229677066206932,-758704.2233930396614596248 8027367.8694736715406179428,-756725.1854857178404927254 8031073.4454955356195569038,-742841.1959549977909773588 8041025.3970633633434772491,-741202.3504115398973226547 8041085.4333740184083580971,-740522.0770033032167702913 8040908.9322353173047304153,-739934.5327308957930654287 8039494.5276033077389001846,-730286.8064223156543448567 8043265.7350367736071348190,-729173.7228338741697371006 8044031.6221584426239132881,-707466.3108096930664032698 8064043.7281609382480382919,-702487.8805424371967092156 8068655.7719786865636706352,-700508.8426351153757423162 8070607.0664195213466882706,-698746.3211373842786997557 8074868.7607897184789180756,-698220.6705018590437248349 8075341.8639872856438159943,-697633.1262294516200199723 8075756.5997826606035232544,-696829.1768669418524950743 8075815.2102414192631840706,-696087.0098218229832127690 8075579.2793005006387829781,-695128.4376866045640781522 8074927.3642917294055223465,-694046.1895971101475879550 8073981.2402468603104352951,-692778.2605969748692587018 8072323.2547035785391926765,-686408.4480142928659915924 8057190.8337126951664686203,-686377.3898763634497299790 8056186.7913654660806059837,-690644.7112364327767863870 8042500.1383879883214831352,-693798.7263690782710909843 8034369.8655051235109567642,-699055.5666828071698546410 8028014.1747573809698224068,-703972.2146326749352738261 8022899.0452594198286533356,-704460.6845582755049690604 8017467.8551338436082005501,-704479.2749132377794012427 8016939.3951310329139232635,-704386.4344579173484817147 8016516.6962206689640879631,-704126.7260858950903639197 8016164.2186422776430845261,-703737.1078681189101189375 8015952.9937641918659210205,-699655.3560992020647972822 8014296.5964280683547258377,-699228.7798104838002473116 8014156.2369969626888632774,-698746.3211373842786997557 8014261.9290954815223813057,-698375.2932745704893022776 8014437.5926568955183029175,-697466.1469932614127174020 8015248.9537850487977266312,-692500.0731894843047484756 8016551.3739738147705793381,-691139.4150535169756039977 8018079.2572728907689452171,-687119.6682409706991165876 8021781.7261899784207344055,-686593.9062859541736543179 8022251.4877907298505306244,-685728.1746060563018545508 8022075.6411512596532702446,-684398.4632885300088673830 8021310.5129202296957373619,-684274.7873342595994472504 8020606.6046414244920015335,-685542.6050149037037044764 8013322.3785514980554580688,-686222.8784231403842568398 8012030.8735322710126638412,-687088.7214225324569270015 8010915.3831741800531744957,-692932.9946891774889081717 8005987.3612213982269167900,-697280.5774021089309826493 8004509.4375922940671443939,-704553.4136941075557842851 8005811.0507594170048832893,-708734.1284903400810435414 8000709.9760819813236594200,-709630.8069886791054159403 7995846.6352075962349772453,-709878.2702167109819129109 7995085.8629775904119014740,-710465.8144891184056177735 7994616.1119595533236861229,-723267.5559303440386429429 7987826.3474211450666189194,-728926.2596058394992724061 7986656.5289644505828619003,-734832.3151898783398792148 7985194.4066915316507220268,-736965.9758699116064235568 7984550.3039665548130869865,-736842.2999156413134187460 7983790.4642846547067165375,-734121.2062826890032738447 7983790.4642846547067165375,-723236.6091119057964533567 7985719.7945190295577049255,-714702.1890307466965168715 7988821.2787511013448238373,-713588.8828033227473497391 7989523.6873462749645113945,-712723.0398039335850626230 7989581.4167081136256456375,-712011.8195772558683529496 7989348.3950684312731027603,-711393.4398058985825628042 7988061.0070491367951035500,-707311.6880369817372411489 7974263.7251747092232108116,-707621.0449018958024680614 7973622.0301446886733174324,-710898.6246693232096731663 7969242.6415702234953641891,-718876.5586160043021664023 7960608.4105123337358236313,-720670.0269321736413985491 7956470.2657709643244743347)),((-761301.6410717188846319914 8009859.4229130959138274193,-764672.1726139578968286514 8010270.5494405198842287064,-765105.0941136539913713932 8010797.6947713699191808701,-766867.6156113849719986320 8018842.5753356879577040672,-766991.2915656553814187646 8019723.7977412352338433266,-766960.4560667055193334818 8020546.9436763981357216835,-766063.6662488752044737339 8020900.4735544379800558090,-763682.6536602984415367246 8020253.0886142896488308907,-763002.3802520589670166373 8020019.1129028648138046265,-758054.8968032441334798932 8015494.8317113332450389862,-756354.1576229040510952473 8013262.7754890210926532745,-756013.9652590401237830520 8010915.3831741800531744957,-756354.1576229040510952473 8010152.8711501006036996841,-757219.8893028020393103361 8010093.2932578660547733307,-761301.6410717188846319914 8009859.4229130959138274193)),((-314415.7791736071230843663 8154254.7761839749291539192,-319394.2094408630509860814 8157366.1753244427964091301,-330618.8876560217468068004 8167308.6114223692566156387,-334886.2090160911320708692 8167428.7901212638244032860,-340328.3962819929001852870 8166529.9781254148110747337,-346265.5100039612734690309 8164853.0323724383488297462,-347285.8644565735012292862 8163834.4085745373740792274,-347285.8644565735012292862 8162936.2459341324865818024,-353130.2490427098236978054 8161138.3053241176530718803,-354923.7173588819568976760 8160958.9447254370898008347,-355665.7730845095939002931 8161199.0991994673386216164,-367725.4588001076481305063 8168688.0861030211672186852,-368838.6537080403068102896 8169587.1586798103526234627,-372425.5903403817792423069 8173004.6682517323642969131,-374157.1650196717237122357 8178705.0871962811797857285,-374373.6814292625058442354 8179425.5645581595599651337,-374930.2788832288933917880 8182547.5026265401393175125,-374930.2788832288933917880 8183448.0348873659968376160,-373074.9169301773654296994 8203416.1640427159145474434,-372951.2409759070142172277 8204198.6678917976096272469,-369302.4107066861470229924 8208416.5876278001815080643,-364880.5778933937544934452 8210707.7649166425690054893,-364262.1981220393208786845 8211009.2139681084081530571,-360551.4742159362649545074 8212034.3190615437924861908,-358325.0844000709475949407 8212575.6871830569580197334,-356500.6692654576618224382 8212575.6871830569580197334,-355758.6135398300248198211 8212274.3916364107280969620,-342523.9505989083554595709 8206730.1735943248495459557,-342091.0290992152295075357 8206187.4960609087720513344,-333772.9027886672411113977 8195470.6866239896044135094,-333432.8217442945460788906 8194809.9391960911452770233,-333494.6040616855025291443 8192464.1212498266249895096,-328083.2522947335965000093 8180025.7274808306246995926,-317507.9006693732226267457 8175105.2581908330321311951,-310921.5716770977014675736 8169886.7297789873555302620,-306778.0375907898996956646 8162635.0137194916605949402,-314415.7791736071230843663 8154254.7761839749291539192)),((-298985.5619157699984498322 8220299.1448121173307299614,-299418.5947349572088569403 8220841.3329649502411484718,-299634.9998250596108846366 8221565.9059020085260272026,-299851.4049151591607369483 8223196.1274644229561090469,-299727.7289608888677321374 8225974.6323513938114047050,-299511.3238707864657044411 8226880.4422335214912891388,-298831.0504625497851520777 8228271.2331709414720535278,-288626.7267000032588839531 8243452.6631679208949208260,-284483.1926136954571120441 8246179.0146229611709713936,-282133.1268435569945722818 8246601.6288391575217247009,-268774.6766288737417198718 8248480.1329240286722779274,-267970.7272663667681626976 8248541.8635866008698940277,-267414.2411318888189271092 8248298.8702376186847686768,-267166.7779038569424301386 8247752.9276838572695851326,-266857.6436779224895872176 8246238.7635718090459704399,-266084.5298143653781153262 8241818.2048465302214026451,-266053.5829959242837503552 8241091.4344580508768558502,-277804.0231660994468256831 8233529.1847231518477201462,-286183.9317940355394966900 8232924.6855317205190658569,-286895.0407012220821343362 8232924.6855317205190658569,-291718.9595152577967382967 8231594.6123727718368172646,-292213.6633323419373482466 8231049.8974807532504200935,-298490.8580986858578398824 8222834.4047459559515118599,-298985.5619157699984498322 8220299.1448121173307299614)),((-320321.8347576460218988359 8229962.5458649722859263420,-319827.1309405590291135013 8235040.7570098340511322021,-319827.1309405590291135013 8235947.8917335318401455879,-320260.0524402550654485822 8236492.7484609037637710571,-326908.2750694099231623113 8241939.3764892881736159325,-329351.0699753776425495744 8242847.3550463952124118805,-330124.1838389376061968505 8242908.8208849569782614708,-330835.4040656153811141849 8242725.9506133524700999260,-331330.1078826995217241347 8242182.1615991787984967232,-331082.7559741588775068521 8241455.5734806824475526810,-331391.8902000904781743884 8239578.8471431769430637360,-332628.8723817847203463316 8237702.3775628153234720230,-333092.6293804305605590343 8237158.9565887460485100746,-333865.7432439905242063105 8237037.6451956499367952347,-334329.6115621248027309775 8237582.7997531481087207794,-337112.5988319564494304359 8242242.0963095873594284058,-342152.8114166033337824047 8250783.7109698904678225517,-342369.3278261969680897892 8251511.2134703490883111954,-342462.0569620290189050138 8252358.8198936972767114639,-341936.4063265009317547083 8252358.8198936972767114639,-339400.7709652127814479172 8251874.3368008276447653770,-322517.3890745614771731198 8245452.0349454907700419426,-321713.3283925604191608727 8244482.2581625692546367645,-317229.6019423886900767684 8238549.9323515286669135094,-317600.7411246937117539346 8234497.5308964215219020844,-320321.8347576460218988359 8229962.5458649722859263420)),((-131418.8927729424904100597 8450002.4812638033181428909,-128852.3105932132166344672 8443217.6614618655294179916,-125358.1030967037950176746 8433582.1149839032441377640,-125296.3207793128385674208 8432711.8850616905838251114,-125636.4018236883857753128 8431904.8170470520853996277,-127646.3865494513302110136 8429172.3116981964558362961,-128357.6067761290905764326 8429419.5413138214498758316,-129470.8016840617492562160 8430413.9085405077785253525,-130305.6978650098317302763 8430350.8623820859938859940,-132160.9484985729504842311 8428116.6498009692877531052,-134943.9357684046262875199 8422655.4660612400621175766,-135871.6724046760064084083 8404748.7486409302800893784,-140293.5052179655758664012 8369126.5888590998947620392,-140540.8571265090431552380 8368263.9417213918641209602,-141221.1305347457237076014 8366662.4036922659724950790,-145488.4518948150798678398 8368079.4977552164345979691,-152724.2187963774194940925 8375657.2470140000805258751,-153404.4922046140709426254 8378000.7946866620332002640,-153373.5453861758287530392 8378925.4125343877822160721,-148240.4923462056613061577 8399056.8686607982963323593,-148059.1528957030095625669 8399737.8898180592805147171,-147652.9480737982376012951 8400726.4732431881129741669,-147220.0265741050534415990 8401407.8715051133185625076,-146539.7531658655498176813 8401716.9709801133722066879,-145983.1557118992204777896 8401220.8147236928343772888,-145612.1278490854019764811 8401716.9709801133722066879,-145364.6646210535254795104 8402581.9839985016733407974,-141190.1837163046002388000 8420607.5621528327465057373,-140602.7507633884088136256 8423585.7077235877513885498,-140386.2343537948036100715 8425571.2122797425836324692,-140293.5052179655758664012 8427432.9978936556726694107,-140448.1279906769923400134 8428923.3022035919129848480,-143045.5456693561282008886 8448880.8431634176522493362,-143756.7658960339031182230 8453677.0226260423660278320,-156218.3149733956088311970 8456916.9774842560291290283,-156991.4288369555724784732 8456730.2986293677240610123,-157424.3503366515797097236 8456043.9804964791983366013,-156991.4288369555724784732 8455484.2490402068942785263,-154455.7934756673930678517 8452742.9660407453775405884,-151796.4821601060393732041 8451061.2907705008983612061,-150621.5049347824242431670 8446391.3914110194891691208,-151208.9378876986156683415 8442720.4902771990746259689,-161258.6388775336963590235 8433457.5345053281635046005,-161815.2363315000548027456 8433767.9831401184201240540,-165927.9349188579944893718 8437185.1261108089238405228,-169236.5728242177283391356 8440233.3476771246641874313,-170009.5753682836075313389 8441476.8137898929417133331,-170071.4690051658253651112 8442346.2903630267828702927,-169947.7930508954741526395 8443156.2970731183886528015,-169174.6791873355396091938 8444649.7919906452298164368,-176874.3144070349808316678 8447947.1719167567789554596,-183429.8079003578168340027 8449939.2668903525918722153,-184233.7572628676134627312 8450250.6365618128329515457,-185316.0053523592068813741 8451309.2576242219656705856,-186398.3647613420034758747 8453240.8157563824206590652,-186800.3394425969163421541 8454798.0473017022013664246,-188470.1318044959043618292 8462278.6146730221807956696,-188439.1849860548099968582 8463214.1092274021357297897,-188284.5622133433935232460 8464213.0597421079874038696,-185563.4685803939064498991 8466771.9181299507617950439,-184264.7040813087369315326 8467521.0319710131734609604,-181265.3117213746591005474 8468456.9699385277926921844,-177740.1574064241431187838 8468643.9471145477145910263,-176967.0435428642085753381 8468270.2222475465387105942,-164165.3021016385464463383 8470204.4483535364270210266,-160949.3933321108634117991 8471015.7343615088611841202,-158846.6794705158390570432 8471641.6255598831921815872,-149817.5555722754215821624 8475826.3957046549767255783,-146972.6746655615570489317 8479513.8849765919148921967,-160949.3933321108634117991 8502995.4172549173235893250,-167226.5880984519608318806 8505066.4277327470481395721,-172947.1854108295519836247 8506384.6276615001261234283,-173472.8360463547869585454 8505819.4621988646686077118,-174215.0030914736271370202 8505819.4621988646686077118,-179193.4333587295259349048 8506069.2861623000353574753,-179409.9497683231602422893 8506634.6967900414019823074,-179224.3801771706202998757 8507701.4831505790352821350,-177338.1827251692302525043 8513666.3284741286188364029,-172205.1296852019149810076 8520012.6112211719155311584,-157640.7554267539526335895 8535683.0980203952640295029,-157022.3756553966668434441 8536124.2363177780061960220,-144931.7431213575182482600 8542869.1372172459959983826,-144251.4697131208667997271 8542427.5922357551753520966,-144066.0114414595882408321 8541606.7752905581146478653,-143323.8443963407189585268 8537572.8084496092051267624,-144282.4165315619611646980 8509897.8982137348502874374,-143231.1152605086681433022 8505379.9191853515803813934,-129563.5308198938146233559 8501615.7458084020763635635,-115926.8931977172323968261 8499923.0623582229018211365,-115463.0248795801308006048 8499296.1079841032624244690,-115524.9185164623195305467 8498356.2282211538404226303,-118678.9336491077992832288 8479889.9833878260105848312,-122575.1158268721337663010 8461406.7763581350445747375,-129594.4776383320713648573 8457478.3888761196285486221,-129779.9359099933644756675 8458288.2701066788285970688,-132130.0016801318561192602 8460034.9014685377478599548,-133026.7914979621418751776 8460283.1714041661471128464,-133459.7129976581491064280 8459598.0924596134573221207,-133645.2825888106599450111 8458599.9938295967876911163,-133366.9838618260982912034 8456916.9774842560291290283,-131975.4902269116719253361 8451247.8256347682327032089,-131418.8927729424904100597 8450002.4812638033181428909)),((-122977.0905081270466325805 8508642.7878556754440069199,-125512.7258694152260432020 8508830.7921410575509071350,-126904.2195043324754806235 8509522.0657078959047794342,-127213.4650497553229797632 8510211.8223970048129558563,-130274.7510465715749887750 8518065.2058855388313531876,-131635.2978630448924377561 8533415.0009105317294597626,-131758.9738173180376179516 8535179.6204684469848871231,-131511.6219087745703291148 8539904.5999362692236900330,-131295.2168186721683014184 8541923.6505688149482011795,-131047.7535906402918044478 8542869.1372172459959983826,-123564.6347805344703374431 8563139.6324080415070056915,-122296.8170998903806321323 8564088.0930764134973287582,-121616.5436916508915601298 8564404.1221806351095438004,-120750.7006922617147210985 8564468.5591860376298427582,-117658.5791964956006268039 8564277.2990620024502277374,-110948.4629304584959754720 8562886.0322308056056499481,-110329.9718396128009771928 8561811.1747258920222520828,-109371.3997043915587710217 8547157.2547364681959152222,-109402.3465228326531359926 8546210.9867958705872297287,-113298.5287005969876190647 8516304.7046306654810905457,-113453.1514733084186445922 8515300.2185145672410726547,-113917.0197914426826173440 8512598.6735706087201833725,-114164.3716999861644580960 8511656.8606676571071147919,-114597.1818801909394096583 8510965.3857190217822790146,-122977.0905081270466325805 8508642.7878556754440069199)),((-92951.7748123833152931184 8551198.1008841041475534439,-104145.6175285921781323850 8551449.4778003897517919540,-106712.0883888302196282893 8553280.9716154523193836212,-107485.2022523901687236503 8554543.7577994950115680695,-106835.7643431033793604001 8559852.6179195307195186615,-104516.6453914059820817783 8578775.7893092986196279526,-98301.3442619442648719996 8589683.3030100073665380478,-97775.5823069277830654755 8590192.3755585104227066040,-89735.8660428556468104944 8589303.3496326003223657608,-86025.2534562438231660053 8587145.7811966780573129654,-85468.6560022774938261136 8586638.7480191215872764587,-84386.4079127859295113012 8584100.4570038504898548126,-84293.5674574654694879428 8583213.9983862955123186111,-85004.7876841432298533618 8578521.6460732165724039078,-91251.0356320432183565572 8554734.7635270673781633377,-92951.7748123833152931184 8551198.1008841041475534439)))'
      ,0) as p1
    )
              select 10 as w,ST_NumGeometries(p1),ST_Area(ST_Buffer(p1, 10.0 * 1000, 25)),ST_NumGeometries(ST_Buffer(p1, 10.0 * 1000, 25))from viewy
    union all select 20 as w,ST_NumGeometries(p1),ST_Area(ST_Buffer(p1, 20.0 * 1000, 25)),ST_NumGeometries(ST_Buffer(p1, 20.0 * 1000, 25))from viewy
    union all select 50 as w,ST_NumGeometries(p1),ST_Area(ST_Buffer(p1, 50.0 * 1000, 25)),ST_NumGeometries(ST_Buffer(p1, 50.0 * 1000, 25))from viewy
    union all select 100 as w,ST_NumGeometries(p1),ST_Area(ST_Buffer(p1, 100.0 * 1000, 25)),ST_NumGeometries(ST_Buffer(p1, 100.0 * 1000, 25))from viewy
    union all select -10 as w,ST_NumGeometries(p1),ST_Area(ST_Buffer(p1, -10.0 * 1000, 25)),ST_NumGeometries(ST_Buffer(p1, -10.0 * 1000, 25))from viewy
    union all select -20 as w,ST_NumGeometries(p1),ST_Area(ST_Buffer(p1, -20.0 * 1000, 25)),ST_NumGeometries(ST_Buffer(p1, -20.0 * 1000, 25))from viewy
    union all select -50 as w,ST_NumGeometries(p1),ST_Area(ST_Buffer(p1, -50.0 * 1000, 25)),ST_NumGeometries(ST_Buffer(p1, -50.0 * 1000, 25))from viewy
    union all select -100 as w,ST_NumGeometries(p1),ST_Area(ST_Buffer(p1, -100.0 * 1000, 25)),ST_NumGeometries(ST_Buffer(p1, -100.0 * 1000, 25))from viewy

Checked are 10,20,50,100 kilometer, inflate/deflate

But for many, in the unit tests below, distance of 1,2,5 k are kept too
because they could cause self-intersections in the past

Values are not identical. We might check area with less precision.

gr:
    336277774579
    442312549699
    680433051228
    910463608938

    139313936462
     96993175731
     31392586710
      2033070670

it:
     655017269701
     749348852219
    1018311961402
    1436442592714

    477946399339
    404696093915
    238752534654
     69771921799

nl:
    123407901256
    145045853638
    201203536326
    303295187184

    64669284652
    46684337476
    10245330928
    0

no:
    2102032030338
    2292165016326
    2725461758029
    3374949015149

    1361202945650
    1089854028969
     649632754053
     306749522531

uk:
    857680535981
    970483182932
   1247820319617
   1659854829029

    572557777232
    479260087245
    274834862993
     78209736228

*/

template <typename MP, typename P>
void test_one(std::string const& caseid, std::string const& wkt, double expected_area, double distance,
              ut_settings settings = ut_settings())
{
    bg::strategy::buffer::join_round join_round(100);
    bg::strategy::buffer::end_flat end_flat;

    // Test with a high tolerance, even a difference of 1000 is only ~1.0e-6%

    settings.tolerance = 10000.0;

#if ! defined(BOOST_GEOMETRY_USE_RESCALING)
    // in case robustness policies are changed, areas should be adapted
    settings.tolerance = boost::starts_with(caseid, "no") ? 200000.0 : 100000.0;
#endif

    test_one<MP, P>(caseid, wkt, join_round, end_flat,
        expected_area, distance * 1000.0, settings);
}


template <bool Clockwise, typename P>
void test_all()
{
    typedef bg::model::polygon<P, Clockwise> pt;
    typedef bg::model::multi_polygon<pt> mpt;

    std::string base_folder = "data/";
    std::string gr = read_from_file<mpt>(base_folder + "gr.wkt");
    std::string it = read_from_file<mpt>(base_folder + "it.wkt");
    std::string nl = read_from_file<mpt>(base_folder + "nl.wkt");
    std::string no = read_from_file<mpt>(base_folder + "no.wkt");
    std::string uk = read_from_file<mpt>(base_folder + "uk.wkt");

    test_one<mpt, pt>("gr10", gr,    336279815682, 10);
    test_one<mpt, pt>("gr20", gr,    442317491749, 20);
    test_one<mpt, pt>("gr50", gr,    680442278645, 50);
    test_one<mpt, pt>("gr100", gr,   910474621215, 100);

    test_one<mpt, pt>("gr10", gr,    139313156846, -10);
    test_one<mpt, pt>("gr20", gr,     96991350242, -20);
    test_one<mpt, pt>("gr50", gr,     31391928002, -50);
    test_one<mpt, pt>("gr100", gr,     2035400805, -100);

    test_one<mpt, pt>("it1", it,     569862998347, 1);
    test_one<mpt, pt>("it2", it,     579239208963, 2);
    test_one<mpt, pt>("it5", it,     607625463736, 5);
    test_one<mpt, pt>("it10", it,    655018578530, 10);
    test_one<mpt, pt>("it20", it,    749353305743, 20);
    test_one<mpt, pt>("it50", it,   1018323115670, 50);
    test_one<mpt, pt>("it100", it,  1436451405439, 100);

    test_one<mpt, pt>("it1", it,     551474421881, -1);
    test_one<mpt, pt>("it2", it,     542617730624, -2);
    test_one<mpt, pt>("it5", it,     517402445790, -5);
    test_one<mpt, pt>("it10", it,    477945510429, -10);
    test_one<mpt, pt>("it20", it,    404693983797, -20);
    test_one<mpt, pt>("it50", it,    238748449624, -50);
    test_one<mpt, pt>("it100", it,    69768648896, -100);
    test_one<mpt, pt>("it200", it,              0, -200);

    test_one<mpt, pt>("nl1", nl,      97391170774, 1);
    test_one<mpt, pt>("nl2", nl,     100816707832, 2);
    test_one<mpt, pt>("nl5", nl,     110239801028, 5);
    test_one<mpt, pt>("nl10", nl,    123408274536, 10);
    test_one<mpt, pt>("nl20", nl,    145046915403, 20);
    test_one<mpt, pt>("nl50", nl,    201207309002, 50);
    test_one<mpt, pt>("nl100", nl,   303300936340, 100);

    test_one<mpt, pt>("nl1", nl,      90095050333, -1);
    test_one<mpt, pt>("nl2", nl,      86601861798, -2);
    test_one<mpt, pt>("nl5", nl,      77307843754, -5);
    test_one<mpt, pt>("nl10", nl,     64668870425, -10);
    test_one<mpt, pt>("nl20", nl,     46683531062, -20);
    test_one<mpt, pt>("nl50", nl,     10244523910, -50);
    test_one<mpt, pt>("nl100", nl,              0, -100);

    test_one<mpt, pt>("no1", no,    1819566570720, 1);
    test_one<mpt, pt>("no2", no,    1865041238129, 2, ut_settings::ignore_validity());
    test_one<mpt, pt>("no5", no,    1973615533600, 5);
    test_one<mpt, pt>("no10", no,   2102034240506, 10);
    test_one<mpt, pt>("no20", no,   2292171257647, 20);
    test_one<mpt, pt>("no50", no,   2725475403816, 50);
    test_one<mpt, pt>("no100", no,  3374987120112, 100);

    test_one<mpt, pt>("no1", no,    1725145487969, -1);
    test_one<mpt, pt>("no2", no,    1678942603503, -2);
    test_one<mpt, pt>("no5", no,    1547329249723, -5);
    test_one<mpt, pt>("no10", no,   1361198873951, -10);
    test_one<mpt, pt>("no20", no,   1089847815351, -20);
    test_one<mpt, pt>("no50", no,    649622162382, -50);
    test_one<mpt, pt>("no100", no,   306739133606, -100);

    test_one<mpt, pt>("uk1", uk,     733080790315, 1);
    test_one<mpt, pt>("uk2", uk,     749555939251, 2);
    test_one<mpt, pt>("uk5", uk,     793752660191, 5);
    test_one<mpt, pt>("uk10", uk,    857682286960, 10);
    test_one<mpt, pt>("uk20", uk,    970488082763, 20);
    test_one<mpt, pt>("uk50", uk,   1247830325401, 50);
    test_one<mpt, pt>("uk100", uk,  1659861958875, 100);

    test_one<mpt, pt>("uk1", uk,     699378146599, -1);
    test_one<mpt, pt>("uk2", uk,     683086442146, -2);
    test_one<mpt, pt>("uk5", uk,     637325279340, -5);
    test_one<mpt, pt>("uk10", uk,    572556625332, -10);
    test_one<mpt, pt>("uk20", uk,    479258129205, -20);
    test_one<mpt, pt>("uk50", uk,    274828071591, -50);
    test_one<mpt, pt>("uk100", uk,    78205461294, -100);
}

int test_main(int, char* [])
{
    test_all<true, bg::model::point<double, 2, bg::cs::cartesian> >();
    test_all<false, bg::model::point<double, 2, bg::cs::cartesian> >();
    
    return 0;
}

