/*****************************************************************************
 * camcapt.c
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 *
 * Copyright (C) 2006 Stefan Sikora hoshy ['at'] schrauberstube.de
 * Copyright (C) 2006 Zoltan Csala zcsala ['at'] users.sourceforge.net
 *
 * Released under the terms of the GPL.
 * *NO WARRANTY*
 *****************************************************************************/
 
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <jasper/jasper.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <locale.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <errno.h>
#include <fcntl.h>
#include <netdb.h>

#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>

#include "camcapt.h"

#include "theme_support.h"
#include "gyachi_lib.h"
#include "gy_config.h"


#define dir_exists(x) (access((x), R_OK) == 0)

/***** Variables for frame recording */
/* from the capturc file -- currently not referenced outside of capcapt.[ch] */
static char *frame_dir               = NULL; /* Directory where frames are stored */
static char *custom_encoder          = NULL; /* Path to user-selected encoder */
static char *encoder_options         = NULL; /* Options to encode frames into movie */
static int del_frames_after_encoding = 0;    /* Whether captured frames should be deleted after encoding */
       int capture_auto_on           = 0;
       int max_fps                   = 0;


static char *JpgFilePath = NULL;

static int IsRecording = 0; /* Whether we are recording or not */
static int app_debug = 0;

static GtkWidget *saved_butRecord=NULL;
char *local_address=NULL;

/***** End of variables for frame recording */

static int MovieNumber = 0;
static int FrameNumber = 1; /* Number for every frame recorded */

int jasper_started=0;

/* HACK -- packet_size is defined in the caller... */
extern unsigned long packet_size;

int check_fps(int max_fps)
{
	if (max_fps <= 0)   max_fps=99999;
	if (max_fps > 9999) max_fps=8;

	return(max_fps);
}

/* Gets properties from corresponding widgets and put them into variables */
void getCaptureProperties(GtkWidget *w)
{
	GtkTextIter StartIter, EndIter;
	GtkWidget *fileselDirCaptures;
	GtkWidget *fileselCustEnc;
	GtkWidget *txtCmdOptions;
	GtkWidget *txtFps;
	GtkTextBuffer *bufCmd;
	char      *t_string;

	fileselDirCaptures = g_object_get_data(G_OBJECT(w), "fileselDirCaptures");
	fileselCustEnc     = g_object_get_data(G_OBJECT(w), "fileselCustEnc");
	txtCmdOptions      = g_object_get_data(G_OBJECT(w), "txtCmdOptions");
	txtFps             = g_object_get_data(G_OBJECT(w), "txtFps");

	if (frame_dir) {
		free(frame_dir);
		frame_dir=NULL;
	}
	t_string=gtk_file_chooser_get_current_folder(GTK_FILE_CHOOSER(fileselDirCaptures));
	if (t_string) {
		frame_dir=strdup(t_string);
	}

	if (custom_encoder) {
		free(custom_encoder);
		custom_encoder=NULL;
	}
	t_string=gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(fileselCustEnc));
	if (t_string) {
		custom_encoder = strdup(t_string);
	}

	if (encoder_options) {
		free(encoder_options);
		encoder_options=NULL;
	}
	bufCmd = gtk_text_view_get_buffer( GTK_TEXT_VIEW(txtCmdOptions));
	gtk_text_buffer_get_bounds( bufCmd, &StartIter, &EndIter);
	encoder_options = strdup(gtk_text_buffer_get_text(bufCmd, &StartIter, &EndIter, FALSE));

	if (txtFps) {
		max_fps = check_fps(atoi(gtk_entry_get_text(GTK_ENTRY(txtFps))));
	}

	if (JpgFilePath) {
		free(JpgFilePath);
		JpgFilePath=NULL;
	}
} /* getCaptureProperties */


/* read capture configuration file - taken from Hoshy, by Z.Cs. */
void read_capturc () {
	FILE *f;
	int   max_len;
	char *line;
	char  *(pieces[3]);
	char *capturc_path;

	pieces[0]=GYACH_CFG_DIR;
	pieces[1]="/capturc";
	pieces[2]=NULL;
	capturc_path = gyachi_filename(pieces);

	/* Free old values. Set defaults */
	if (frame_dir) {
		free(frame_dir);
		frame_dir=NULL;
	}
	if (custom_encoder) {
		free(custom_encoder);
		custom_encoder=NULL;
	}
	if (JpgFilePath) {
		free(JpgFilePath);
		JpgFilePath=NULL;
	}
	if (encoder_options) {
		free(encoder_options);
		encoder_options=NULL;
	}

	f = fopen(capturc_path, "r");
	if (f == NULL) {
		/* no config available */
		free(capturc_path);
		max_fps=8;
		return;
	}

	/* process config file */
	max_len = max_linelen(f);
	line=malloc(max_len);
	while (fgets(line, max_len, f)) {
		char *p, *key, *value;
				
		/* strip comments */
		if ( (p = strchr(line, '#'))) {
			*p='\n';
		}

		/* Skip empty lines */
		if (*line == '\n') {
			continue;
		}

		if ( (key = strtok(line, "=")) && (value = strtok(NULL, "\n"))) {
			if (strcmp(key,"frame_directory")==0) {
				if (frame_dir) free(frame_dir);
				frame_dir = strdup(value);
			}
			else if (strcmp(key, "custom_encoder") == 0) {
				if (custom_encoder) free(custom_encoder);
			        custom_encoder = strdup(value);
			}
			else if (strcmp(key, "del_frames_after_encoding") == 0) {
				del_frames_after_encoding = atoi(value);
			}
			else if (strcmp(key, "encoder_options") == 0) {
				if (encoder_options) free(encoder_options);
				encoder_options = strdup(value);
			}
			else if (strcmp(key, "capture_auto_on") == 0) {
				capture_auto_on = atoi(value);
			}
			else if (strcmp(key, "max_fps") == 0) {
				max_fps = atoi(value);
			}

		} else {
			fprintf(stderr, "%s", line);
			fprintf(stderr, "Problems in configuration file %s\n", capturc_path);
		}		
	}

	max_fps = check_fps(max_fps);

	fclose(f);
	free(capturc_path);
	free(line);
} /* read_capturc */


/* read capture configuration file - again, taken from Hoshy, by Z.Cs. */
void write_capturc() {
	FILE *f;
	char  *(pieces[3]);
	char *capturc_path;

	pieces[0]=GYACH_CFG_DIR;
	pieces[1]="/capturc";
	pieces[2]=NULL;
	capturc_path = gyachi_filename(pieces);
	
	if ((f = fopen(capturc_path, "w")) == NULL) {
		fprintf(stderr, "Problems when writing configuration to %s\n",
				capturc_path);
		free(capturc_path);
		return;
	}
	fprintf(f, "# Configuration file for webcam capture settings\n\n");
	if (frame_dir) {
		fprintf(f, "# Directory where captured frames are stored\n");
		fprintf(f, "frame_directory=%s\n", frame_dir);
	}

	if (custom_encoder) {
		fprintf(f, "# Program that encodes captured frames into movie\n");
		fprintf(f, "custom_encoder=%s\n", custom_encoder);
	}

	if (encoder_options) {
		fprintf(f, "# Command-line options for encoder program\n");
		fprintf(f, "encoder_options=%s\n", encoder_options);
	}

	fprintf(f, "# 1 for automatic deleting frames after encoding, 0 if not\n");
	fprintf(f, "del_frames_after_encoding=%d\n", del_frames_after_encoding);
	
	fprintf(f, "# 1 for automatic start recording when cam starts, 0 for not\n");
	fprintf(f, "capture_auto_on=%d\n", capture_auto_on);

	fprintf(f, "# max capture rate. Frames per second.\n");
	fprintf(f, "max_fps=%d\n", max_fps);
	
	free(capturc_path);
	fclose(f);
} /* write_capturc */


/******* Callbacks ******/


void on_buttCaptCancel_clicked (GtkButton *button, gpointer user_data)
{
	GtkWidget *w=user_data;

	gtk_widget_destroy(w);
}

gboolean on_buttCapt_delete_event(GtkWidget *widget, GdkEvent *event, gpointer user_data)
{
	GtkWidget *w=user_data;

	if (!GTK_IS_WINDOW(widget)) {
		/* window has been deleted / destroyed */
		return(TRUE);
	}

	gtk_widget_destroy(w);
	return(TRUE);
}

void on_buttCaptApply_clicked (GtkButton *button, gpointer user_data)
{
	GtkWidget *w=user_data;

	getCaptureProperties(w);
	write_capturc();


}


void on_buttCaptOK_clicked(GtkButton* button, gpointer user_data)
{
	GtkWidget *w=user_data;

	getCaptureProperties(w);
	write_capturc();
	gtk_widget_destroy(w);
}


void on_buttCaptHelp_clicked(GtkButton *button, gpointer user_data)
{
}


void on_cboxCaptured_changed(GtkComboBox* combobox, gpointer user_data)
{
}


void on_cbutDelAfterEnc_toggled(GtkToggleButton *togglebutton, gpointer user_data)
{
}


void on_buttHelpEnc_clicked(GtkButton *button, gpointer user_data)
{

}


void on_buttDelFrames_clicked(GtkButton *button, gpointer user_data)
{

}


void on_buttEncode_clicked(GtkButton *button, gpointer user_data)
{
    
}


void on_buttCloseEnc_clicked(GtkButton *button, gpointer user_data)
{
	GtkWidget *w=user_data;

	gtk_widget_destroy(w);
}

gboolean on_buttCloseEnc_delete_event(GtkWidget *widget, GdkEvent *event, gpointer user_data)
{
	GtkWidget *w=user_data;

	if (!GTK_IS_WINDOW(widget)) {
		/* window has been deleted / destroyed */
		return(TRUE);
	}

	gtk_widget_destroy(w);
	return(TRUE);
}


/******* Callbacks - record/pause ******/

void on_butRecord_toggled(GtkButton *button, gpointer user_data)
{
	GtkWidget *butPause = (GtkWidget *)user_data;

	IsRecording = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button));
	if ( IsRecording ) {
		gtk_button_set_label(button, "gtk-media-stop" );
		gyachi_set_tooltip(GTK_WIDGET(button), _("Stop Recording"));
		MovieNumber++;
		FrameNumber = 1;
	}
	else {
		gtk_button_set_label(button, "gtk-media-record" );
		gyachi_set_tooltip(GTK_WIDGET(button), _("Start Recording"));
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(butPause), FALSE);
	}
	gtk_widget_set_sensitive(butPause, (IsRecording == 1) );
} /* on_butRecord_toggled */


void on_butPause_toggled(GtkButton *button, gpointer user_data)
{
	IsRecording = !gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button));
} /* on_butPause_toggled */


/******* GUI Creation  *******/

void on_capt_setup_menu(GtkMenuItem *menuitem, gpointer  user_data) {
	GtkWidget *parent = user_data;
	char      *uploader;
	GtkWidget *winSetupCapture;
	GtkWidget *vboxWinCapture;
	GtkWidget *tblWinCapture;
	GtkWidget *lblDirCaptures;
	GtkWidget *fileselDirCaptures;
	GtkWidget *lblEncoder;
	GtkWidget *fileselCustEnc;
	GtkWidget *lblCmdOptions;
	GtkWidget *scrwinCmdOptions;
	GtkWidget *txtCmdOptions;
	GtkWidget *lblFps;
	GtkWidget *txtFps;
	GtkWidget *hseparator1;
	GtkWidget *hbuttonbox1;
	GtkWidget *buttCaptHelp;
	GtkWidget *buttCaptCancel;
	GtkWidget *buttCaptApply;
	GtkWidget *buttCaptOK;
	GtkTextBuffer *bufCmd;

	winSetupCapture = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_container_set_border_width(GTK_CONTAINER(winSetupCapture), 5);
	gtk_window_set_title(GTK_WINDOW(winSetupCapture), _("Webcam - Frame Capture Setup"));
	gtk_window_set_position(GTK_WINDOW(winSetupCapture), GTK_WIN_POS_CENTER_ON_PARENT);
	gtk_window_set_transient_for(GTK_WINDOW(winSetupCapture), GTK_WINDOW(parent));
	gtk_window_set_destroy_with_parent(GTK_WINDOW(winSetupCapture), TRUE);
	if (parent) gtk_window_present(GTK_WINDOW(parent));

	vboxWinCapture = gtk_vbox_new(FALSE, 5);
	gtk_container_add(GTK_CONTAINER(winSetupCapture), vboxWinCapture);

	uploader = g_object_get_data(G_OBJECT(parent), "uploader");
	tblWinCapture = gtk_table_new(uploader?4:3, 2, FALSE);
	gtk_box_pack_start(GTK_BOX(vboxWinCapture), tblWinCapture, TRUE, TRUE, 0);
	gtk_table_set_row_spacings(GTK_TABLE(tblWinCapture), 5);
	gtk_table_set_col_spacings(GTK_TABLE(tblWinCapture), 5);

	/* row 1, capture directory */
	lblDirCaptures = gtk_label_new(_("Directory:"));
	gtk_table_attach(GTK_TABLE(tblWinCapture), lblDirCaptures, 0, 1, 0, 1,
			 (GtkAttachOptions)(GTK_FILL),
			 (GtkAttachOptions)(GTK_FILL), 0, 0);
	gtk_misc_set_alignment(GTK_MISC(lblDirCaptures), 0, 0.5);

	fileselDirCaptures = gtk_file_chooser_button_new(_("Select A File"), GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER);
	gtk_table_attach(GTK_TABLE(tblWinCapture), fileselDirCaptures, 1, 2, 0, 1,
			 (GtkAttachOptions)(GTK_EXPAND | GTK_FILL),
			 (GtkAttachOptions)(GTK_FILL), 0, 0);
	g_object_set(fileselDirCaptures, "show-hidden", TRUE, NULL);

	/* row 2, encoder */
	lblEncoder = gtk_label_new(_("Encoder:"));
	gtk_table_attach(GTK_TABLE(tblWinCapture), lblEncoder, 0, 1, 1, 2,
			 (GtkAttachOptions)(GTK_FILL),
			 (GtkAttachOptions)(GTK_FILL), 0, 0);
	gtk_misc_set_alignment(GTK_MISC(lblEncoder), 0, 0.5);

	fileselCustEnc = gtk_file_chooser_button_new(_("Select A File"), GTK_FILE_CHOOSER_ACTION_OPEN);
	gtk_table_attach(GTK_TABLE(tblWinCapture), fileselCustEnc, 1, 2, 1, 2,
			 (GtkAttachOptions)(GTK_FILL),
			 (GtkAttachOptions)(GTK_FILL), 0, 0);

	/* row 3, encoder options */
	lblCmdOptions = gtk_label_new(_("Command\nOptions:"));
	gtk_table_attach(GTK_TABLE(tblWinCapture), lblCmdOptions, 0, 1, 2, 3,
			 (GtkAttachOptions)(GTK_FILL),
			 (GtkAttachOptions)(0), 0, 0);
	gtk_misc_set_alignment(GTK_MISC(lblCmdOptions), 0, 0.5);

	scrwinCmdOptions = gtk_scrolled_window_new(NULL, NULL);
	gtk_table_attach(GTK_TABLE(tblWinCapture), scrwinCmdOptions, 1, 2, 2, 3,
			 (GtkAttachOptions)(GTK_FILL),
			 (GtkAttachOptions)(GTK_FILL), 0, 0);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrwinCmdOptions), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrwinCmdOptions), GTK_SHADOW_IN);

	txtCmdOptions = gtk_text_view_new();
	gtk_container_add(GTK_CONTAINER(scrwinCmdOptions), txtCmdOptions);
	gyachi_set_tooltip(txtCmdOptions, _("Command options required for encoder"));

	/* row 4, frames per second (capture/uploader only) */
	if (uploader) {
		char fps_text[6];

		lblFps = gtk_label_new(_("FPS"));
		gtk_table_attach(GTK_TABLE(tblWinCapture), lblFps, 0, 1, 3, 4,
				 (GtkAttachOptions)(GTK_FILL),
				 (GtkAttachOptions)(GTK_FILL), 0, 0);
		gtk_misc_set_alignment(GTK_MISC(lblCmdOptions), 0, 0.5);

		txtFps = gtk_entry_new();
		gtk_table_attach(GTK_TABLE(tblWinCapture), txtFps, 1, 2, 3, 4,
				 (GtkAttachOptions)(GTK_FILL),
				 (GtkAttachOptions)(GTK_FILL), 0, 0);
		gtk_entry_set_max_length(GTK_ENTRY(txtFps), 4);
		sprintf(fps_text, "%d", max_fps);
		gtk_entry_set_text(GTK_ENTRY(txtFps), fps_text);
		gtk_editable_set_editable(GTK_EDITABLE(txtFps), TRUE);
		gyachi_set_tooltip(txtFps, _("Maximum capture rate, in Frames Per Second"));

		g_object_set_data(G_OBJECT(winSetupCapture), "txtFps", txtFps);
	}

	hseparator1 = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(vboxWinCapture), hseparator1, TRUE, TRUE, 0);

	hbuttonbox1 = gtk_hbutton_box_new();
	gtk_box_pack_start(GTK_BOX(vboxWinCapture), hbuttonbox1, TRUE, TRUE, 0);
#ifdef HAVE_GTK_WIDGET_TOOLTIP // GTK 2+ >= 2.12
	gtk_button_box_set_layout(GTK_BUTTON_BOX(hbuttonbox1), GTK_BUTTONBOX_CENTER);
#else
	gtk_button_box_set_layout(GTK_BUTTON_BOX(hbuttonbox1), GTK_BUTTONBOX_SPREAD);
#endif
	gtk_box_set_spacing(GTK_BOX(hbuttonbox1), 5);
  
	buttCaptHelp = gtk_button_new_from_stock("gtk-help");
	gtk_box_pack_start(GTK_BOX(hbuttonbox1), buttCaptHelp,   TRUE, TRUE, 0);

	buttCaptCancel = gtk_button_new_from_stock("gtk-cancel");
	gtk_box_pack_start(GTK_BOX(hbuttonbox1), buttCaptCancel, TRUE, TRUE, 0);
	GTK_WIDGET_SET_FLAGS(buttCaptCancel, GTK_CAN_DEFAULT);

	buttCaptApply = gtk_button_new_from_stock("gtk-apply");
	gtk_box_pack_start(GTK_BOX(hbuttonbox1), buttCaptApply,  TRUE, TRUE, 0);
	GTK_WIDGET_SET_FLAGS(buttCaptApply, GTK_CAN_DEFAULT);

	buttCaptOK = gtk_button_new_from_stock("gtk-ok");
	gtk_box_pack_start(GTK_BOX(hbuttonbox1), buttCaptOK,     TRUE, TRUE, 0);
	GTK_WIDGET_SET_FLAGS(buttCaptOK, GTK_CAN_DEFAULT);

	g_object_set_data(G_OBJECT(winSetupCapture), "fileselDirCaptures", fileselDirCaptures);
	g_object_set_data(G_OBJECT(winSetupCapture), "fileselCustEnc",     fileselCustEnc);
	g_object_set_data(G_OBJECT(winSetupCapture), "txtCmdOptions",      txtCmdOptions);

	g_signal_connect((gpointer) buttCaptHelp, "clicked",
			 G_CALLBACK(on_buttCaptHelp_clicked),
			 NULL);
	g_signal_connect((gpointer) buttCaptApply, "clicked",
			 G_CALLBACK(on_buttCaptApply_clicked),
			 winSetupCapture);
	g_signal_connect((gpointer) buttCaptOK, "clicked",
			 G_CALLBACK(on_buttCaptOK_clicked),
			 winSetupCapture);
	g_signal_connect((gpointer) buttCaptCancel, "clicked",
			 G_CALLBACK(on_buttCaptCancel_clicked),
			 winSetupCapture);
	g_signal_connect((gpointer) winSetupCapture, "delete_event",
			 G_CALLBACK(on_buttCapt_delete_event),
			 winSetupCapture);
                  
	/* Set capture properties to corresponding widgets */
	gtk_file_chooser_set_current_folder( GTK_FILE_CHOOSER(fileselDirCaptures), frame_dir_path());
	if (custom_encoder && strcmp(custom_encoder, "") != 0) {
		gtk_file_chooser_set_filename( GTK_FILE_CHOOSER(fileselCustEnc), custom_encoder);
	}
  
	if (encoder_options) {
		bufCmd = gtk_text_view_get_buffer( GTK_TEXT_VIEW(txtCmdOptions));
		gtk_text_buffer_set_text(bufCmd, encoder_options, -1);
	}
  
	gtk_widget_show_all( winSetupCapture);
}


void on_capt_enc_menu(GtkMenuItem *menuitem, gpointer  user_data) {
	GtkWidget *parent = user_data;
	GtkWidget *winEncodeCapt;
	GtkWidget *tblWinEncodeCapt;
	GtkWidget *cboxCaptured;
	GtkWidget *lblSession;
	GtkWidget *lblProgress;
	GtkWidget *cbutDelAfterEnc;
	GtkWidget *hbuttboxEnc;
	GtkWidget *buttHelpEnc;
	GtkWidget *buttDelFrames;
	GtkWidget *alignment2;
	GtkWidget *hbox2;
	GtkWidget *image2;
	GtkWidget *label8;
	GtkWidget *buttEncode;
	GtkWidget *alignment3;
	GtkWidget *hbox3;
	GtkWidget *image3;
	GtkWidget *label9;
	GtkWidget *buttCloseEnc;

	winEncodeCapt = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_widget_set_name(winEncodeCapt, "winEncodeCapt");
	gtk_container_set_border_width(GTK_CONTAINER(winEncodeCapt), 5);
	gtk_window_set_title(GTK_WINDOW(winEncodeCapt), _("Webcam - Captured Frames Encoding"));
	gtk_window_set_position(GTK_WINDOW(winEncodeCapt), GTK_WIN_POS_CENTER_ON_PARENT);
	gtk_window_set_transient_for(GTK_WINDOW(winEncodeCapt), GTK_WINDOW(parent));
	gtk_window_set_destroy_with_parent(GTK_WINDOW(winEncodeCapt), TRUE);
	if (parent) gtk_window_present(GTK_WINDOW(parent));

	tblWinEncodeCapt = gtk_table_new(4, 2, FALSE);
	gtk_container_add(GTK_CONTAINER(winEncodeCapt), tblWinEncodeCapt);
	gtk_table_set_row_spacings(GTK_TABLE(tblWinEncodeCapt), 5);
	gtk_table_set_col_spacings(GTK_TABLE(tblWinEncodeCapt), 5);

	cboxCaptured = gtk_combo_box_entry_new_text();
	gtk_table_attach(GTK_TABLE(tblWinEncodeCapt), cboxCaptured, 1, 2, 0, 1,
			 (GtkAttachOptions)(GTK_EXPAND | GTK_FILL),
			 (GtkAttachOptions)(GTK_FILL), 0, 0);

	lblSession = gtk_label_new(_("Session:"));
	gtk_table_attach(GTK_TABLE(tblWinEncodeCapt), lblSession, 0, 1, 0, 1,
			 (GtkAttachOptions)(GTK_FILL),
			 (GtkAttachOptions)(GTK_FILL), 0, 0);
	gtk_misc_set_alignment(GTK_MISC(lblSession), 0, 0.5);

	lblProgress = gtk_label_new("");
	gtk_table_attach(GTK_TABLE(tblWinEncodeCapt), lblProgress, 0, 2, 2, 3,
			 (GtkAttachOptions)(GTK_EXPAND | GTK_FILL),
			 (GtkAttachOptions)(GTK_EXPAND | GTK_FILL), 0, 0);
	gtk_label_set_use_markup(GTK_LABEL(lblProgress), TRUE);
	gtk_label_set_line_wrap(GTK_LABEL(lblProgress), TRUE);
	gtk_misc_set_alignment(GTK_MISC(lblProgress), 0, 0.5);

	cbutDelAfterEnc = gtk_check_button_new_with_mnemonic(_("Delete frame images after encoding"));
	gtk_table_attach(GTK_TABLE(tblWinEncodeCapt), cbutDelAfterEnc, 1, 2, 1, 2,
			 (GtkAttachOptions)(GTK_FILL),
			 (GtkAttachOptions)(0), 0, 0);

	hbuttboxEnc = gtk_hbutton_box_new();
	gtk_table_attach(GTK_TABLE(tblWinEncodeCapt), hbuttboxEnc, 0, 2, 3, 4,
			 (GtkAttachOptions)(GTK_EXPAND | GTK_FILL),
			 (GtkAttachOptions)(0), 0, 0);
	gtk_box_set_spacing(GTK_BOX(hbuttboxEnc), 5);

	buttHelpEnc = gtk_button_new_from_stock("gtk-help");
	gtk_container_add(GTK_CONTAINER(hbuttboxEnc), buttHelpEnc);
	GTK_WIDGET_SET_FLAGS(buttHelpEnc, GTK_CAN_DEFAULT);

	buttDelFrames = gtk_button_new();
	gtk_container_add(GTK_CONTAINER(hbuttboxEnc), buttDelFrames);
	GTK_WIDGET_SET_FLAGS(buttDelFrames, GTK_CAN_DEFAULT);

	alignment2 = gtk_alignment_new(0.5, 0.5, 0, 0);
	gtk_container_add(GTK_CONTAINER(buttDelFrames), alignment2);

	hbox2 = gtk_hbox_new(FALSE, 2);
	gtk_container_add(GTK_CONTAINER(alignment2), hbox2);

	image2 = gtk_image_new_from_stock("gtk-delete", GTK_ICON_SIZE_BUTTON);
	gtk_box_pack_start(GTK_BOX(hbox2), image2, FALSE, FALSE, 0);

	label8 = gtk_label_new_with_mnemonic(_("Delete Frames"));
	gtk_box_pack_start(GTK_BOX(hbox2), label8, FALSE, FALSE, 0);

	buttEncode = gtk_button_new();
	gtk_container_add(GTK_CONTAINER(hbuttboxEnc), buttEncode);
	GTK_WIDGET_SET_FLAGS(buttEncode, GTK_CAN_DEFAULT);

	alignment3 = gtk_alignment_new(0.5, 0.5, 0, 0);
	gtk_container_add(GTK_CONTAINER(buttEncode), alignment3);

	hbox3 = gtk_hbox_new(FALSE, 2);
	gtk_container_add(GTK_CONTAINER(alignment3), hbox3);

	image3 = gtk_image_new_from_stock("gtk-goto-last", GTK_ICON_SIZE_BUTTON);
	gtk_box_pack_start(GTK_BOX(hbox3), image3, FALSE, FALSE, 0);

	label9 = gtk_label_new_with_mnemonic(_("Encode"));
	gtk_box_pack_start(GTK_BOX(hbox3), label9, FALSE, FALSE, 0);

	buttCloseEnc = gtk_button_new_from_stock("gtk-close");
	gtk_container_add(GTK_CONTAINER(hbuttboxEnc), buttCloseEnc);
	GTK_WIDGET_SET_FLAGS(buttCloseEnc, GTK_CAN_DEFAULT);

	g_signal_connect((gpointer) cboxCaptured, "changed",
			  G_CALLBACK(on_cboxCaptured_changed),
			  NULL);
	g_signal_connect((gpointer) cbutDelAfterEnc, "toggled",
			  G_CALLBACK(on_cbutDelAfterEnc_toggled),
			  NULL);
	g_signal_connect((gpointer) buttHelpEnc, "clicked",
			  G_CALLBACK(on_buttHelpEnc_clicked),
			  NULL);
	g_signal_connect((gpointer) buttDelFrames, "clicked",
			  G_CALLBACK(on_buttDelFrames_clicked),
			  NULL);
	g_signal_connect((gpointer) buttEncode, "clicked",
			  G_CALLBACK(on_buttEncode_clicked),
			  NULL);
	g_signal_connect((gpointer) buttCloseEnc, "clicked",
			  G_CALLBACK(on_buttCloseEnc_clicked),
			  winEncodeCapt);
	g_signal_connect((gpointer) winEncodeCapt, "delete_event",
			  G_CALLBACK(on_buttCloseEnc_delete_event),
			  winEncodeCapt);

	gtk_widget_show_all(winEncodeCapt);
}

void show_ok_dialog(char *mymsg, GtkWidget *parent, char *ui_name, int error, void *callback)  {
	show_ok_message(mymsg, parent, ui_name, error, callback);
}

/* GUI stuff */

/******* Menu GUI (called from gyache_image_window_new) ******/

GtkWidget *capt_menu_item(GtkWidget *main_window)
{
	GtkWidget *capt_main;
	GtkWidget *capt_menu;
	GtkWidget *capt_setup_menu;
	GtkWidget *capt_enc_menu;

	capt_main = gtk_menu_item_new_with_label (_("Capture"));
	gtk_widget_show (capt_main);
  
	capt_menu = gtk_menu_new();
	gtk_menu_item_set_submenu (GTK_MENU_ITEM (capt_main), capt_menu);

	capt_setup_menu = gtk_image_menu_item_new_with_label (_("Setup ..."));
	gtk_container_add (GTK_CONTAINER (capt_menu), capt_setup_menu);
	gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(capt_setup_menu), 
				      GTK_WIDGET(gtk_image_new_from_stock(GTK_STOCK_PREFERENCES, GTK_ICON_SIZE_MENU) ));

	capt_enc_menu = gtk_image_menu_item_new_with_label (_("Encode ..."));
	gtk_container_add (GTK_CONTAINER (capt_menu), capt_enc_menu);
	gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(capt_enc_menu), 
				      GTK_WIDGET(gtk_image_new_from_stock(GTK_STOCK_GOTO_LAST, GTK_ICON_SIZE_MENU) ));

	g_signal_connect(capt_setup_menu, "activate", (GCallback)on_capt_setup_menu, main_window);
	g_signal_connect(capt_enc_menu,   "activate", (GCallback)on_capt_enc_menu,   main_window);

	return(capt_main);
}

GtkWidget *record_quit_bar(GtkWidget *main_window, void *on_shutdown_app) {
	GtkWidget *hbox1;
	GtkWidget *butRecord;
	GtkWidget *butPause;
	GtkWidget *butSeparator;
	GtkWidget *btnClose;

	hbox1 = gtk_hbox_new(FALSE, 1);
	IsRecording = 0; /* Initially, by default, when we startup, we are NOT recording */
	butRecord = (GtkWidget*) gtk_toggle_button_new();
	gtk_button_set_label(GTK_BUTTON(butRecord), "gtk-media-record");
	gtk_button_set_use_stock(GTK_BUTTON(butRecord), TRUE);
	saved_butRecord = butRecord;
	gtk_box_pack_start(GTK_BOX(hbox1), butRecord, FALSE, FALSE, 1 );	
	gyachi_set_tooltip(butRecord, _("Start Recording"));

	butPause = (GtkWidget*) gtk_toggle_button_new();
	gtk_button_set_label(GTK_BUTTON(butPause), "gtk-media-pause");
	gtk_button_set_use_stock(GTK_BUTTON(butPause), TRUE);
	gtk_box_pack_start(GTK_BOX(hbox1), butPause, FALSE, FALSE, 1 );	
	gyachi_set_tooltip(butPause, _("Pause Recording"));
	gtk_widget_set_sensitive( butPause, FALSE );
    
	butSeparator = gtk_label_new("");
	gtk_box_pack_start(GTK_BOX(hbox1), butSeparator, TRUE, TRUE, 1 );	

	btnClose = (GtkWidget*) gtk_button_new_from_stock ("gtk-quit");
	gtk_box_pack_start(GTK_BOX(hbox1), btnClose, FALSE, FALSE, 1 );	
	gyachi_set_tooltip(btnClose, _("Close Window"));

	g_signal_connect ( (gpointer)btnClose, "clicked",
			G_CALLBACK(on_shutdown_app), GTK_OBJECT(main_window));

	g_signal_connect ((gpointer) butRecord, "toggled",
                    G_CALLBACK (on_butRecord_toggled),
                    butPause);

	g_signal_connect ((gpointer) butPause, "toggled",
                    G_CALLBACK (on_butPause_toggled),
                    NULL);

	return(hbox1);
}

/******* library support  *******/

int is_recording()
{
	return(IsRecording);
}

GtkWidget *but_record()
{
	return(saved_butRecord);
}

char *frame_dir_path()
{
	if ((frame_dir == 0) || (*frame_dir == 0)) {
		char *(pieces[4]);
		char *t_filename;

		pieces[0]=GYACH_CFG_DIR;
		pieces[1]="/capture_dir";
		pieces[2]=NULL;
		t_filename=gyachi_filename(pieces);

		if (frame_dir) free(frame_dir);
		frame_dir = t_filename;

		if (!dir_exists(frame_dir)) {
			mkdir(frame_dir, 0700);
		}

		write_capturc();
	}

	return(frame_dir);
}

void set_debug_state(int debug_state)
{
	app_debug = debug_state;
}

/* Sets label containing captured filename. Returns it for making actual file */
char* setCapturedFilename(char *name, GtkWidget *FrameLabel)
{
	time_t time_now = time( NULL );
	char *(pieces[4]);
	struct tm   *tm_now = localtime( &time_now );
	static int   movie_exists = -1;
	struct stat  stat_buf;
	static char *JpgNameBuf= NULL;
	static int   JpgNameBuf_size;
	static char *stampbuf = NULL;
	static char *timebuf = NULL;
	char *timebuf_ptr;
	int   rv;

	if (stampbuf == 0) {
		stampbuf = malloc(11); /* YYYY-MM-DD */
		strftime( stampbuf, 11, "%Y-%m-%d", tm_now );
	}

	/* create JpgNameBuf upon initial call */
	if (!JpgNameBuf) {
		/* name '-' date '-' cap##-#####.jpg (but allow for ######) */
		JpgNameBuf_size = strlen(name) + 1 + 10 + 1 + 18;
		JpgNameBuf = malloc(JpgNameBuf_size);
	}
	/* Find 1st number that is  not in use :) */
	if (FrameNumber == 1) {
		while (1) {
			sprintf(JpgNameBuf, "%s-%s-cap%02d-%05d.jpg",
				name, stampbuf, MovieNumber, 1);

			pieces[0]=frame_dir_path();
			pieces[1]="/";
			pieces[2]=JpgNameBuf;
			pieces[3]=NULL;
			if (JpgFilePath) free(JpgFilePath);
			JpgFilePath=gyachi_filename(pieces);
			rv=stat(JpgFilePath, &stat_buf);
			if (rv == -1) {
				if (errno == ENOENT) break;
				if (errno == EACCES) break;
			}
			MovieNumber++;
		}
	}
	else {
		sprintf(JpgNameBuf, "%s-%s-cap%02d-%05d.jpg",
			name, stampbuf, MovieNumber, FrameNumber);
	}

	if (FrameNumber == 1) {
		pieces[0]=frame_dir_path();
		pieces[1]="/";
		pieces[2]=JpgNameBuf;
		pieces[3]=NULL;
		if (JpgFilePath) free(JpgFilePath);
		JpgFilePath=gyachi_filename(pieces);
		rv=stat(JpgFilePath, &stat_buf);
		if (rv != -1) movie_exists = 1;
		else movie_exists = 0;
	}

	/* Now enrich the filename with markup and set to label */
	if (!timebuf) {
		timebuf = malloc(JpgNameBuf_size + 50); /* allow for markup */
	}
	strcpy(timebuf, "<small>");
	timebuf_ptr = timebuf+7;
	if (movie_exists) {
		strcpy(timebuf_ptr, "<span color=\"red\">");
		timebuf_ptr += 18;
	}
	strcpy(timebuf_ptr, JpgNameBuf);
	timebuf_ptr += strlen(JpgNameBuf);
	if (movie_exists) {
		strcpy(timebuf_ptr, "</span>");
		timebuf_ptr += 7;
	}
	strcpy(timebuf_ptr, "</small>");

	gtk_label_set_markup(GTK_LABEL(FrameLabel), timebuf);
	FrameNumber++;
	return JpgNameBuf;
}

void write_image_to_file(char *jpg_buf, int packet_size, char *name, GtkWidget *FrameLabel, char *ui_name, GtkWidget *main_window)
{
	char *(pieces[8]);
	char *t_filename;
	int  ImgFile;

	/* Construct file name */
	if (is_recording() == 1) {
		pieces[0]=frame_dir_path();
		pieces[1]="/";
		pieces[2]=setCapturedFilename(name, FrameLabel);
		pieces[3]=NULL;
		t_filename=gyachi_filename(pieces);
		if (app_debug) { printf("Path: %s\n", t_filename); fflush(stdout); }
		ImgFile = creat(t_filename, 0664);
		if (ImgFile >= 0) {
			write(ImgFile, jpg_buf, packet_size);
			close(ImgFile);
		}
		else {
			char *t_errno_message;
			char *t_message_buff;

			t_errno_message=strdup(strerror(errno));
			pieces[0]=_("Unable to open output file: ");
			pieces[1]=t_filename;
			pieces[2]="\n";
			pieces[3]=_("Error was: ");
			pieces[4]=t_errno_message;
			pieces[5]="\n";
			pieces[6]=_("Perhaps you need to Configure the image path in Webcam -> Capture -> Setup");
			pieces[7]=NULL;
			t_message_buff=build_string(pieces);

			/* Disable recording, since we can't write the images... */
			if (but_record() != NULL) {
				gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(but_record()), FALSE);
			}

			show_ok_dialog(t_message_buff, main_window, ui_name, 0, NULL);
			free(t_errno_message);
			free(t_message_buff);
		}
		free(t_filename);
	}
}

void set_dummy_image(GtkWidget *current_image) {
	gtk_image_set_from_stock(GTK_IMAGE(current_image), GTK_STOCK_STOP, GTK_ICON_SIZE_DIALOG);
}
void set_dummy_image_start(GtkWidget *current_image) {
	gtk_image_set_from_stock(GTK_IMAGE(current_image), GTK_STOCK_CONVERT, GTK_ICON_SIZE_DIALOG);
}
void set_dummy_image_start2(GtkWidget *current_image) {
	gtk_image_set_from_stock(GTK_IMAGE(current_image), GTK_STOCK_REFRESH, GTK_ICON_SIZE_DIALOG);
}


unsigned char * image_2_jpg(char *in_img, int size, char *format)
{
	char *out_img = NULL;
	jas_stream_t *in, *out;
	jas_image_t *image;
	int infmt;
	int outfmt;
	char *outopts;

	if (app_debug) { printf("Jasper-1\n");  fflush(stdout); }

	if (! jasper_started) {
		if( jas_init() ) {
			if (app_debug) {printf("Could not init jasper\n");}
			return NULL;
		}
		jasper_started=1;
		setlocale( LC_NUMERIC, "C");
	}

	if (app_debug) {printf("Jasper-2\n");  fflush(stdout);}
	outfmt = jas_image_strtofmt(format);
	if (app_debug) {printf("Jasper-3\n");  fflush(stdout);}

	if(!(in = jas_stream_memopen((unsigned char *)in_img, size))) {
		if (app_debug) {printf("Could not open jasper input stream\n");}
		return NULL;
	}

	if (app_debug) {printf("Jasper-4\n");  fflush(stdout);}

	infmt = jas_image_getfmt(in);

	if (app_debug) {
		printf( "Got input image format: %d %s\n", infmt, jas_image_fmttostr(infmt));
		fflush(stdout);
	}

	if(infmt <= 0) {return NULL; }

	if (app_debug) {printf("Jasper-5\n");  fflush(stdout);}

	if(!strcmp(jas_image_fmttostr(infmt), format)) {
		/* image is already jpeg */
		jas_stream_close(in);
		return NULL;
	}

	if (app_debug) {printf("Jasper-6\n");  fflush(stdout);}

	if(!(image = jas_image_decode(in, infmt, NULL))) {
		if (app_debug) {printf( "Could not decode image format\n");}
		return NULL;
	}

	if (app_debug) {printf("Jasper-7\n");  fflush(stdout);}

	if(!(out = jas_stream_memopen(out_img, 0))) {
		if (app_debug) {printf( "Could not open output stream\n");}
		return NULL;
	}

	if (app_debug) {printf( "Encoding to format: %d %s\n", outfmt, format);}

	/* We need compression options, otherwise Jasper creates HUGE 
	   Jpeg-2000 images from PNM (over 130kb!), these options get 
	   us to 3.5-5kb with reasonable quality - most of the images created
	   seem to be exactly 4096 bytes, and normal Yahoo webcam 
	   images are usually between 2.5kb-5.0kb */

	/* This is what I was using before */
	/* snprintf(outopts, 93, "%s", "cblkwidth=64\ncblkheight=64\nnumrlvls=4\nrate=0.0165\nprcheight=128\nprcwidth=2048");  */
	
	/* Patch suggested by one random user who was using 
	   Jasper 1.701 (an unsupported version), send in JPC 
	   format, instead of JP2 using the following options... - NOTE: The JPC 
	   implementation in Jasper is incomplete...after more testing
	   this patch DID work it appears
	*/

	if (!strcmp(format, "jpc")) {
		outopts ="cblkwidth=64\ncblkheight=64\nnumrlvls=4\nrate=0.0165\nprcheight=128\nprcwidth=2048\nmode=real";
	}
	else {
		outopts = NULL;
	}
    
	if((jas_image_encode(image, out, outfmt, outopts))) {
		if (app_debug) {printf("Could not encode image format\n");}
		return NULL;
	}

	if (app_debug) {printf("Jasper-8\n");  fflush(stdout);}

	jas_stream_flush(out);

	size = ((jas_stream_memobj_t *)out->obj_)->bufsize_;
	if (app_debug) {printf( "Encoded size is: %d\n", size);}
	jas_stream_close(in);
	out_img=malloc(size+1);
	memcpy(out_img, ((jas_stream_memobj_t *)out->obj_)->buf_, size);
	packet_size=size;

	if (app_debug) {printf("Jasper-9\n");  fflush(stdout);}

	jas_stream_close(out);
	jas_image_destroy(image);

	/* jas_image_clearfmts(); */ /* this is bad */

	if (app_debug) {printf("Jasper-10\n");  fflush(stdout);}

	return out_img;	
}

/******* cam socket support  *******/

void set_local_addresses()
{
	char command[] = "/sbin/ifconfig `netstat -nr | grep '^0\\.0' | tr -s ' ' ' ' | cut -f 8 -d' '` | grep inet | tr -s ' ' ':' | cut -f4 -d:";
	char buff[1024];
	char addresses[1024];
	struct hostent * hn;
	FILE * f = NULL;

	
	/* We could do something like this: Make up an IP address as the 
	   'external' IP address that Yahoo will show to the people viewing our
	   cam, but this is pretty useless because Yahoo will send the 
	   viewer BOTH the 'external' (fake) IP and your real IP, and it is
	   completely up to the viewer's program to show either one or 
	   both.  Basically, faking the IP does not provide any level of 
	   privacy at all, as most third-party apps probably will show 
	   either the real IP or both.  The point: If you don't want your IP
	  exposed, just stay off the webcam system completely.   */

	/* 
	   if (local_address) {free(local_address);}
	   local_address= strdup("172.248.22.61");
	   return;
	*/

	gethostname(buff,sizeof(buff));

	hn = gethostbyname(buff);
	if(hn) {
		char *quad = hn->h_addr_list[0];
		snprintf(addresses, sizeof(addresses), "%d.%d.%d.%d",
			 quad[0], quad[1], quad[2], quad[3] );
	} 
	if((!hn || (!strcmp(addresses,"127.0.0.1"))) 
	   && (f = popen(command, "r")) != NULL ) {
		int i=0;

		do {
			int r = fgetc(f);
			buff[i] = (char)r;
			if(buff[i]=='\r' || buff[i]=='\n' || r==EOF)
				buff[i]='\0';
			else if(i >= sizeof(buff)-1) {
				buff[i]='\0';
				/*return error?*/
			}
		} while(buff[i++]);

		pclose(f);
		strncpy(addresses, buff, sizeof(addresses));
	}
	else {
		addresses[0]=0;
	}
	if (local_address) {free(local_address);}
	local_address= strdup(addresses);
}


